package org.maachang.engine.servlet;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Set;

import org.maachang.engine.util.RandomUtil;

/**
 * ページ内セッションを管理.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class PageSessionManager {
    
    /**
     * ランダムキー名長.
     */
    public static final int RANDOM_SIZE = 48 ;
    
    /**
     * ページセッション監視スレッド.
     */
    private PageSessionMonitor monitor = null ;
    
    /**
     * コンストラクタ.
     */
    private PageSessionManager() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * タイムアウト値を設定してオブジェクトを生成します.
     * <BR>
     * @param timeout 対象のタイムアウト値を設定します.
     */
    public PageSessionManager( long time ) {
        this.monitor = new PageSessionMonitor( time ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    protected void destroy() {
        if( monitor != null ) {
            monitor.clear() ;
        }
        monitor = null ;
    }
    
    /**
     * 新しいページ内セッションを発行.
     * <BR><BR>
     * 新しいページ内セッションを発行します.
     * <BR>
     * @return String 新しいページ内セッションIDが返されます.
     */
    public synchronized String createSession() {
        String ret = null ;
        Hashtable<String,PageSessionChild> man = monitor.getManager() ;
        if( man == null ) {
            return null ;
        }
        for( ;; ) {
            ret = RandomUtil.randomString( RANDOM_SIZE,true,true,true ) ;
            synchronized( man ) {
                if( man.get( ret ) == null ) {
                    man.put( ret,new PageSessionChild( ret ) ) ;
                    break ;
                }
            }
        }
        return ret ;
    }
    
    /**
     * 指定セッションIDが存在しない場合は、生成.
     * <BR><BR>
     * 指定セッションIDが存在しない場合は、生成する.
     * <BR>
     * @param sessionId 対象のセッションIDを設定します.
     * @return String 新しいページ内セッションIDが返されます.
     */
    public synchronized String createSession( String sessionId ) {
        if( sessionId == null || ( sessionId = sessionId.trim() ).length() != RANDOM_SIZE ) {
            return createSession() ;
        }
        Hashtable<String,PageSessionChild> man = monitor.getManager() ;
        if( man == null ) {
            return null ;
        }
        String ret = null ;
        synchronized( man ) {
            if( man.get( sessionId ) == null ) {
                man.put( sessionId,new PageSessionChild( sessionId ) ) ;
                ret = sessionId ;
            }
            else {
                ret = createSession() ;
            }
        }
        return ret ;
    }
    
    /**
     * ページ内セッションを取得.
     * <BR><BR>
     * ページ内セッション情報を取得します.
     * <BR>
     * @param sessionId 対象のページ内セッションIDを設定します.
     * @return PageSession ページ内セッションが返されます.
     */
    public synchronized PageSession getSession( String sessionId ) {
        if( sessionId == null || ( sessionId = sessionId.trim() ).length() != RANDOM_SIZE ) {
            return null ;
        }
        Hashtable<String,PageSessionChild> man = monitor.getManager() ;
        if( man == null ) {
            return null ;
        }
        PageSessionChild ret = null ;
        synchronized( man ) {
            ret = man.get( sessionId ) ;
            if( ret != null ) {
                ret.update() ;
            }
        }
        return ret ;
    }
    
    /**
     * 設定タイムアウト値を取得.
     * <BR><BR>
     * 設定されているタイムアウト値を取得します.
     * <BR>
     * @return long timeout 設定されているタイムアウト値が返されます.
     */
    public synchronized long getTimeout() {
        return monitor.getTimeout() ;
    }
}

/**
 * ページ内セッション１要素.
 */
class PageSessionChild implements PageSession {
    
    /**
     * セッションID.
     */
    private String sessionId = null ;
    
    /**
     * セッション情報管理.
     */
    private HashMap<String,Object> values = null ;
    
    /**
     * 生成時間.
     */
    private long createTime = -1L ;
    
    /**
     * 更新時間.
     */
    private long updateTime = -1L ;
    
    /**
     * コンストラクタ.
     */
    private PageSessionChild() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ページ内セッション情報を生成します.
     * <BR>
     * @param sessionId 対象のセッションIDを設定します.
     */
    protected PageSessionChild( String sessionId ) {
        this.sessionId = sessionId ;
        this.values = new HashMap<String,Object>() ;
        long time = System.currentTimeMillis() ;
        this.createTime = time ;
        this.updateTime = time ;
    }
    
    protected void finalize() throws Exception {
        this.sessionId = null ;
        this.values = null ;
        this.createTime = -1L ;
        this.updateTime = -1L ;
    }
    
    /**
     * セッション内容をクリア.
     * <BR><BR>
     * セッション内容をクリアします.
     */
    public void clear() {
        if( this.values != null ) {
            this.values.clear() ;
        }
    }
    
    /**
     * セッションIDを取得.
     * <BR><BR>
     * セッションIDを取得します.
     * <BR>
     * @return String セッションIDが返されます.
     */
    public String getSessionId() {
        return this.sessionId ;
    }
    
    /**
     * セッション生成時間を取得.
     * <BR><BR>
     * セッション生成時間を取得します.
     * <BR>
     * @return long セッション生成IDが返されます.
     */
    public synchronized long getCreateTime() {
        return this.createTime ;
    }
    
    /**
     * セッション更新時間を取得.
     * <BR><BR>
     * セッション更新時間が返されます.
     * <BR>
     * @return long セッション更新時間が返されます.
     */
    public synchronized long getUpdateTime() {
        return this.updateTime ;
    }
    
    /**
     * 更新.
     */
    protected synchronized void update() {
        this.updateTime = System.currentTimeMillis() ;
    }
    
    /**
     * セッション情報を登録.
     * <BR><BR>
     * セッション情報を登録します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @param value 対象の要素を設定します.
     */
    public void put( String key,Object value ) {
        if( key == null || value == null ||
            ( key = key.trim() ).length() <= 0 ) {
            return ;
        }
        this.values.put( key,value ) ;
    }
    
    /**
     * セッション情報を取得.
     * <BR><BR>
     * セッション情報を取得します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @return Object セッション情報が返されます.
     */
    public Object get( String key ) {
        if( key == null || ( key = key.trim() ).length() <= 0 ) {
            return null ;
        }
        return this.values.get( key ) ;
    }
    
    /**
     * セッション情報を削除.
     * <BR><BR>
     * セッション情報を削除します.
     * <BR>
     * @param key 対象のキー名を設定します.
     */
    public void remove( String key ) {
        if( key == null || ( key = key.trim() ).length() <= 0 ) {
            return ;
        }
        this.values.remove( key ) ;
    }
    
}

/**
 * ページセッションタイムアウト監視.
 */
class PageSessionMonitor extends Thread {
    
    /**
     * タイムアウト最小値.
     */
    private static final long MIN_TIMEOUT = 60L * 1000L ;
    
    /**
     * タイムアウト最大値.
     */
    private static final long MAX_TIMEOUT = 12L * 60L * 60L * 1000L ;
    
    /**
     * ページ内セッション管理.
     */
    private Hashtable<String,PageSessionChild> manager = null ;
    
    /**
     * セッション破棄時間.
     */
    private long time = -1L ;
    
    /**
     * 停止フラグ.
     */
    private volatile boolean stopFlag = true ;
    
    /**
     * コンストラクタ.
     */
    private PageSessionMonitor() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * タイムアウト値を設定してオブジェクトを生成します.
     * <BR>
     * @param timeout 対象のタイムアウト値を設定します.
     */
    public PageSessionMonitor( long time ) {
        if( time <= MIN_TIMEOUT ) {
            time = MIN_TIMEOUT ;
        }
        if( time >= MAX_TIMEOUT ) {
            time = MAX_TIMEOUT ;
        }
        this.manager = new Hashtable<String,PageSessionChild>() ;
        this.time = time ;
        this.stopFlag = false ;
        this.setDaemon( true ) ;
        this.start() ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    protected void clear() {
        this.threadStop() ;
        this.manager = null ;
    }
    
    /**
     * マネージャオブジェクトを取得.
     */
    protected Hashtable<String,PageSessionChild> getManager() {
        return manager ;
    }
    
    /**
     * 監視時間を取得.
     */
    protected long getTimeout() {
        return this.time ;
    }
    
    /**
     * スレッド停止.
     */
    protected synchronized void threadStop() {
        this.stopFlag = true ;
    }
    
    /**
     * スレッド停止状態を取得.
     */
    protected synchronized boolean isThreadStop() {
        return this.stopFlag ;
    }
    
    /**
     * 停止時間.
     */
    private static final long WAIT = 5000L ;
    
    /**
     * スレッド実行.
     */
    public void run() {
        
        try {
            for( ;; ) {
                try {
                    Thread.sleep( WAIT ) ;
                } catch( Exception ee ) {
                    threadStop() ;
                }
                if( isThreadStop() == true ) {
                    break ;
                }
                Set keySet = null ;
                if( ( keySet = manager.keySet() ) != null ) {
                    Object[] objs = keySet.toArray() ;
                    keySet = null ;
                    if( objs != null ) {
                        int len = objs.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            if( isThreadStop() == true ) {
                                break ;
                            }
                            try {
                                Thread.sleep( 30L ) ;
                            } catch( Exception ee ) {
                                threadStop() ;
                                break ;
                            }
                            try {
                                if( manager == null ) {
                                    threadStop() ;
                                    break ;
                                }
                                synchronized( manager ) {
                                    PageSessionChild ch = manager.get( ( String )objs[ i ] ) ;
                                    if( ch != null && ch.getUpdateTime() + time <= System.currentTimeMillis() ) {
                                        manager.remove( ( String )objs[ i ] ) ;
                                    }
                                }
                            } catch( Exception ee ) {
                            }
                        }
                    }
                }
            }
        } catch( Exception e ) {
        } catch( OutOfMemoryError er ) {
        } catch( ThreadDeath dt ) {
            threadStop() ;
            throw dt ;
        }
        
    }
    
    
}

