package org.maachang.engine.servlet;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.engine.util.FileUtil;

/**
 * HTTP変換.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
class ConvertHttpParams {

    /**
     * LOG.
     */
    private static final Log LOG = LogFactory.getLog(ConvertHttpParams.class);

    /**
     * キャラクタセット.
     */
    private static final String CHARSET = "UTF8";

    /**
     * MultiPartForm情報.
     */
    private static final String MULTIPART_FORM = "multipart/form-data";

    /**
     * 区切り情報開始値.
     */
    private static final String MULTIPART_START = "boundary=";

    /**
     * パラメータを変換. <BR>
     * <BR>
     * 
     * @param request
     *            対象のリクエストを設定します.
     * @return HashMap<String,Object> 変換されたパラメータが返されます.
     * @exception Exception
     *                例外.
     */
    public static final HashMap<String, Object> getParams(
            HttpServletRequest request) throws Exception {
        HashMap<String, Object> ret = new HashMap<String, Object>();
        String multiPartForm = null;
        String ctxType = request.getContentType();
        if (ctxType != null && ctxType.startsWith(MULTIPART_FORM)) {
            int p = ctxType.indexOf(MULTIPART_START);
            if (p != -1) {
                multiPartForm = ctxType.substring(p + MULTIPART_START.length(),
                        ctxType.length()).trim();
            }
        }
        // HTTPがMultipartFormの場合.
        if (multiPartForm != null) {
            // multiPartFormを解析.
            analysisMultiPartForm(ret, multiPartForm,
                    getInputStreamByBytes(request.getInputStream()));
        }
        // 通常のHTTP.
        else {
            request.setCharacterEncoding(CHARSET);
            Map map = request.getParameterMap();
            if (map != null) {
                Object[] objs = map.keySet().toArray();
                if (objs != null && objs.length > 0) {
                    int len = objs.length;
                    for (int i = 0; i < len; i++) {
                        String key = (String) objs[i];
                        Object params = map.get(key);
                        putRequestParams(ret, key, params);
                    }
                }
            }
        }
        LOG.info( new StringBuilder().
            append( "## (url):" ).append( request.getRequestURL() ).
            append( " [" ).append( request.getRemoteAddr() ).
            append( "] - $params:" ).append( ret ).toString() );
        return ret;
    }

    private static final void analysisMultiPartForm(
            HashMap<String, Object> out, String multiPartForm, byte[] binary)
            throws Exception {
        int p = 0;
        int b = 0;
        String enter = "\r\n";
        int enterLen = enter.length();
        String target = "--" + multiPartForm;
        String end = target + "--";
        String keyName = null;
        String fileName = null;
        String mimeType = null;
        boolean dataFlag = false;
        boolean keyFlag = false;
        for (;;) {
            // データ条件で無い場合.
            if (dataFlag == false) {
                p = getIndexOf(binary, b, enter);
                if (p == -1) {
                    break;
                }
                String one = new String(binary, b, p - b, CHARSET);
                b = p + enterLen;
                // データ開始値.
                if (keyFlag == false) {
                    // データの開始値の場合.
                    if (one.equals(target) == true) {
                        keyFlag = true;
                    }
                    // 全てのデータ終了値の場合.
                    else if (one.equals(end) == true) {
                        break;
                    }
                    // その他.
                    else {
                        throw new IllegalArgumentException("不正な条件を検出しました");
                    }
                }
                // データキー名など.
                else {
                    // データ情報の場合.
                    if (one.length() <= 0) {
                        dataFlag = true;
                    }
                    // データキー名などの情報.
                    else {
                        // アップロードファイル名.
                        if (one.startsWith("Content-Disposition:")) {
                            int filePnt = one.indexOf("; filename=\"");
                            if (filePnt != -1) {
                                filePnt += "; filename=\"".length();
                                fileName = FileUtil.getFileName(
                                        one.substring(filePnt, one.indexOf(
                                                "\"", filePnt))).trim();
                            }
                        }
                        // データキー名.
                        int namePnt = one.indexOf("; name=\"");
                        if (namePnt != -1) {
                            namePnt += "; name=\"".length();
                            keyName = one.substring(namePnt,
                                    one.indexOf("\"", namePnt)).trim();
                        }
                        // MimeType.
                        else if (one.startsWith("Content-Type:")) {
                            mimeType = one.substring("Content-Type:".length(),
                                    one.length()).trim();
                        }
                        // その他.
                        else {
                            throw new IllegalArgumentException("不正な条件を検出しました");
                        }
                    }
                }
            }
            // データ情報.
            else {
                // update情報.
                if (fileName != null) {
                    p = getIndexOf(binary, b, target);
                    if (p == -1) {
                        throw new IllegalArgumentException("不正な条件を検出しました");
                    }
                    int len = p - (b + enterLen);
                    byte[] update = new byte[len];
                    System.arraycopy(binary, b, update, 0, len);
                    b = p;
                    HttpBinary bin = new HttpBinaryImpl(fileName, mimeType,
                            update);
                    putRequestParams(out, keyName, bin);
                } else {
                    p = getIndexOf(binary, b, target);
                    if (p == -1) {
                        throw new IllegalArgumentException("不正な条件を検出しました");
                    }
                    String one = new String(binary, b, p - (b + enterLen),
                            CHARSET);
                    b = p;
                    putRequestParams(out, keyName, one);
                }
                keyName = null;
                fileName = null;
                mimeType = null;
                dataFlag = false;
                keyFlag = false;
            }
        }
    }

    private static final byte[] getInputStreamByBytes(InputStream stream)
            throws Exception {
        BufferedInputStream buf = null;
        ByteArrayOutputStream bo = null;
        byte[] ret = null;
        try {
            buf = new BufferedInputStream(stream);
            bo = new ByteArrayOutputStream();
            byte[] bin = new byte[8192];
            for (;;) {
                int len = stream.read(bin);
                if (len <= 0) {
                    if (len <= -1) {
                        break;
                    }
                    continue;
                }
                bo.write(bin, 0, len);
            }
            buf.close();
            buf = null;
            ret = bo.toByteArray();
            bo.close();
            bo = null;
        } finally {
            if (buf != null) {
                try {
                    buf.close();
                } catch (Exception t) {
                }
                buf = null;
            }
            if (bo != null) {
                try {
                    bo.close();
                } catch (Exception e) {
                }
                bo = null;
            }
        }
        return ret;
    }

    private static final void putRequestParams(HashMap<String, Object> map,
            String key, Object value) {
        if (value == null) {
            return;
        }
        if (value instanceof HttpBinary) {
            Object o = map.get(key);
            if (o == null) {
                map.put(key, value);
            } else if (o.getClass().getName().equals(ArrayList.class.getName())) {
                ArrayList<Object> lst = (ArrayList<Object>) o;
                lst.add(value);
            } else {
                ArrayList<Object> lst = new ArrayList<Object>();
                lst.add(o);
                lst.add(value);
                map.put(key, lst);
            }
        } else if (value instanceof String
                || (value instanceof String[] && ((String[]) value).length == 1)) {
            if (value instanceof String[]) {
                value = ((String[]) value)[0];
            }
            Object o = map.get(key);
            if (o != null) {
                ArrayList<Object> lst = null;
                if (o.getClass().getName().equals(ArrayList.class.getName())) {
                    lst = (ArrayList<Object>) o;
                } else {
                    lst = new ArrayList<Object>();
                    lst.add(o);
                    map.put(key, lst);
                }
                lst.add(value);
            } else {
                map.put(key, value);
            }
        } else if (value instanceof String[]) {
            Object o = map.get(key);
            ArrayList<Object> lst = null;
            if (o != null) {
                if (o.getClass().getName().equals(ArrayList.class.getName())) {
                    lst = (ArrayList<Object>) o;
                } else {
                    lst = new ArrayList<Object>();
                    lst.add(o);
                    map.put(key, lst);
                }
            } else {
                lst = new ArrayList<Object>();
                map.put(key, lst);
            }
            String[] values = (String[]) value;
            int len = values.length;
            for (int i = 0; i < len; i++) {
                lst.add(values[i]);
            }
        }
    }

    private static int getIndexOf(byte[] binary, int pos, String data)
            throws Exception {
        if (data == null || data.length() <= 0) {
            return -1;
        }
        byte[] chk = data.getBytes(CHARSET);
        int chkLen = chk.length;
        int len = binary.length;
        int ret = -1;
        for (int i = pos; i < len; i++) {
            if (binary[i] == chk[0] && i + chkLen < len) {
                ret = i;
                for (int j = 1; j < chkLen; j++) {
                    if (binary[i + j] != chk[j]) {
                        ret = -1;
                        break;
                    }
                }
                if (ret != -1) {
                    return ret;
                }
            }
        }
        return -1;
    }

}
