package org.maachang.engine.servlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.dao.dbms.Record;
import org.maachang.engine.Action;
import org.maachang.engine.util.FileUtil;
import org.maachang.engine.util.Reflect;
import org.maachang.engine.util.SequenceSync;
import org.maachang.engine.util.StringUtil;

/**
 * Action用サーブレット.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class ActionServlet extends HttpServlet {
    private static final long serialVersionUID = 3853007460537014586L;

    /**
     * LOG.
     */
    protected static final Log LOG = LogFactory.getLog(ActionServlet.class);

    /**
     * ページ内セッション名.
     */
    public static final String INNER_PAGE_SESSION = "_INNER_SESSION" ;

    /**
     * リファラー.
     */
    public static final String REFERER = "Referer" ;

    /**
     * シーケンスID格納Attribute名.
     */
    public static final String SEQUENCE_ID_ATTRIBUTE_NAME = "_SEQUENCE_ID" ;

    /**
     * デフォルトエラーメッセージ.
     */
    private static final String DEFAULT_ERROR_MESSAGE = "エラーが発生しました";

    /**
     * エラーページ名.
     */
    protected static final String DEFAULT_ERROR_PAGE_NAME = "/error.html";

    /**
     * アクションパッケージ名.
     */
    private static final String ACTION_PACKAGE = "actionPackage";

    /**
     * フォームパッケージ名.
     */
    private static final String BEAN_PACKAGE = "formPackage" ;

    /**
     * 拡張子名.
     */
    private static final String PLUS_NAME = "plus";

    /**
     * エラーページ名
     */
    private static final String ERROR_PAGE_NAME = "errorPage";

    /**
     * 実行アクションパッケージ名.
     */
    private String actionPackage = null;

    /**
     * 実行フォームパッケージ名.
     */
    private String formPackage = null ;

    /**
     * 有効拡張子名.
     */
    private String plusName = null;

    /**
     * エラーページ遷移先.
     */
    private String errorPage = null ;
    
    /**
     * リクエストシーケンスID.
     */
    private final SequenceSync sequence = new SequenceSync() ;
    
    /**
     * 初期化処理. <BR>
     * <BR>
     * このサーブレットの初期化処理を行います. <BR>
     * 
     * @param cfg
     *            サーブレットコンフィグが設定されます.
     * @exception ServletException
     *                サーブレット例外.
     */
    public void init(ServletConfig cfg) throws ServletException {
        LOG.info( "# actionPackage:" + cfg.getInitParameter(ACTION_PACKAGE) ) ;
        LOG.info( "# formPackage:" + cfg.getInitParameter(BEAN_PACKAGE) ) ;
        LOG.info( "# plusName:" + cfg.getInitParameter(PLUS_NAME) ) ;
        LOG.info( "# errorPage:" + cfg.getInitParameter(ERROR_PAGE_NAME) ) ;
        
        this.actionPackage = cfg.getInitParameter(ACTION_PACKAGE);
        this.formPackage = cfg.getInitParameter(BEAN_PACKAGE);
        this.plusName = cfg.getInitParameter(PLUS_NAME);
        this.errorPage = cfg.getInitParameter(ERROR_PAGE_NAME);
        
        if (this.actionPackage.endsWith(".") == false) {
            this.actionPackage = this.actionPackage + ".";
        }
        if (this.formPackage.endsWith(".") == false) {
            this.formPackage = this.formPackage + ".";
        }
        if (this.plusName.startsWith(".") == false) {
            this.plusName = "." + this.plusName;
        }
        if( this.errorPage == null || ( this.errorPage = this.errorPage.trim() ).length() <= 0 ) {
            this.errorPage = DEFAULT_ERROR_PAGE_NAME ;
        }
    }

    /**
     * HTTP-GETで処理された場合の処理を記述. <BR>
     * <BR>
     * HTTP-GETで処理された場合の処理を記述します. <BR>
     * 
     * @param request
     *            HTTPリクエストが設定されます.
     * @param response
     *            HTTPレスポンスが設定されます.
     * @exception ServletException
     *                サーブレット例外.
     * @exceptin IOException I/O例外.
     */
    protected void doGet(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {
        this.doPost(request, response);
    }

    /**
     * HTTP-POSTで処理された場合の処理を記述. <BR>
     * <BR>
     * HTTP-POSTで処理された場合の処理を記述します. <BR>
     * 
     * @param request
     *            HTTPリクエストが設定されます.
     * @param response
     *            HTTPレスポンスが設定されます.
     * @exception ServletException
     *                サーブレット例外.
     * @exceptin IOException I/O例外.
     */
    protected void doPost(HttpServletRequest request,
        HttpServletResponse response) throws ServletException, IOException {
        
        // フォワード先.
        String forwardURL = null;
        
        try {
            
            // パラメータをAttributeにセット
            setAttributeMap(request);

            // RequestWrapper.
            request = new RequestWrapper( sequence.getId(),request ) ;

            // Actionオブジェクトを取得/生成.
            String acName = getActionName( request ) ;
            Action action = getAction( acName ) ;
            acName = null ;

            // Actionに定義されているFormオブジェクトを取得.
            Object formObject = action.getFormObject() ;

            // Actionに定義されているFormオブジェクトが取得できない場合.
            if( formObject == null ) {
                // ネーミングルールでのFormオブジェクトを取得.
                String foName = getFormName( request ) ;
                formObject = getForm( foName ) ;
                foName = null ;
            }

            // Formオブジェクトを取得/生成.

            // 基本情報を生成.
            Record record = null;
            String result = null;
            ResultMessage resultAction = new ResultMessageImpl();

            // Action実行.
            try {
                // ローカル設定.
                ThreadLocalParam.getInstance().put(LocalInfo.REQUEST_LOCAL_NAME, request);
                ThreadLocalParam.getInstance().put(LocalInfo.RESPONSE_LOCAL_NAME,response);
                record = ( Record )GlobalInfo.getRecordFactory().getRecord();
                ThreadLocalParam.getInstance().put(LocalInfo.RECORD_LOCAL_NAME,record);
                
                // ページ内セッションをローカル設定.
                String innerSessionKey = ( String )request.getAttribute( INNER_PAGE_SESSION ) ;
                if( "POST".equals( request.getMethod() ) &&
                    innerSessionKey != null && GlobalInfo.getPageSessionManager() != null ) {
                    PageSession session = GlobalInfo.getPageSessionManager().getSession( innerSessionKey ) ;
                    if( session != null ) {
                        ThreadLocalParam.getInstance().put(LocalInfo.INNER_SESSION, session) ;
                    }
                }
                
                // パラメータ生成.
                Parameter params = new ParameterImpl(request);
                if( formObject != null ) {
                    params.getBean( formObject ) ;
                }
                
                // Validate実行処理
                if( action.isValidate() == true ) {
                    Validate validate = new Validate( resultAction,params ) ;
                    forwardURL = action.validate(validate,resultAction,formObject,params) ;
                    if( forwardURL != null || validate.isValidate() == true ) {
                        if( forwardURL == null ) {
                            forwardURL = validate.getErrorFroward() ;
                        }
                        if( forwardURL != null ) {
                            String errmsg = resultAction.getErrorMessage();
                            if (errmsg == null) {
                                errmsg = DEFAULT_ERROR_MESSAGE;
                            }
                            request.setAttribute(ResultMessage.ERROR_MESSAGE, errmsg);
                            record.rollback() ;
                            forward( forwardURL,request,response ) ;
                            return ;
                        }
                    }
                }
                
                // action 実行.
                result = action.execution(resultAction,formObject,params);
                
                // パラメータが存在する場合は、Attributeに再設定.
                if( formObject != null ) {
                    params.setBean( formObject ) ;
                }
                
                // 処理結果判別.
                if (resultAction.getState() == ResultMessage.SUCCESS) {
                    record.commit();
                } else {
                    record.rollback() ;
                }
            } catch (Exception e) {
                try {
                    record.rollback() ;
                } catch (Exception ee) {
                }
                throw e;
            } finally {
                try {
                    record.close();
                } catch (Exception e) {
                }
                ThreadLocalParam.getInstance().clear();
            }

            // 処理結果のForward先が設定されている.
            if (result != null) {
                request.setAttribute(ResultMessage.STATE, String
                        .valueOf(resultAction.getState()));
                // エラー条件の場合は、表示用としてエラーメッセージをセット.
                if (resultAction.getState() != ResultMessage.SUCCESS) {
                    String errmsg = resultAction.getErrorMessage();
                    if (errmsg == null) {
                        errmsg = DEFAULT_ERROR_MESSAGE;
                    }
                    request.setAttribute(ResultMessage.ERROR_MESSAGE, errmsg);
                // 正常の場合は、正常メッセージが存在する場合は、その内容をセット.
                } else {
                    String message = resultAction.getSuccessMessage();
                    if (message != null) {
                        request.setAttribute(ResultMessage.SUCCESS_MESSAGE,
                                message);
                    }
                }
                forwardURL = result;
            }

        } catch (Exception e) {
            LOG.error("## error", e);
            errorPage(request, response);
            return;
        } catch (Error err) {
            LOG.error("## error", err);
            errorPage(request, response);
            return;
        }

        // Forward情報がある場合は、Forward実行.
        if (forwardURL != null) {
            forward( forwardURL,request,response ) ;
            return;
        }
    }
    
    /**
     * パス＋要求情報名を取得.
     * <BR><BR>
     * パス＋要求情報名を取得します.
     * <BR>
     * @param req 対象のリクエストを設定します.
     * @param name 対象の条件を設定します.
     * @return String 結果が返されます.
     */
    public static final String getMarge( HttpServletRequest request,String name ) {
        String server = request.getServletPath() ;
        int p = server.lastIndexOf( "/" ) ;
        server = server.substring( 0,p+1 ) ;
        return request.getContextPath() + FileUtil.convertFullPath( server,name ) ;
    }
    
    /**
     * 対象のサーバパス+対象名を取得.
     * <BR><BR>
     * 対象のサーバパス+対象名を取得します.
     * <BR>
     * @param req 対象のリクエストを設定します.
     * @param name 対象の条件を設定します.
     * @return String 結果が返されます.
     */
    public static final String getServerName( HttpServletRequest request,String name ) {
        String server = request.getServletPath() ;
        int p = server.lastIndexOf( "/" ) ;
        server = server.substring( 0,p+1 ) ;
        return FileUtil.convertFullPath( server,name ) ;
    }
    
    /**
     * 現在のシーケンスIDを取得.
     * <BR><BR>
     * 現在のシーケンスIDを取得します.
     * <BR>
     * @param request 対象のリクエストを設定します.
     * @return int 対象のリクエストIDが返されます.
     */
    public static final int getRequestSequenceId( HttpServletRequest request ) {
        if( request != null && request instanceof RequestWrapper ) {
            return ( ( RequestWrapper )request ).getSequenceId() ;
        }
        return -1 ;
    }
    
    /**
     * リファラーから、前回のServletPath名を取得.
     * <BR><BR>
     * リファラーから、前回のServletPath名を取得します.<BR>
     * また、リファラーの内容とドメイン名+ContextPathが一致しない場合は、
     * [null]が返されます.
     * <BR>
     * @param req 対象のリクエストを設定します.
     * @return String 前回のServletPathが返されます.
     */
    public static final String getBeforeServletPath( HttpServletRequest req ) {
        String refere = req.getHeader( REFERER ) ;
        if( req == null || refere == null || ( refere = refere.trim() ).length() <= 0 ) {
            return null ;
        }
        String target = req.getRequestURL().toString() ;
        int p = target.indexOf( req.getRequestURI() ) ;
        if( p == -1 ) {
            return null ;
        }
        target = target.substring( 0,p ) + req.getContextPath() ;
        if( target.startsWith( "https://" ) ) {
            target = target.substring( "https://".length() ) ;
        }
        else {
            target = target.substring( "http://".length() ) ;
        }
        String ret = null ;
        if( refere.startsWith( "https://" ) ) {
            String ch = "https://"+target ;
            if( refere.startsWith( ch ) == true ) {
                ret = refere.substring( ch.length() ) ;
            }
        }
        else if( refere.startsWith( "http://" ) ) {
            String ch = "http://"+target ;
            if( refere.startsWith( ch ) == true ) {
                ret = refere.substring( ch.length() ) ;
            }
        }
        if( ret != null ) {
            int x = ret.indexOf( "?" ) ;
            if( x != -1 ) {
                ret = ret.substring( 0,p ) ;
            }
        }
        return ret ;
    }
    
    /**
     * フォワード処理.
     */
    private void forward( String forward,HttpServletRequest request,
        HttpServletResponse response ) throws ServletException, IOException {
        RequestDispatcher disp = request.getRequestDispatcher(forward);
        disp.forward(request, response);
    }

    /**
     * error処理.
     */
    private final void errorPage(HttpServletRequest req,
            HttpServletResponse res) {
        try {
            RequestDispatcher disp = req.getRequestDispatcher(errorPage);
            disp.forward(req, res);
        } catch (Exception e) {
            try {
                res.sendError(HttpServletResponse.SC_NOT_FOUND);
            } catch (Exception ee) {
            }
        }
    }

    /**
     * URLから、パッケージ名を生成.
     */
    private static final String convertPackageName(boolean mode,String name, String plus) {
        String pkgName = name.substring(0, name.length() - plus.length());
        ArrayList<String> lst = StringUtil.cutString(pkgName, "/");
        int len = lst.size();
        StringBuilder buf = new StringBuilder();
        for (int i = 0; i < len - 1; i++) {
            if (i != 0) {
                buf.append(".");
            }
            buf.append(lst.get(i));
        }
        pkgName = lst.get(len - 1);
        if (len != 1) {
            buf.append(".");
        }
        String fd = null ;
        if( mode == true ) {
            fd = "Action" ;
        }
        else {
            fd = "Form" ;
        }
        buf.append(pkgName.substring(0, 1).toUpperCase()).append(
                pkgName.substring(1, pkgName.length())).append(fd);
        pkgName = buf.toString();
        buf = null;
        lst = null;
        return StringUtil.changeString(pkgName, "/", ".");
    }

    /**
     * Attributeに情報を設定.
     */
    private static final void setAttributeMap(HttpServletRequest req)
            throws Exception {
        HashMap<String, Object> params = ConvertHttpParams.getParams(req);
        if (params != null && params.size() > 0) {
            Object[] obj = params.keySet().toArray();
            int len = params.size();
            for (int i = 0; i < len; i++) {
                String key = (String) obj[i];
                Object o = params.get(key);
                req.setAttribute(key, o);
            }
        }
    }
    
    /**
     * 条件に応じて、Servlet名からオブジェクト名を生成.
     */
    private final String getActionName( HttpServletRequest request )
        throws Exception {
        String pkgName = convertPackageName(true,request.getServletPath(),plusName);
        return new StringBuilder().append(actionPackage).append(pkgName).toString();
    }
    
    /**
     * 条件に応じて、Servlet名からフォーム名を生成.
     */
    private final String getFormName( HttpServletRequest request )
        throws Exception {
        String pkgName = convertPackageName(false,request.getServletPath(),plusName);
        return new StringBuilder().append(formPackage).append(pkgName).toString();
    }
    
    /**
     * Actionオブジェクトを生成.
     */
    private final Action getAction( String pkgName )
        throws Exception {
        Hashtable<String, Action> mp = null ;
        synchronized (NamingByObjectManager.getInstance()) {
            mp = GlobalInfo.getActionObject() ;
            if (mp == null) {
                mp = new Hashtable<String, Action>();
                GlobalInfo.setActionObject( mp ) ;
            }
        }
        Action action = null;
        try {
            synchronized (mp) {
                Object obj = mp.get(pkgName);
                if (obj == null) {
                    obj = Reflect.newObject( pkgName,Thread.currentThread().getContextClassLoader() ) ;
                    if( obj instanceof Action ) {
                        Action ac = ( Action ) obj ;
                        ac.setErrorPageName( errorPage ) ;
                        // Action内のDI対象オブジェクトを登録.
                        DiContainer container = GlobalInfo.getDiContainer() ;
                        container.setAction( ac ) ;
                        mp.put(pkgName, ac);
                    }
                    else {
                        throw new IOException( "指定オブジェクト["+pkgName+
                            "]はActionオブジェクトではありません" ) ;
                    }
                }
                action = (Action) obj;
            }
        } catch (Exception e) {
            throw e;
        } catch (Error e) {
            throw e;
        }
        return action ;
    }
    
    /**
     * フォームオブジェクトを生成.
     */
    private final Object getForm( String pkgName ) {
        Set<String> mp = null ;
        synchronized (NamingByObjectManager.getInstance()) {
            mp = GlobalInfo.getNotFormObject() ;
            if (mp == null) {
                mp = Collections.synchronizedSet( new HashSet<String>() ) ;
                GlobalInfo.setNotFormObject( mp ) ;
            }
        }
        if( mp.contains( pkgName ) == true ) {
            return null ;
        }
        Object ret = null ;
        try {
            ret = Reflect.newObject( pkgName,Thread.currentThread().getContextClassLoader() ) ;
        } catch( Exception e ) {
            mp.add( pkgName ) ;
            ret = null ;
        }
        return ret ;
    }
    
}

class RequestWrapper extends HttpServletRequestWrapper {
    private int id = -1 ;
    public RequestWrapper( int id,HttpServletRequest req ) {
        super( req ) ;
        this.id = id ;
        req.setAttribute( ActionServlet.SEQUENCE_ID_ATTRIBUTE_NAME,new Integer( id ) ) ;
    }
    public RequestDispatcher getRequestDispatcher(String path) {
        return super.getRequestDispatcher( ActionServlet.getServerName( this,path ) ) ;
    }
    public int getSequenceId() {
        return this.id ;
    }
}
