package org.maachang.engine;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import javax.servlet.RequestDispatcher;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.dao.dbms.Record;
import org.maachang.engine.conf.Config;
import org.maachang.engine.servlet.ActionServlet;
import org.maachang.engine.servlet.DownloadFilter;
import org.maachang.engine.servlet.GlobalInfo;
import org.maachang.engine.servlet.HttpdTimestamp;
import org.maachang.engine.servlet.LocalInfo;
import org.maachang.engine.servlet.PageSession;
import org.maachang.engine.servlet.Parameter;
import org.maachang.engine.servlet.ResultMessage;
import org.maachang.engine.servlet.Validate;
import org.maachang.engine.util.FileUtil;

/**
 * Actionオブジェクト.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public abstract class Action<T> {
    /**
     * LOG.
     */
    protected static final Log LOG = LogFactory.getLog(Action.class);

    /**
     * エラーページ名.
     */
    private String errorPageName = null ;

    /**
     * Formクラス定義.
     */
    private final Class defineForm ;

    /**
     * Validateフラグ.
     */
    private final boolean validateFlag ;

    /**
     * コンストラクタ.
     */
    public Action() {
        defineForm = null ;
        validateFlag = false ;
    }

    /**
     * コンストラクタ.
     * <BR><BR>
     * このActionに対してバリデーションチェックを行うかを設定します.
     * <BR>
     * @param formClass 対象となるフォームオブジェクトクラスを設定します.
     */
    protected Action( boolean validateFlag ) {
        this.defineForm = null ;
        this.validateFlag = validateFlag ;
    }

    /**
     * コンストラクタ.
     * <BR><BR>
     * このActionに対して専用のFormオブジェクトクラスを定義します.<BR>
     * この条件を指定しない場合、ActionServlet定義(web.xml)の
     * formPackage　の条件となります.
     * <BR>
     * @param formClass 対象となるフォームオブジェクトクラスを設定します.
     */
    protected Action( Class formClass ) {
        if( formClass != null ) {
            defineForm = formClass ;
        }
        else {
            defineForm = null ;
        }
        this.validateFlag = false ;
    }

    /**
     * コンストラクタ.
     * <BR><BR>
     * このActionに対して専用のFormオブジェクトクラスを定義します.<BR>
     * この条件を指定しない場合、ActionServlet定義(web.xml)の
     * formPackage　の条件となります.<BR>
     * このActionに対してバリデーションチェックを行うかを設定します.
     * <BR>
     * @param validateFlag このActionがバリデーション(入力内容チェック)を
     *                     行う場合は[true]を設定します.
     * @param formClass 対象となるフォームオブジェクトクラスを設定します.
     */
    protected Action( boolean validateFlag,Class formClass ) {
        if( formClass != null ) {
            defineForm = formClass ;
        }
        else {
            defineForm = null ;
        }
        this.validateFlag = validateFlag ;
    }

    /**
     * アクション処理. <BR>
     * <BR>
     * 対象のアクション処理を設定します. <BR>
     * 
     * @param result
     *            処理結果のメッセージを設定します.
     * @param formObject
     *            対象のフォームオブジェクトが設定されます.
     * @param parameter
     *            設定されたパラメータが設定されます.
     * @return String forward先を設定します.
     * @exception Exception
     *                例外.
     */
    public abstract String execution(ResultMessage result,T formObject,Parameter parameter) throws Exception;

    /**
     * Validate処理.
     * <BR><BR>
     * Validate処理を行います.
     * <BR>
     * @param validate
     *            Validateオブジェクトを設定します.
     * @param result
     *            処理結果のメッセージを設定します.
     * @param formObject
     *            対象のフォームオブジェクトが設定されます.
     * @param parameter
     *            設定されたパラメータが設定されます.
     * @return String forward先を設定します.<BR>
     *            [null]を設定することで、正常終了となります.
     * @exception Exception
     *                例外.
     */
    public String validate(Validate validate,ResultMessage result,T formObject,Parameter parameter) throws Exception {
        if( validateFlag == true ) {
            throw new IllegalArgumentException( "Validate処理を実装していません" ) ;
        }
        return null ;
    }

    /**
     * キャッシュをOFFに設定.
     * <BR><BR>
     * キャッシュをOFFに設定します.
     */
    protected void cacheOff() {
        HttpServletResponse res = LocalInfo.getResponse() ;
        res.setHeader( "Expires",HttpdTimestamp.getTimestamp( 0L ) ) ;
        res.setHeader( "Pragma","no-cache" ) ;
        res.setHeader( "Last-Modified",HttpdTimestamp.getNowTimestamp() ) ;
        res.setHeader( "Cache-Control","private" ) ;
        res.addHeader( "Cache-Control","no-cache" ) ;
        res.addHeader( "Cache-Control","no-store" ) ;
        res.addHeader( "Cache-Control","max-age=0" ) ;
    }

    /**
     * 指定URLにリダイレクト. <BR>
     * <BR>
     * 指定URLにリダイレクトします. <BR>
     * 
     * @param url
     *            リダイレクト先URLを設定します.
     * @exception Exception
     *                例外.
     */
    protected void redirect(String url) throws Exception {
        if (url == null || (url = url.trim()).length() <= 0) {
            return;
        }
        try {
            HttpServletRequest req = LocalInfo.getRequest() ;
            HttpServletResponse res = LocalInfo.getResponse() ;
            if (req == null || res == null) {
                return;
            }
            String ctx = req.getContextPath();
            if (url.startsWith("/") && url.startsWith(ctx + "/") == false) {
                url = ctx + url;
            }
            res.sendRedirect(url);
        } finally {
            LocalInfo.clear() ;
        }
    }

    /**
     * 指定URLにフォワード. <BR>
     * <BR>
     * 指定URLにフォワードします. <BR>
     * 
     * @param url
     *            リダイレクト先URLを設定します.
     * @exception Exception
     *                例外.
     */
    protected void forward(String url) throws Exception {
        if (url == null || (url = url.trim()).length() <= 0) {
            return;
        }
        try {
            HttpServletRequest req = LocalInfo.getRequest() ;
            HttpServletResponse res = LocalInfo.getResponse() ;
            if (req == null || res == null) {
                return;
            }
            String ctx = req.getContextPath();
            if (url.startsWith("/") && url.startsWith(ctx + "/") == false) {
                url = ctx + url;
            }
            RequestDispatcher disp = req.getRequestDispatcher(url);
            disp.forward(req, res);
        } finally {
            LocalInfo.clear() ;
        }
    }
    
    /**
     * ダウンロード情報を設定.
     * <BR><BR>
     * ダウンロード情報を設定します.
     * <BR>
     * @param fileName ダウンロードファイル名を設定します.
     * @param binary 対象のバイナリ情報を設定します.
     * @exception Exception 例外.
     */
    protected void forwardDownload( String fileName,byte[] binary )
        throws Exception {
        forwardDownload( fileName,null,new ByteArrayInputStream( binary ) ) ;
    }
    
    /**
     * ダウンロード情報を設定.
     * <BR><BR>
     * ダウンロード情報を設定します.
     * <BR>
     * @param fileName ダウンロードファイル名を設定します.
     * @param binary 対象のバイナリ情報を設定します.
     * @exception Exception 例外.
     */
    protected void forwardDownload( String fileName,InputStream stream )
        throws Exception {
        forwardDownload( fileName,null,stream ) ;
    }
    
    /**
     * ダウンロード情報を設定.
     * <BR><BR>
     * ダウンロード情報を設定します.
     * <BR>
     * @param fileName ダウンロードファイル名を設定します.
     * @param mimeType ダウンロードMimeTypeを設定します.
     * @param binary 対象のバイナリ情報を設定します.
     * @exception Exception 例外.
     */
    protected void forwardDownload( String fileName,String mimeType,byte[] binary )
        throws Exception {
        forwardDownload( fileName,mimeType,new ByteArrayInputStream( binary ) ) ;
    }
    
    /**
     * ダウンロード情報を設定.
     * <BR><BR>
     * ダウンロード情報を設定します.
     * <BR>
     * @param fileName ダウンロードファイル名を設定します.
     * @param mimeType ダウンロードMimeTypeを設定します.
     * @param binary 対象のバイナリ情報を設定します.
     * @exception Exception 例外.
     */
    protected void forwardDownload( String fileName,String mimeType,InputStream stream )
        throws Exception {
        if( fileName == null || ( fileName = fileName.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ファイル名は不正です" ) ;
        }
        String key = DownloadFilter.setDownloadInfo( LocalInfo.getRequest(),mimeType,stream ) ;
        this.redirect( fileName+"?"+DownloadFilter.DOWNLOAD_PARAM+"="+key ) ;
    }
    
    /**
     * リクエストオブジェクトを取得.
     * <BR><BR>
     * リクエストオブジェクトを取得します.
     * <BR>
     * @return HttpServletRequest リクエストオブジェクトが返されます.
     */
    protected HttpServletRequest request() {
        return LocalInfo.getRequest() ;
    }
    
    /**
     * レスポンスオブジェクトを取得.
     * <BR><BR>
     * レスポンスオブジェクトを取得します.
     * <BR>
     * @return HttpServletResponse レスポンスオブジェクトが返されます.
     */
    protected HttpServletResponse response() {
        return LocalInfo.getResponse() ;
    }
    
    /**
     * レコードオブジェクトを取得.
     * <BR><BR>
     * レコードオブジェクトを取得します.
     * <BR>
     * @return Record レコードオブジェクトが返されます.
     */
    protected Record record() {
        return LocalInfo.getRecord() ;
    }
    
    /**
     * シーケンスIDを取得.
     * <BR><BR>
     * 対象のシーケンスIDを取得します.
     * <BR>
     * @return int シーケンスIDが返されます.
     */
    protected int getSequenceId() {
        return ActionServlet.getRequestSequenceId( request() ) ;
    }
    
    /**
     * ページ内セッションを取得.
     * <BR><BR>
     * ページ内セッションを取得します.
     * <BR>
     * @return PageSession ページ内セッションを取得します.
     */
    protected PageSession pageSession() {
        return LocalInfo.getInnerSession() ;
    }
    
    /**
     * 基本コンフィグ情報を取得.
     * <BR><BR>
     * 基本コンフィグ情報を取得します.
     * <BR>
     * @return Config 基本コンフィグ情報が返されます.
     */
    protected Config getConfig() {
        return GlobalInfo.getConfig() ;
    }
    
    /**
     * リアルパスを取得.
     * <BR><BR>
     * リアルパスを取得します.
     * <BR>
     * @param path 対象のパスを設定します.
     * @return String リアルパスが返されます.
     * @exception Exception 例外.
     */
    public String getRealPath( String path )
        throws Exception {
        return FileUtil.marge( GlobalInfo.getRealPath(),path ) ;
    }
    
    /**
     * エラーページ名を設定.
     * <BR><BR>
     * エラーページ名を設定します.
     * <BR>
     * @param errorPageName 対象のエラーページ名を設定します.
     */
    public void setErrorPageName( String errorPageName ) {
        if( this.errorPageName == null ) {
            this.errorPageName = errorPageName ;
        }
    }
    
    /**
     * エラーページ名を取得.
     * <BR><BR>
     * エラーページ名を取得します.
     * <BR>
     * @return String エラーページ名が返されます.
     */
    public String getErrorPageName() {
        return errorPageName ;
    }
    
    /**
     * 指定フォームオブジェクトを取得.
     * <BR><BR>
     * 指定されたフォームオブジェクトを取得します.
     * <BR>
     * @return Object 指定されたフォームオブジェクトが返されます.<BR>
     *                [null]の場合は、フォームオブジェクト定義が存在しないか、
     *                デフォルトコンストラクタでインスタンス生成できない条件です.
     */
    public Object getFormObject() {
        Object ret = null ;
        if( defineForm != null ) {
            try {
                ret = defineForm.newInstance() ;
            } catch( Exception e ) {
                ret = null ;
            }
        }
        return ret ;
    }
    
    /**
     * Validate条件を取得.
     * <BR><BR>
     * Validate条件を取得します.
     * <BR>
     * @return boolean [true]の場合、Validateチェックします.
     */
    public boolean isValidate() {
        return validateFlag ;
    }
}
