package org.maachang.dao.dbms;

import java.io.IOException;
import java.sql.Connection;

import org.maachang.dao.dbms.kind.SupportKind;
import org.maachang.dao.dbms.pool.ConnectManager;
import org.maachang.dao.dbms.pool.ConnectPoolFactory;

/**
 * レコード管理.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class RecordFactory {

    /**
     * サポートDB種類情報.
     */
    private SupportKind kind = null ;
    
    /**
     * レコード管理オブジェクト.
     */
    private ConnectManager manager = null;
    
    /**
     * AutoComitモード.
     */
    private boolean autoComit = false ;

    /**
     * コンストラクタ.
     */
    public RecordFactory() {

    }

    /**
     * オブジェクト生成. <BR>
     * <BR>
     * コネクション条件を設定してオブジェクトを生成します. <BR>
     * 
     * @param kind
     *            対象のDBMS種類を設定します.
     * @param driver
     *            対象のドライバ名を設定します.
     * @param url
     *            接続先のURLを設定します.
     * @param user
     *            対象のユーザ名を設定します.
     * @param passwd
     *            対象のポート番号を設定します.
     * @param max
     *            最大プーリングコネクション管理要素を設定します.
     * @param autoComit
     *            AutoComitモードを設定します.
     * @param checkSQL
     *            コネクションチェックSQLを設定します.
     * @exception Exception
     *                例外.
     */
    public void create(SupportKind kind,String driver, String url, String user, String passwd,
            int max, boolean autoComit,String checkSQL) throws Exception {
        if (driver == null || (driver = driver.trim()).length() <= 0
                || url == null || (url = url.trim()).length() <= 0) {
            throw new IllegalArgumentException("必須条件[adapter or driver or url]が設定されていません");
        }
        else if( kind == null ) {
            throw new IllegalArgumentException("非サポートのDBMS種類が設定されています");
        }
        // アダプタ名を設定.
        this.kind = kind ;
        // ドライバをロード.
        Class.forName(driver);
        // コネクションオブジェクトを取得.
        ConnectManager manager = ConnectPoolFactory.get(url, user, passwd, max,
                checkSQL);
        // AutoComitモード.
        autoComit = false ;

        // メンバー変数に設定.
        this.manager = manager;
    }

    /**
     * 新しいレコードオブジェクトを取得. <BR>
     * <BR>
     * 新しいレコードオブジェクトを取得します. <BR>
     * 
     * @return Record 新しいレコードオブジェクトが返されます.
     * @exception Exception
     *                例外.
     */
    public Record getRecord() throws Exception {
        Connection conn = manager.getConnection();
        if (conn == null) {
            throw new IOException("コネクションオブジェクトの取得に失敗しました");
        }
        conn.setAutoCommit( autoComit ) ;
        return new BaseRecord(kind,conn);
    }

    /**
     * サポートDBMS種類オブジェクトを取得.
     * <BR><BR>
     * サポートDBMS種類オブジェクトを取得します.
     * <BR>
     * @return SupportKind サポートDBMS種類が返されます.
     */
    public SupportKind getSupportKind() {
        return kind ;
    }

    /**
     * コネクションマネージャを取得. <BR>
     * <BR>
     * コネクションマネージャを取得します. <BR>
     * 
     * @return ConnectManager コネクションマネージャが返されます.
     */
    public ConnectManager getConnectionManager() {
        return manager;
    }
    
    /**
     * AutoComitモードを取得.
     * <BR><BR>
     * AutoComitモードを取得します.
     * <BR>
     * @return boolean [true]の場合はAutoComitが有効です.
     */
    public boolean isAutoComit() {
    	return autoComit ;
    }
}
