package org.maachang.dao.dbms ;

import java.io.IOException;
import java.sql.ResultSet;
import java.util.ArrayList;

import org.maachang.dao.dbms.kind.SupportKind;
import org.maachang.engine.util.Reflect;

/**
 * Dao実行オブジェクト.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class ExecutionDao {
    
    /**
     * SQL文[select].
     */
    private static final String RESULT_BY_SQL = "select" ;
    
    /**
     * SQL文[insert].
     */
    private static final String INSERT_BY_SQL = "insert" ;
    
    /**
     * シーケンス付加対象パラメータ.
     */
    public static final String SEQ_COLUMN = "id" ;
    
    /**
     * テーブルカラム生成日付付加パラメータ.
     */
    public static final String CREATE_TIME = "create_time" ;
    
    /**
     * テーブルカラム更新日付付加パラメータ.
     */
    public static final String UPDATE_TIME = "update_time" ;
    
    /**
     * SQL実行.
     * <BR><BR>
     * 指定SQLを実行します.
     * <BR>
     * @param out select文の実行結果が返されます.
     * @param outId insert文の実行時のシーケンスIDが返されます.
     * @param record SQL実行レコードを設定します.
     * @param type select文実行結果で返されるオブジェクトクラスを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param offset 取得オフセット値を設定します.
     * @param limit 取得リミット値を設定します.
     * @param params パラメータ群を設定します.
     * @return int 実行件数が返されます.
     * @exception Exception 例外.
     */
    public static final int execution( ArrayList out,Long[] outId,
        Record record,Class type,String model,MetaColumn meta,
        String sql,int offset,int limit,ArrayList<Object> params )
        throws Exception {
        Object[] o = null ;
        if( params != null && params.size() > 0 ) {
            int len = params.size() ;
            o = new Object[ len ] ;
            for( int i = 0 ; i < len ; i ++ ) {
                o[ i ] = params.get( i ) ;
            }
        }
        return execution( out,outId,record,type,model,meta,sql,offset,limit,o ) ;
    }
    
    /**
     * SQL実行.
     * <BR><BR>
     * 指定SQLを実行します.
     * <BR>
     * @param out select文の実行結果が返されます.
     * @param outId insert文の実行時のシーケンスIDが返されます.
     * @param record SQL実行レコードを設定します.
     * @param type select文実行結果で返されるオブジェクトクラスを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param offset 取得オフセット値を設定します.
     * @param limit 取得リミット値を設定します.
     * @param params パラメータ群を設定します.
     * @return int 実行件数が返されます.
     * @exception Exception 例外.
     */
    public static final int execution( ArrayList out,Long[] outId,
        Record record,Class type,String model,MetaColumn meta,
        String sql,int offset,int limit,Object[] params )
        throws Exception {
        if( record == null ) {
            return 0 ;
        }
        if( sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "SQL文は不正です" ) ;
        }
        SupportKind kind = record.getSupportKind() ;
        if( kind == null ) {
            throw new IOException( "サポート外のアダプタです" ) ;
        }
        String lower = sql.toLowerCase() ;
        boolean selectFlag = lower.startsWith( RESULT_BY_SQL ) ;
        boolean insertFlag = lower.startsWith( INSERT_BY_SQL ) ;
        
        // select実行.
        if( selectFlag == true ) {
            ResultSet rs = null ;
            try {
                // SQL文実行.
                if( params == null || params.length <= 0 ) {
                    rs = record.executeQuery( sql ) ;
                }
                else {
                    rs = record.executeQuery( sql,params ) ;
                }
                if( lower.startsWith( "select count(*)" ) ) {
                    int count = 0 ;
                    for( ;; ) {
                        if( rs.next() == false ) {
                            break ;
                        }
                        count = rs.getInt( 1 ) ;
                        break ;
                    }
                    rs.close() ;
                    rs = null ;
                    return count ;
                }
                else {
                    out.clear() ;
                    if( limit <= 0 ) {
                        limit = Integer.MAX_VALUE ;
                    }
                    if( offset <= 0 ) {
                        offset = 0 ;
                    }
                    pushResult( out,type,rs,offset,limit ) ;
                    rs.close() ;
                    rs = null ;
                    return out.size() ;
                }
            } finally {
                if( rs != null ) {
                    try {
                        rs.close() ;
                    } catch( Exception e ) {
                    }
                }
                rs = null ;
            }
        }
        // select以外で実行.
        else {
            boolean seqFlag = isSequence( meta ) ;
            Long seq = null ;
            // 最初にシーケンスIDを取得.
            if( outId != null && outId.length >= 1 &&
                seqFlag == true && insertFlag == true ) {
                String s = kind.getSequenceId( model ) ;
                if( s != null ) {
                    seq = getSequenceId( record,s ) ;
                    if( seq != null ) {
                        // シーケンスIDが取得できた場合は、
                        // パラメータ0にセット.
                        //(insert文生成時に、カラムのはじめを[id]に必ず設定し、
                        //params[0] == nullにする必要がある).
                        params[ 0 ] = seq ;
                    }
                }
            }
            // SQL文実行.
            int ret = 0 ;
            if( params == null || params.length <= 0 ) {
                ret = record.executeUpdate( sql ) ;
            }
            else {
                ret = record.executeUpdate( sql,params ) ;
            }
            // SQL実行後にシーケンスIDを取得.
            if( outId != null && outId.length >= 1 &&
                seqFlag == true && insertFlag == true && seq != null ) {
                String s = kind.getInsertIdBySQL() ;
                if( s != null ) {
                    seq = getInsertId( record,s ) ;
                }
            }
            // シーケンスが取得できた場合は、outIdに設定.
            if( outId != null ) {
                outId[ 0 ] = seq ;
            }
            return ret ;
        }
    }
    
    /**
     * シーケンス付与対象のパラメータが存在するかチェック.
     * <BR><BR>
     * シーケンス付与対象のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、シーケンス付与条件です.
     */
    public static final boolean isSequence( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( SEQ_COLUMN.equals( meta.getColumnName( i ).toLowerCase() ) ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * 生成時間のパラメータが存在するかチェック.
     * <BR><BR>
     * 生成時間のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、生成時間パラメータが存在します.
     */
    public static final boolean isCreateTime( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( CREATE_TIME.equals( meta.getColumnName( i ).toLowerCase() ) ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * 更新時間のパラメータが存在するかチェック.
     * <BR><BR>
     * 更新時間のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、更新時間パラメータが存在します.
     */
    public static final boolean isUpdateTime( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( UPDATE_TIME.equals( meta.getColumnName( i ).toLowerCase() ) ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * シーケンスIDが利用できる場合は取得.
     */
    private static final Long getSequenceId( Record record,String sql )
        throws Exception {
        ResultSet result = null ;
        try {
            // シーケンスが利用できる場合.
            result = record.executeQuery( sql ) ;
            if( result != null ) {
                result.next() ;
                Long ret = result.getLong( 1 ) ;
                return ret ;
            }
            return null ;
        } catch( Exception e ) {
            throw e ;
        } finally {
            if( result != null ) {
                try {
                    result.close() ;
                } catch( Exception e ) {
                }
            }
        }
    }
    
    /**
     * Inset後にシーケンスIDを取得できる場合.
     */
    private static final Long getInsertId( Record record,String sql )
        throws Exception {
        ResultSet result = null ;
        try {
            // シーケンスが利用できる場合.
            result = record.executeQuery( sql ) ;
            if( result != null ) {
                result.next() ;
                Long ret = result.getLong( 1 ) ;
                return ret ;
            }
            return null ;
        } catch( Exception e ) {
            throw e ;
        } finally {
            if( result != null ) {
                try {
                    result.close() ;
                } catch( Exception e ) {
                }
            }
        }
    }
    
    /**
     * 実行結果を情報に設定.
     */
    private static final void pushResult( ArrayList out,Class type,ResultSet result,int offset,int limit )
        throws Exception {
//        ResultUtil.setPosition( result,offset ) ;
//        ResultUtil.firsetResult( result ) ;
//        for( int i = 0 ; i < limit ; i ++ ) {
//            Object o = Reflect.newObject( type ) ;
//            DbUtil.convertResultByObject( o,result ) ;
//            out.add( o ) ;
//            if( result.next() == false ) {
//                break ;
//            }
//        }
        ResultUtil.setPosition( result,offset ) ;
        for( int i = 0 ; i < limit ; i ++ ) {
            if( result.next() == false ) {
                break ;
            }
            Object o = Reflect.newObject( type ) ;
            DbUtil.convertResultByObject( o,result ) ;
            out.add( o ) ;
        }
    }
}

