package org.maachang.dao ;

import java.io.IOException;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;

import org.maachang.dao.dbms.DbUtil;
import org.maachang.dao.dbms.ExecutionDao;
import org.maachang.dao.dbms.MetaColumn;
import org.maachang.dao.dbms.Record;
import org.maachang.engine.servlet.GlobalInfo;
import org.maachang.engine.servlet.LocalInfo;
import org.maachang.engine.util.Reflect;

/**
 * Daoオブジェクト.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public abstract class Dao<T> {
    
    /**
     * Entryタイプ.
     */
    private Class entryType = null ;
    
    /**
     * モデル名.
     */
    private String model = null ;
    
    /**
     * メタカラム.
     */
    private MetaColumn meta = null ;
    
    /**
     * コンストラクタ.
     */
    public Dao() {
        throw new UnsupportedOperationException(
            "このコンストラクタは非サポートです" ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 利用対象のEntryタイプを設定してオブジェクトを生成します.
     * <BR>
     * @param type 利用対象のEntryタイプを設定します.
     * @exception Exception 例外.
     */
    protected Dao( Class<T> type ) throws Exception {
        if( type == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        Record rc = GlobalInfo.getRecordFactory().getRecord() ;
        try {
            this.entryType = type ;
            this.model = modelName( type ) ;
            this.meta = getMetaColumn( rc,model ) ;
            rc.close() ;
            rc = null ;
        } finally {
            if( rc != null ) {
                try {
                    rc.close() ;
                } catch( Exception e ) {
                }
            }
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 利用対象のEntryタイプを設定してオブジェクトを生成します.
     * <BR>
     * @param type 利用対象のEntryタイプを設定します.
     * @param record 対象のレコードオブジェクトを設定します.
     * @exception Exception 例外.
     */
    protected Dao( Class<T> type,Record record ) throws Exception {
        if( type == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.entryType = type ;
        this.model = modelName( type ) ;
        this.meta = getMetaColumn( record,model ) ;
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * コネクションをクローズします.
     * <BR>
     * @exception Exception 例外.
     */
    protected void close()
        throws Exception {
        getRecord().close() ;
    }
    
    /**
     * コミット処理.
     * <BR><BR>
     * コミット処理を実行します.
     * <BR>
     * @return boolean [true]が返された場合、コミットが実行されたことを
     *                 意味します.
     * @exception Exception 例外.
     */
    protected boolean commit()
        throws Exception {
        return getRecord().commit() ;
    }
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * ロールバック処理を実行します.
     * <BR>
     * @return boolean [true]が返された場合、ロールバックが実行されたことを
     *                 意味します.
     * @exception Exception 例外.
     */
    protected boolean rollback()
        throws Exception {
        return getRecord().rollback() ;
    }
    
    /**
     * データ保存.
     * <BR><BR>
     * データを保存します.
     * <BR>
     * @param o 保存対象のEntryを設定します.
     * @exception Exception 例外.
     */
    public T save( T o ) throws Exception {
        initCheck() ;
        if( o == null ) {
            throw new IllegalArgumentException( "保存対象のオブジェクトが設定されていません" ) ;
        }
        Record record = getRecord() ;
        boolean idFlag = ExecutionDao.isSequence( meta ) ;
        // データが存在するかチェック.
        // または、IDカラムが存在しない場合は、強制的に
        // insertにする.
        if( checkEntry( record,idFlag,o ) == true ) {
            // データが存在する場合は、update.
            StringBuilder buf = new StringBuilder() ;
            buf.append( "update " ).append( meta.getTable() ).append( " set " ) ;
            int len = meta.size() ;
            int cnt = 0 ;
            ArrayList<Object> pms = new ArrayList<Object>() ;
            for( int i = 0 ; i < len ; i ++ ) {
                String column = meta.getColumnName( i ) ;
                if( ExecutionDao.SEQ_COLUMN.equals( column ) == false ) {
                    Object value = null ;
                    // 更新カラムが存在する場合.
                    if( ExecutionDao.UPDATE_TIME.equals( column ) ) {
                        value = new java.sql.Timestamp( System.currentTimeMillis() ) ;
                        Reflect.executionMethod( convertMethodName( "set",column ),o,value ) ;
                    }
                    else {
                        try {
                            value = Reflect.executionMethod( convertMethodName( "get",column ),o ) ;
                        } catch( Exception e ) {
                            value = null ;
                        }
                        if( value == null ) {
                            try {
                                value = Reflect.executionMethod( convertMethodName( "is",column ),o ) ;
                            } catch( Exception e ) {
                                value = null ;
                            }
                        }
                    }
                    if( value != null ) {
                        if( cnt != 0 ) {
                            buf.append( "," ) ;
                        }
                        buf.append( column ).append( "=" ).
                            append( "?" ) ;
                        pms.add( value ) ;
                        cnt ++ ;
                    }
                }
            }
            buf.append( " where id=?" ) ;
            if( pms.size() <= 0 ) {
                throw new IllegalArgumentException( "保存対象のパラメータが全てNULLです" ) ;
            }
            String sql = buf.toString() ;
            buf = null ;
            Object id = Reflect.executionMethod( "getId",o ) ;
            pms.add( id ) ;
            ExecutionDao.execution( null,null,record,entryType,model,meta,sql,-1,-1,pms ) ;
        }
        // データが存在しない場合Insert.
        else {
            StringBuilder buf = new StringBuilder() ;
            buf.append( "insert into " ).append( meta.getTable() ).append( " (" ) ;
            int len = meta.size() ;
            ArrayList<Object> pms = new ArrayList<Object>() ;
            if( idFlag == true && record.isSequence() == true ) {
                pms.add( new Object() ) ;
                buf.append( ExecutionDao.SEQ_COLUMN ) ;
                if( len != 1 ) {
                    buf.append( "," ) ;
                }
            }
            int cnt = 0 ;
            for( int i = 0 ; i < len ; i ++ ) {
                String column = meta.getColumnName( i ) ;
                if( ExecutionDao.SEQ_COLUMN.equals( column ) == false ) {
                    Object value = null ;
                    // 生成/更新カラムが存在する場合.
                    if( ExecutionDao.CREATE_TIME.equals( column ) ||
                        ExecutionDao.UPDATE_TIME.equals( column ) ) {
                        value = new java.sql.Timestamp( System.currentTimeMillis() ) ;
                        Reflect.executionMethod( convertMethodName( "set",column ),o,value ) ;
                    }
                    else {
                        try {
                            value = Reflect.executionMethod( convertMethodName( "get",column ),o ) ;
                        } catch( Exception e ) {
                            value = null ;
                        }
                        if( value == null ) {
                            try {
                                value = Reflect.executionMethod( convertMethodName( "is",column ),o ) ;
                            } catch( Exception e ) {
                                value = null ;
                            }
                        }
                    }
                    if( cnt != 0 ) {
                        buf.append( "," ) ;
                    }
                    buf.append( column ) ;
                    if( value == null ) {
                        pms.add( null ) ;
                    }
                    else {
                        pms.add( value ) ;
                    }
                    cnt ++ ;
                }
            }
            buf.append( ") values (" ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) {
                    buf.append( "," ) ;
                }
                buf.append( "?" ) ;
            }
            buf.append( ")" ) ;
            String sql = buf.toString() ;
            Long[] resId = new Long[ 1 ] ;
            ExecutionDao.execution( null,resId,record,entryType,model,meta,sql,-1,-1,pms ) ;
            if( resId[ 0 ] != null ) {
                Reflect.executionMethod( "setId",o,resId[ 0 ] ) ;
            }
        }
        return o ;
    }
    
    /**
     * アップデート.
     * <BR><BR>
     * 指定条件のデータをアップデートします.
     * <BR>
     * @param set update文の[set]以降の条件を設定します.
     * @param params 対象のパラメータ群を設定します.
     * @exception Exception 例外.
     */
    public void update( String set,Object... params ) throws Exception {
        initCheck() ;
        if( set == null || ( set = set.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "set引数が存在しません" ) ;
        }
        StringBuilder buf = new StringBuilder() ;
        if( set.toLowerCase().startsWith( "set " ) == false ) {
            buf.append( "update " ).append( meta.getTable() ).
                append( " set " ).append( set ) ;
        }
        else {
            buf.append( "update " ).append( meta.getTable() ).append( " " ).append( set ) ;
        }
        String sql = buf.toString() ;
        buf = null ;
        if( ExecutionDao.isUpdateTime( meta ) == true ) {
            boolean[] addFlag = new boolean[ 1 ] ;
            sql = addUpdateTime( addFlag,sql ) ;
            if( addFlag[ 0 ] == true ) {
                params = addUpdateTimeParam( params ) ;
            }
        }
        ExecutionDao.execution( null,null,getRecord(),entryType,model,meta,sql,-1,-1,params ) ;
    }
    
    /**
     * 削除.
     * <BR><BR>
     * 指定Entryの情報を削除します.
     * <BR>
     * @param o 削除対象のEntryを設定します.
     * @exception Exception 例外.
     */
    public void delete( T o ) throws Exception {
        initCheck() ;
        if( o == null ) {
            throw new IllegalArgumentException( "削除対象のオブジェクトが設定されていません" ) ;
        }
        if( ExecutionDao.isSequence( meta ) == true ) {
            StringBuilder buf = new StringBuilder() ;
            buf.append( "delete from " ).append( meta.getTable() ).append( " where id=?;" ) ;
            Object id = Reflect.executionMethod( "getId",o ) ;
            if( o == null ) {
                throw new IllegalArgumentException( "IDが存在しません" ) ;
            }
            String sql = buf.toString() ;
            buf = null ;
            ExecutionDao.execution( null,null,getRecord(),entryType,model,meta,sql,-1,-1,new Object[]{id} ) ;
        }
        else {
            StringBuilder buf = new StringBuilder() ;
            buf.append( "delete from " ).append( meta.getTable() ).append( " where " ) ;
            int len = meta.size() ;
            ArrayList<Object> pms = new ArrayList<Object>() ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) {
                    buf.append( " and " ) ;
                }
                String column = meta.getColumnName( i ) ;
                Object value = null ;
                try {
                    value = Reflect.executionMethod( convertMethodName( "get",column ),o ) ;
                } catch( Exception e ) {
                    value = null ;
                }
                if( value == null ) {
                    try {
                        value = Reflect.executionMethod( convertMethodName( "is",column ),o ) ;
                    } catch( Exception e ) {
                        value = null ;
                    }
                }
                if( value == null ) {
                    buf.append( column ).append( " is null" ) ;
                }
                else {
                    buf.append( column ).append( "=?" ) ;
                    pms.add( value ) ;
                }
            }
            buf.append( ";" ) ;
            if( pms.size() <= 0 ) {
                pms = null ;
            }
            String sql = buf.toString() ;
            buf = null ;
            ExecutionDao.execution( null,null,getRecord(),entryType,model,meta,sql,-1,-1,pms ) ;
        }
    }
    
    /**
     * 削除.
     * <BR><BR>
     * 指定条件のデータを削除します.
     * <BR>
     * @param where 条件文を設定します.
     * @param params 対照のパラメータ群を設定します.
     * @exception Exception 例外.
     */
    public void delete( String where,Object... params ) throws Exception {
        initCheck() ;
        if( where == null || ( where = where.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "where引数が存在しません" ) ;
        }
        StringBuilder buf = new StringBuilder() ;
//        if( where.toLowerCase().startsWith( "where " ) == false ) {
//            buf.append( "delete " ).append( meta.getTable() ).
//                append( " from where " ).append( where ) ;
//        }
//        else {
            buf.append( "delete from " ).append( meta.getTable() ).
                append( " " ).append( where ) ;
//        }
        String sql = buf.toString() ;
        buf = null ;
        ExecutionDao.execution( null,null,getRecord(),entryType,model,meta,sql,-1,-1,params ) ;
    }
    
    /**
     * 取得.
     * <BR><BR>
     * 指定条件のデータを取得します.
     * <BR>
     * @param where 条件文を設定します.
     * @param params 対照のパラメータ群を設定します.
     * @exception List<T> 情報が返されます.
     * @exception Exception 例外.
     */
    public List<T> find( String where,Object... params ) throws Exception {
        return find( where,-1,-1,params ) ;
    }
    
    /**
     * 取得.
     * <BR><BR>
     * 指定条件のデータを取得します.
     * <BR>
     * @param where 条件文を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param limit 対象のリミットを設定します.
     * @param params 対照のパラメータ群を設定します.
     * @exception List<T> 情報が返されます.
     * @exception Exception 例外.
     */
    public List<T> find( String where,int offset,int limit,Object... params ) throws Exception {
        initCheck() ;
        String sql = null ;
        if( where == null || ( where = where.trim() ).length() <= 0 ) {
            sql = new StringBuilder().append( "select * from " ).
                append( meta.getTable() ).toString() ;
            params = null ;
        }
        else {
            StringBuilder buf = new StringBuilder() ;
//            if( where.toLowerCase().startsWith( "where " ) == false ) {
//                buf.append( "select * from " ).append( meta.getTable() ).
//                    append( " where " ).append( where ) ;
//            }
//            else {
                buf.append( "select * from " ).append( meta.getTable() ).
                    append( " " ).append( where ) ;
//            }
            sql = buf.toString() ;
            buf = null ;
        }
        ArrayList<T> ret = new ArrayList<T>() ;
        ExecutionDao.execution( ret,null,getRecord(),entryType,model,meta,sql,offset,limit,params ) ;
        return ret ;
    }
    
    /**
     * 最初の情報を取得.
     * <BR><BR>
     * 最初の情報を取得します.
     * <BR>
     * @exception T 最初の情報が返されます.
     * @exception Exception 例外.
     */
    public T findFirst() throws Exception {
        List<T> res = find( null,0,1,( Object[] )null ) ;
        if( res != null && res.size() > 0 ) {
            return res.get( 0 ) ;
        }
        return null ;
    }
    
    /**
     * 最初の情報を取得.
     * <BR><BR>
     * 最初の情報を取得します.
     * <BR>
     * @param where 条件文を設定します.
     * @param params 対照のパラメータ群を設定します.
     * @exception T 最初の情報が返されます.
     * @exception Exception 例外.
     */
    public T findFirst( String where,Object... params ) throws Exception {
        List<T> res = find( where,0,1,params ) ;
        if( res != null && res.size() > 0 ) {
            return res.get( 0 ) ;
        }
        return null ;
    }
    
    /**
     * 全取得.
     * <BR><BR>
     * 全データ情報を取得します.
     * <BR>
     * @exception List<T> 情報が返されます.
     * @exception Exception 例外.
     */
    public List<T> findAll() throws Exception {
        return findAll( -1,-1 ) ;
    }
    
    /**
     * 全取得.
     * <BR><BR>
     * 全データ情報を取得します.
     * <BR>
     * @param offset 対象のオフセット値を設定します.
     * @param limit 対象のリミットを設定します.
     * @exception List<T> 情報が返されます.
     * @exception Exception 例外.
     */
    public List<T> findAll( int offset,int limit ) throws Exception {
        initCheck() ;
        String sql = new StringBuilder().append( "select * from " ).append( meta.getTable() ).
            append( ";" ).toString() ;
        ArrayList<T> ret = new ArrayList<T>() ;
        ExecutionDao.execution( ret,null,getRecord(),entryType,model,meta,sql,offset,limit,( Object[] )null ) ;
        return ret ;
    }
    
    /**
     * 情報数を取得.
     * <BR><BR>
     * 指定条件の情報数を取得します.
     * <BR>
     * @param where 条件文を設定します.
     * @param params 対照のパラメータ群を設定します.
     * @exception int 情報数が返されます.
     * @exception Exception 例外.
     */
    public int count() throws Exception {
        return count( null,( Object[] )null ) ;
    }
    
    /**
     * 情報数を取得.
     * <BR><BR>
     * 指定条件の情報数を取得します.
     * <BR>
     * @param where 条件文を設定します.
     * @param params 対照のパラメータ群を設定します.
     * @exception int 情報数が返されます.
     * @exception Exception 例外.
     */
    public int count( String where,Object... params ) throws Exception {
        initCheck() ;
        String sql = null ;
        if( where == null || ( where = where.trim() ).length() <= 0 ) {
            sql = new StringBuilder().append( "select count(*) from " ).
                append( meta.getTable() ).toString() ;
            params = null ;
        }
        else {
            StringBuilder buf = new StringBuilder() ;
//            if( where.toLowerCase().startsWith( "where " ) == false ) {
//                buf.append( "select count(*) from " ).append( meta.getTable() ).
//                    append( " where " ).append( where ) ;
//            }
//            else {
                buf.append( "select count(*) from " ).append( meta.getTable() ).
                    append( " " ).append( where ) ;
//            }
            sql = buf.toString() ;
            buf = null ;
        }
        return ExecutionDao.execution( null,null,getRecord(),entryType,model,meta,sql,-1,-1,params ) ;
    }
    
    /**
     * レコード取得.
     * <BR><BR>
     * レコードを取得します.
     * <BR>
     * @return Record 対象のレコード情報が返されます.
     */
    protected Record getRecord() {
        return LocalInfo.getRecord() ;
    }
    
    /**
     * 初期化確認.
     */
    private void initCheck()
        throws Exception {
        if( entryType == null || model == null || meta == null ) {
            throw new IOException( "初期化処理が行われていないか不正です" ) ;
        }
    }
    
    /**
     * 情報存在時のチェック処理.
     */
    private boolean checkEntry( Record record,boolean idFlag,T entry )
        throws Exception {
        if( idFlag == false ) {
            return false ;
        }
        Object value = Reflect.executionMethod( "getId",entry ) ;
        if( value == null ) {
            return false ;
        }
        StringBuilder buf = new StringBuilder() ;
        buf.append( "select count(*) from " ).
            append( DbUtil.convertJavaNameByDBName( model ) ).
            append( " where id=?;" ) ;
        String sql = buf.toString() ;
        buf = null ;
        
        ResultSet rs = null ;
        try {
            boolean ret = false ;
            rs = record.executeQuery( sql,new Object[]{value} ) ;
            if( rs != null ) {
                if( rs.next() == true ) {
                    ret = true ;
                }
                rs.close() ;
                rs = null ;
            }
            return ret ;
        } finally {
            if( rs != null ) {
                try {
                    rs.close() ;
                } catch( Exception e ) {
                }
            }
        }
    }
    
    /**
     * 指定カラムを対象メソッド名に変換.
     */
    private String convertMethodName( String type,String column ) {
        String method = DbUtil.convertDBNameByJavaName( false,column ) ;
        return new StringBuilder().append( type ).
            append( method.substring( 0,1 ).toUpperCase() ).
            append( method.substring( 1,method.length() ) ).toString() ;
    }
    
    /**
     * モデル名を取得.
     */
    private static final String modelName( Class o ) {
        String name = o.getName() ;
        int p = name.lastIndexOf( "." ) ;
        if( p != -1 ) {
            return name.substring( p+1,name.length() ) ;
        }
        return null ;
    }
    
    /**
     * 指定テーブルからMetaColumn情報を取得.
     */
    private static final MetaColumn getMetaColumn( Record record,String model )
        throws Exception {
        return record.getMetaColumn( DbUtil.convertJavaNameByDBName( model ) ) ;
    }
    
    /**
     * アップロード時間をSQL文に追加.
     */
    private static final String addUpdateTime( boolean[] addFlag,String sql ) throws Exception {
        addFlag[0] = false ;
        String lower = sql.toLowerCase() ;
        int s = lower.indexOf( " set " ) ;
        int e = lower.indexOf( " where " ) ;
        if( s == -1 ) {
            return sql ;
        }
        else if( e == -1 ) {
            e = lower.length() ;
        }
        s += " set ".length() ;
        int x = lower.indexOf( " "+ExecutionDao.UPDATE_TIME,s ) ;
        int y = lower.indexOf( ","+ExecutionDao.UPDATE_TIME,s ) ;
        if( x == -1 || x > y ) {
            x = y ;
        }
        lower = null ;
        if( x == -1 || x > e ) {
            sql = new StringBuilder().append( sql.substring( 0,s ) ).
                append( ExecutionDao.UPDATE_TIME ).append( " =?, " ).
                append( sql.substring( s,sql.length() ) ).toString() ;
            addFlag[0] = true ;
        }
        return sql ;
    }
    
    /**
     * アップロード時間をパラメータに追加.
     */
    private static final Object[] addUpdateTimeParam( Object[] params ) throws Exception {
        Object[] ret = null ;
        if( params == null || params.length <= 0 ) {
            ret = new Object[ 1 ] ;
        }
        else {
            int len = params.length ;
            ret = new Object[ len+1 ] ;
            System.arraycopy( params,0,ret,1,len ) ;
        }
        ret[ 0 ] = new java.sql.Timestamp( System.currentTimeMillis() ) ;
        return ret ;
    }
}

