/*
 * @(#)JRcSpringServer.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.spring ;

import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.JRcServer.DeleteSessionTrigger;
import com.JRcServer.JRcBaseDefineBean;
import com.JRcServer.commons.InitPackage;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.sys.GetEnvironment;
import com.JRcServer.commons.sys.Initializer;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.ObjectArray;
import com.JRcServer.server.InitJRcServer;
import com.JRcServer.server.JRcServerShutdownHook;
import com.JRcServer.server.control.JRcRemoteControlDef;
import com.JRcServer.server.control.JRcRemoteControlServer;

/**
 * JRcServerSpringバージョン用サーバ機能.
 *  
 * @version 2006/09/13
 * @author  masahito suzuki
 * @since   JRcServer-Spring 1.00
 */
public class JRcSpringServer {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( JRcSpringServer.class ) ;
    
    /**
     * キャッシュサイズデフォルト値.
     */
    private static final int DEF_CHACHE_LENGTH = InitJRcServer.DEF_CHACHE_LENGTH ;
    
    /**
     * バインドポートデフォルト値.
     */
    private static final int DEF_BIND_PORT = InitJRcServer.DEF_BIND_PORT ;
    
    /**
     * セッションタイムアウトデフォルト値.
     */
    private static final long DEF_SESSION_TIMEOUT = InitJRcServer.DEF_SESSION_TIMEOUT ;
    
    /**
     * 最大コネクション数デフォルト値.
     */
    private static final int DEF_MAX_CONNECTION = InitJRcServer.DEF_MAX_CONNECTION ;
    
    /**
     * 受信タイムアウトデフォルト値.
     */
    private static final int DEF_RECV_TIMEOUT = InitJRcServer.DEF_RECEIVE_TIMEOUT ;
    
    /**
     * JRcSpring環境変数.
     */
    private static final String ENV_NAME = "JRC_SPRING_HOME" ;
    
    /**
     * コンフィグファイル名.
     */
    private static final String CONF_NAME = "jrcsv.conf" ;
    
    /**
     * 初期化オブジェクト.
     */
    private Initializer init = null ;
    
    /**
     * コンストラクタ.
     */
    private JRcSpringServer() {
        
    }
    
    /**
     * メインメソッド.
     * <BR><BR>
     * JRcSpringを開始するメインメソッド.
     * <BR>
     * @param args 対象の引数情報が設定されます.
     */
    public static final void main( String[] args ) {
        
        LOG.info( "## [JRcSpring] -" + JRcSpringService.VERSION + " -開始します" ) ;
        
        new InitPackage().init() ;
        new JRcSpringServer().execution( args ) ;
        new InitPackage().destroy() ;
        
        LOG.info( "## [JRcSpring] -" + JRcSpringService.VERSION + " -終了します" ) ;
        
    }
    
    /**
     * JRcSpringサーバ実行オブジェクト.
     */
    private final void execution( String[] args ) {
        
        int port = -1 ;
        JRcSpringConf conf = null ;
        JRcBaseDefineBean bean = null ;
        JRcRemoteControlServer control = null ;
        
        try {
            
            // 環境変数登録チェック.
            if(
                GetEnvironment.getInstance().get( ENV_NAME ) == null ||
                GetEnvironment.getInstance().get( ENV_NAME ).trim().length() <= 0
            ) {
                throw new AccessException( "環境変数[" + ENV_NAME +
                    "]は設定されていないため起動に失敗しました" ) ;
            }
            
            // 起動コンフィグファイルを取得.
            conf = new JRcSpringConf() ;
            conf.create(
                new StringBuffer().
                append( "$(" ).
                append( ENV_NAME ).
                append( ")" ).
                append( BaseDef.FILE_SEPARATOR ).
                append( "conf" ).
                append( BaseDef.FILE_SEPARATOR ).
                append( CONF_NAME ).
                toString()
            ) ;
            
            // 起動Beanを取得.
            bean = this.getBean( conf ) ;
            
            // シャットダウンコマンド検知用オブジェクトを生成.
            control = new JRcRemoteControlServer() ;
            
            // 終了待ちポート番号を取得.
            port = conf.getExitReceivePort() ;
            if( port > 0 ) {
                control.create( port ) ;
            }
            else {
                control.create() ;
            }
            
            // Springオブジェクトを取得.
            this.createSpring( conf,bean,control ) ;
            
            // 初期化オブジェクト生成.
            init = new InitJRcServer( new InitJRcSpring() ) ;
            
            // 初期化処理.
            init.init( bean ) ;
            
            // シャットダウンフックに登録.
            JRcServerShutdownHook.registShutdownHook( init ) ;
            
            // 処理待機.
            this.waitJRcSpringServer( control ) ;
            
        } catch( Exception e ) {
            LOG.error( "## 処理中にエラーが発生しました",e ) ;
        } finally {
            
            // 終了化.
            init.destroy() ;
            
        }
        
    }
    
    /**
     * 起動Beanを生成.
     */
    private JRcBaseDefineBean getBean( JRcSpringConf conf )
        throws Exception {
        
        int intData = -1 ;
        long longData = -1 ;
        String string = null ;
        Object callback = null ;
        
        JRcBaseDefineBean ret = null ;
        
        // spring定義ファイルが生成されていない場合.
        if( conf.getBeanFileLength() <= 0 ) {
            throw new AccessException( "登録されているSpring定義内容を持つ" +
                "xmlファイルが[" + CONF_NAME +
                "に登録されていません"
            ) ;
        }
        
        // 起動Bean生成.
        ret = new JRcBaseDefineBean() ;
        ret.setBaseDirectory( GetEnvironment.getInstance().get( ENV_NAME ) ) ;
        
        // バインドポート番号を取得.
        intData = conf.getBindPort() ;
        
        if( intData < 0 || intData > 65535 ) {
            intData = DEF_BIND_PORT ;
        }
        
        ret.setBindPort( intData ) ;
        
        // バインドアドレスを取得.
        string = conf.getBindAddress() ;
        if( string != null && string.length() > 0 ){
            try {
                ret.setBindAddress( InetAddress.getByName( string ) ) ;
            } catch( Exception e ) {
                ret.setBindAddress( null ) ;
            }
        }
        else{
            ret.setBindAddress( null ) ;
        }
        
        // キャッシュサイズを取得.
        intData = conf.getCacheSize() ;
        if( intData > 0 ) {
            ret.setCacheSize( intData ) ;
        }
        else {
            ret.setCacheSize( DEF_CHACHE_LENGTH ) ;
        }
        
        // セッションタイムアウトを取得.
        longData = conf.getSessionTimeout() ;
        if( intData > 0 ) {
            ret.setSessionTimeout( longData ) ;
        }
        else {
            ret.setSessionTimeout( DEF_SESSION_TIMEOUT ) ;
        }
        
        // セッションコールバックを取得.
        string = conf.getSessionCallback() ;
        if( string == null || string.length() <= 0 ){
            try {
                callback = UtilCom.createObject( string,null ) ;
                if( ( callback instanceof DeleteSessionTrigger ) == false ) {
                    callback = null ;
                }
            } catch( Throwable e ) {
                callback = null ;
            }
        }
        else{
            callback = null ;
        }
        
        ret.setCallback( ( DeleteSessionTrigger )callback ) ;
        
        // 最大コネクション数を取得.
        intData = conf.getMaxConnection() ;
        if( intData > 0 ) {
            ret.setMaxConnect( intData ) ;
        }
        else {
            ret.setMaxConnect( DEF_MAX_CONNECTION ) ;
        }
        
        // 受信タイムアウトを取得.
        intData = conf.getReceiveTimeout() ;
        if( intData > 0 ) {
            ret.setReceiveTimeout( intData ) ;
        }
        else {
            ret.setReceiveTimeout( DEF_RECV_TIMEOUT ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * Spring起動処理.
     */
    private void createSpring(
        JRcSpringConf conf,
        JRcBaseDefineBean bean,
        JRcRemoteControlServer control )
        throws Exception {
        
        int i ;
        int len ;
        
        String name = null ;
        String[] names = null ;
        ObjectArray ary = null ;
        ApplicationContext ctx = null ;
        JRcSpringOption option = null ;
        
        len = conf.getBeanFileLength() ;
        
        // SpringBeanファイル群を生成.
        if( len > 0 ) {
            
            ary = new ObjectArray() ;
            
            for( i = 0 ; i < len ; i ++ ) {
                
                if(
                    ( name = conf.getBeanFile( i ) ) == null ||
                    ( name = name.trim() ).length() <= 0
                ) {
                    continue ;
                }
                
                ary.add( name ) ;
                
            }
            
            if( ( len = ary.size() ) > 0 ) {
                names = new String[ len ] ;
                System.arraycopy( ary.getObjects(),0,names,0,len ) ;
            }
            
        }
        
        // SpringBeanファイル群が0件の場合.
        if( names == null ) {
            throw new AccessException( "有効なSpringBeanファイルは存在しません" ) ;
        }
        
        // SpringBeanファイルが複数件(1件以上)存在する.
        ctx = new ClassPathXmlApplicationContext( names ) ;
        
        // ApplicationContextをオプションに登録.
        option = new JRcSpringOption() ;
        option.setAppeicationContext( ctx ) ;
        option.setControl( control ) ;
        bean.setOption( option ) ;
        
    }
    
    /**
     * 処理待機.
     */
    private void waitJRcSpringServer( JRcRemoteControlServer control )
        throws Exception {
        
        LOG.info( "## [JRcSpring] - 処理終了待ちコントローラ - start." ) ;
        
        try {
            
            for( ;; ) {
                
                UtilCom.idleTime() ;
                
                if( control != null && control.isThread() == true ) {
                    
                    if( control.getStatus() == JRcRemoteControlDef.REMOTE_SHUTDOWN ) {
                        
                        LOG.info( "## [JRcSpring] - シャットダウン命令が呼び出されました" ) ;
                        break ;
                        
                    }
                    
                }
                else {
                    control = null ;
                    break ;
                }
                
            }
            
        } catch( Exception e ) {
            LOG.error( "## [JRcSpring] - 処理待機中にエラーが発生",e ) ;
        }
        
        LOG.info( "## [JRcSpring] - 処理終了待ちコントローラ - end." ) ;
        
    }
    
}

