/*
 * @(#)JRcSpringListener.java
 *
 * Copyright (c) 2006 KBMJ, Inc. All Rights Reserved
 */
package com.JRcServer.spring ;

import java.net.InetAddress;
import java.util.ArrayList;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.JRcServer.DeleteSessionTrigger;
import com.JRcServer.JRcBaseDefineBean;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.server.InitJRcServer;

/**
 * JRcServerSpringバージョン用リスナ.
 * <BR><BR>
 * JRcServerSpringバージョンを有効にするリスナー.
 * 
 * @version 2006/09/08
 * @author  Masahito Suzuki
 * @since   JRcServer-Spring 1.00
 */
public class JRcSpringListener implements ServletContextListener {

    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( JRcSpringListener.class ) ;
    
    /**
     * キャッシュサイズデフォルト値.
     */
    private static final int DEF_CHACHE_LENGTH = InitJRcServer.DEF_CHACHE_LENGTH ;
    
    /**
     * バインドポートデフォルト値.
     */
    private static final int DEF_BIND_PORT = InitJRcServer.DEF_BIND_PORT ;
    
    /**
     * セッションタイムアウトデフォルト値.
     */
    private static final long DEF_SESSION_TIMEOUT = InitJRcServer.DEF_SESSION_TIMEOUT ;
    
    /**
     * 最大コネクション数デフォルト値.
     */
    private static final int DEF_MAX_CONNECTION = InitJRcServer.DEF_MAX_CONNECTION ;
    
    /**
     * 受信タイムアウトデフォルト値.
     */
    private static final int DEF_RECV_TIMEOUT = InitJRcServer.DEF_RECEIVE_TIMEOUT ;
    
    /**
     * リスナーオプション : 基本ディレクトリ名.
     */
    public static final String BASE_DIRECTORY = "jrcspring.base-directory" ;
    
    /**
     * リスナーオプション : バインドポート名.
     */
    public static final String BIND_PORT = "jrcspring.bind-port" ;
    
    /**
     * リスナーオプション : バインドアドレス名.
     */
    public static final String BIND_ADDRESS = "jrcspring.bind-address" ;
    
    /**
     * リスナーオプション : キャッシュサイズ.
     */
    public static final String CACHE_SIZE = "jrcspring.cache-size" ;
    
    /**
     * リスナーオプション : セッションタイムアウト.
     */
    public static final String SESSION_TIMEOUT = "jrcspring.session-timeout" ;
    
    /**
     * リスナーオプション : セッションタイムアウトコールバックオブジェクト.
     */
    public static final String SESSION_CALLBACK = "jrcspring.session-callback" ;
    
    /**
     * リスナーオプション : コネクション最大数.
     */
    public static final String MAX_CONNECTION = "jrcspring.max-connection" ;
    
    /**
     * リスナーオプション : 受信タイムアウト.
     */
    public static final String RECEIVE_TIMEOUT = "jrcspring.receive-timeout" ;
    
    
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実行します.
     * <BR>
     * @param sce サーブレットコンテキストイベント.
     */
    public void contextInitialized( ServletContextEvent sce ) {
        
        long time ;
        
        ServletContext servletContext = null ;
        ApplicationContext ctx = null ;
        JRcSpringOption option = null ;
        JRcBaseDefineBean bean = null ;
        
        String string = null ;
        String basePath = null ;
        int bindPort = -1 ;
        InetAddress bindAddr = null ;
        int cacheSize = -1 ;
        long sessionTimeout = -1 ;
        Object callback = null ;
        int maxConnect = -1 ;
        int receiveTimeout = -1 ;
        
        try {
            
            LOG.info( "## JRcServer-Spring verion Loading ..." ) ;
            time = System.currentTimeMillis() ;
            
            servletContext = sce.getServletContext() ;
            ctx = WebApplicationContextUtils.getWebApplicationContext( servletContext ) ;
            if( ctx == null ) {
                throw new AccessException(
                    ApplicationContext.class.getName() +
                    " の取得に失敗しました" ) ;
            }
            
            bean = new JRcBaseDefineBean() ;
            
            // 基本ディレクトリを取得.
            string = ( String )servletContext.getInitParameter( BASE_DIRECTORY ) ;
            if( string != null && string.length() > 0 ){
                basePath = string ;
            }
            else{
                basePath = JRcSpringListener.getBeforeRealPath( servletContext ) ;
            }
            
            if( IOCom.isDirExists( basePath ) == false ) {
                throw new InputException(
                    "対象の基本ディレクトリ[" + basePath +
                    "]は存在しません" ) ;
            }
            
            // バインドポート番号を取得.
            string = ( String )servletContext.getInitParameter( BIND_PORT ) ;
            if( string != null && string.length() > 0 ){
                try {
                    bindPort = Integer.parseInt( string ) ;
                } catch( Exception e ) {
                    bindPort = DEF_BIND_PORT ;
                }
            }
            else{
                bindPort = DEF_BIND_PORT ;
            }
            
            // バインドアドレスを取得.
            string = ( String )servletContext.getInitParameter( BIND_ADDRESS ) ;
            if( string != null && string.length() > 0 ){
                try {
                    bindAddr = InetAddress.getByName( string ) ;
                } catch( Exception e ) {
                    bindAddr = null ;
                }
            }
            else{
                bindAddr = null ;
            }
            
            // キャッシュサイズを取得.
            string = ( String )servletContext.getInitParameter( CACHE_SIZE ) ;
            if( string != null && string.length() > 0 ){
                
                if( "false".equals( string.trim().toLowerCase() ) == true ) {
                    cacheSize = -1 ;
                }
                else {
                    try {
                        cacheSize = Integer.parseInt( string ) ;
                    } catch( Exception e ) {
                        cacheSize = DEF_CHACHE_LENGTH ;
                    }
                }
            }
            else{
                cacheSize = DEF_CHACHE_LENGTH ;
            }
            
            // セッションタイムアウトを取得.
            string = ( String )servletContext.getInitParameter( SESSION_TIMEOUT ) ;
            if( string != null && string.length() > 0 ){
                try {
                    sessionTimeout = Long.parseLong( string ) ;
                } catch( Exception e ) {
                    sessionTimeout = DEF_SESSION_TIMEOUT ;
                }
            }
            else{
                sessionTimeout = DEF_SESSION_TIMEOUT ;
            }
            
            // セッションコールバックを取得.
            string = ( String )servletContext.getInitParameter( SESSION_CALLBACK ) ;
            if( string != null && string.length() > 0 ){
                try {
                    callback = UtilCom.createObject( string,null ) ;
                    if( ( callback instanceof DeleteSessionTrigger ) == false ) {
                        callback = null ;
                    }
                } catch( Throwable e ) {
                    callback = null ;
                }
            }
            else{
                callback = null ;
            }
            
            // 最大コネクション数を取得.
            string = ( String )servletContext.getInitParameter( MAX_CONNECTION ) ;
            if( string != null && string.length() > 0 ){
                try {
                    maxConnect = Integer.parseInt( string ) ;
                    if( maxConnect <= 0 ) {
                        maxConnect = DEF_MAX_CONNECTION ;
                    }
                } catch( Exception e ) {
                    maxConnect = DEF_MAX_CONNECTION ;
                }
            }
            else{
                maxConnect = DEF_MAX_CONNECTION ;
            }
            
            // 受信タイムアウトを取得.
            string = ( String )servletContext.getInitParameter( RECEIVE_TIMEOUT ) ;
            if( string != null && string.length() > 0 ){
                try {
                    receiveTimeout = Integer.parseInt( string ) ;
                    if( receiveTimeout <= 0 ) {
                        receiveTimeout = DEF_RECV_TIMEOUT ;
                    }
                } catch( Exception e ) {
                    receiveTimeout = DEF_RECV_TIMEOUT ;
                }
            }
            else{
                receiveTimeout = DEF_RECV_TIMEOUT ;
            }
            
            // オプション情報を設定.
            option = new JRcSpringOption() ;
            option.setAppeicationContext( ctx ) ;
            
            // Beanに取得内容をセット.
            bean.setBaseDirectory( basePath ) ;
            bean.setCacheSize( cacheSize ) ;
            bean.setSessionTimeout( sessionTimeout ) ;
            bean.setBindAddress( bindAddr ) ;
            bean.setBindPort( bindPort ) ;
            bean.setCallback( ( DeleteSessionTrigger )callback ) ;
            bean.setMaxConnect( maxConnect ) ;
            bean.setReceiveTimeout( receiveTimeout ) ;
            bean.setOption( option ) ;
            
            LOG.info( "## startup - args[" + bean.toString() + "]" ) ;
            
            // JRcSpring初期化処理.
            new InitJRcServer( new InitJRcSpring() ).init() ;
            
        } catch ( Exception e ) {
            try {
                new InitJRcServer( new InitJRcSpring() ).destroy() ;
            } catch( Exception ee ) {
            }
            throw new RuntimeException( e );
        }
        
        time = System.currentTimeMillis() - time ;
        
        LOG.info( "## JRcServer-Spring Loaded time(" +
            time + "msec)" ) ;
        
    }
    
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実行します.
     * <BR>
     * @param sce サーブレットコンテキストイベント.
     */
    public void contextDestroyed( ServletContextEvent sce ) {
        
        LOG.info( "## JRcServer-Spring Shutdown ..." ) ;
        
        try {
            new InitJRcServer( new InitJRcSpring() ).destroy() ;
        } catch( Exception e ) {
        }
        
        LOG.info( "## JRcServer-Spring Shutdown ... OK" ) ;
        
    }
    
    /**
     * リアルパスの前パス情報を取得.
     */
    private static final String getBeforeRealPath( ServletContext servletContext )
        throws Exception {
        
        int i ;
        int len ;
        boolean flg = false ;
        
        String path = null ;
        ArrayList lst = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        path = servletContext.getRealPath( "/" ) ;
        
        if( path != null && path.length() > 0 ) {
            
            if( path.indexOf( "\\" ) != -1 ) {
                lst = UtilCom.pauseString( path,"\\" ) ;
            }
            else {
                lst = UtilCom.pauseString( path,"/" ) ;
            }
            
            if( lst != null && ( len = lst.size() ) > 0 ) {
                
                buf = new StringBuffer() ;
                len = len - 1 ;
                
                if( len > 0 ) {
                    for( i = 0 ; i < len ; i ++ ) {
                        buf.append( BaseDef.FILE_SEPARATOR ) ;
                        buf.append( lst.get( i ) ) ;
                    }
                }
                else {
                    buf.append( BaseDef.FILE_SEPARATOR ) ;
                }
                
                ret = buf.toString() ;
                flg = true ;
                
            }
            
        }
        
        if( flg == false ) {
            throw new AccessException(
                "ServletContext.getRealPath()の情報は存在しないか不正です"
            ) ;
        }
        
        return ret ;
        
    }

}
