/*
 * @(#)InitJRcSpring.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.spring ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;

import com.JRcServer.JRCommand;
import com.JRcServer.JRcBaseDefineBean;
import com.JRcServer.commons.beans.BaseBean;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.sys.Initializer;
import com.JRcServer.commons.util.array.ObjectArray;
import com.JRcServer.server.JRcBaseManager;
import com.JRcServer.server.JRcManagerFactory;

/**
 * JRcServerSpringバージョン用初期化処理.
 *  
 * @version 2006/09/08
 * @author  masahito suzuki
 * @since   JRcServer-Spring 1.00
 */
public class InitJRcSpring implements Initializer {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( InitJRcSpring.class ) ;
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.<BR>
     * また、このメソッドの場合、デフォルト値での初期化処理になります.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public void init() throws AccessException {
        throw new AccessException(
            InitJRcSpring.class.getName() +
            " ではこのメソッド[init()]は対応していません" ) ;
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.
     * <BR>
     * @param bean 設定値が設定されているBeanオブジェクトを設定します.
     * @exception AccessException アクセス例外.
     */
    public void init( BaseBean bean ) throws AccessException {
        
        Object option = null ;
        
        if(
            ( option = ( ( JRcBaseDefineBean )bean ).getOption() ) == null ||
            ( option instanceof JRcSpringOption ) == false ||
            ( ( JRcSpringOption )option ).getAppeicationContext() == null
        ) {
            throw new AccessException(
                "指定された起動Beanのオプション条件は不正です"
            ) ;
        }
        
        try {
            
            LOG.info( "## [JRcSpring] init - start" ) ;
            
            // 有効となるBeanオブジェクト名を取得.
            ( ( JRcSpringOption )option ).setServiceNames(
                InitJRcSpring.getServiceNames(
                    ( ( JRcSpringOption )option ).getAppeicationContext()
                )
            ) ;
            
            // Springサービスを生成して登録.
            ( ( JRcBaseDefineBean )bean ).setService( new JRcSpringService() ) ;
            
            LOG.info( "## [JRcSpring] init - end:(success)" ) ;
            
        } catch( Exception e ) {
            
            LOG.error( "## [JRcSpring] init - end:(error)" ) ;
            throw new AccessException( e ) ;
            
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実施します.<BR>
     * また、この処理を実施した場合、再び Initializer.init() を
     * 呼び出さないとオブジェクトの利用が出来なくなります.
     */
    public void destroy() {
        
        JRcSpringOption opt = null ;
        JRcBaseManager base = null ;
        
        LOG.info( "## [JRcSpring] destroy - start." ) ;
        
        try {
            
            base = JRcManagerFactory.getJRcBaseManager() ;
            opt = ( JRcSpringOption )base.getOption() ;
            opt.getControl().clear() ;
            
        } catch( Exception e ) {
            
        }
        
        LOG.info( "## [JRcSpring] destroy - end." ) ;
        
    }
    
    /**
     * JRcServerに対応したSpringBean名一覧を取得.
     */
    private static final String[] getServiceNames( ApplicationContext applicationContext ) {
        
        int i ;
        int len ;
        
        String[] names = null ;
        ObjectArray ary = null ;
        String[] ret = null ;
        
        try {
            
            names = applicationContext.getBeanDefinitionNames() ;
            if( names != null && ( len = names.length ) > 0 ) {
                
                ary = new ObjectArray() ;
                for( i = 0 ; i < len ; i ++ ) {
                    if( applicationContext.getBean( names[ i ] ) instanceof JRCommand ) {
                        ary.add( names[ i ] ) ;
                    }
                }
                
                names = null ;
                
                len = ary.size() ;
                if( len > 0 ) {
                    ret = new String[ len ] ;
                    System.arraycopy( ary.getObjects(),0,ret,0,len ) ;
                }
                
                ary = null ;
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
}

