/*
 * @(#)JRcParameter.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer ;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Date;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.ConvertException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.ConvertResourceParam;
import com.JRcServer.commons.serialize.SerializeUtil;
import com.JRcServer.commons.util.ConvertParam;

/**
 * JRcServerパラメータ情報.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerBase 1.00
 */
public class JRcParameter implements Serializable {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            JRcParameter.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 文字列変換キャラクタセット.
     */
    private static final String CHARSET = BaseDef.UTF8 ;
    
    /**
     * パラメータタイプ.
     */
    private int type = -1 ;
    
    /**
     * パラメータ内容.
     */
    private byte[] value = null ;
    
    /**
     * コンストラクタ.
     */
    public JRcParameter() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を指定して、パラメータを生成します.
     * <BR>
     * @param type 対象のパラメータタイプを設定します.<BR>
     *             [JRcParameterType#TYPE_BINARY]を設定した場合(byte[])情報です.<BR>
     *             [JRcParameterType#TYPE_STRING]を設定した場合(String)情報です.<BR>
     *             [JRcParameterType#TYPE_BOOLEAN]を設定した場合(Boolean)情報です.<BR>
     *             [JRcParameterType#TYPE_SHORT]を設定した場合(Short)情報です.<BR>
     *             [JRcParameterType#TYPE_INTEGER]を設定した場合(Integer)情報です.<BR>
     *             [JRcParameterType#TYPE_LONG]を設定した場合(Long)情報です.<BR>
     *             [JRcParameterType#TYPE_FLOAT]を設定した場合(Float)情報です.<BR>
     *             [JRcParameterType#TYPE_DOUBLE]を設定した場合(Double)情報です.<BR>
     *             [JRcParameterType#TYPE_DECIMAL]を設定した場合(BigDecimal)情報です.<BR>
     *             [JRcParameterType#TYPE_TIMESTAMP]を設定した場合(Date)情報です.
     * @param value 対象のパラメータ内容を設定します.
     * @exception InputException 入力例外.
     */
    public JRcParameter( int type,byte[] value )
        throws InputException {
        
        if(
            (
                type != JRcParameterType.TYPE_BINARY &&
                type != JRcParameterType.TYPE_STRING &&
                type != JRcParameterType.TYPE_BOOLEAN &&
                type != JRcParameterType.TYPE_SHORT &&
                type != JRcParameterType.TYPE_INTEGER &&
                type != JRcParameterType.TYPE_LONG &&
                type != JRcParameterType.TYPE_FLOAT &&
                type != JRcParameterType.TYPE_DOUBLE &&
                type != JRcParameterType.TYPE_DECIMAL &&
                type != JRcParameterType.TYPE_TIMESTAMP
            ) ||
            ( value == null || value.length <= 0 )
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.type = type ;
        this.value = value ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        type = -1 ;
        value = null ;
    }
    
    /**
     * パラメータをバイナリで設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setBinary( byte[] o )
        throws InputException {
        
        if( o == null || o.length <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.type = JRcParameterType.TYPE_BINARY ;
        this.value = o ;
        
    }
    
    /**
     * パラメータをバイナリで取得.
     * <BR><BR>
     * @return byte[] パラメータ情報が返されます.
     */
    public byte[] getBinary() {
        
        return this.value ;
        
    }
    
    /**
     * パラメータを文字列で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setString( String o )
        throws InputException {
        
        if( o == null || o.length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = o.getBytes( CHARSET ) ;
            this.type = JRcParameterType.TYPE_STRING ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * @return String パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public String getString()
        throws ConvertException {
        
        String ret = null ;
        
        try {
            ret = new String( this.value,CHARSET ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータをフラグで設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setBoolean( Boolean o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertBoolean( o.booleanValue() ) ;
            this.type = JRcParameterType.TYPE_BOOLEAN ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータをフラグで取得.
     * <BR><BR>
     * @return Boolean パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Boolean getBoolean()
        throws ConvertException {
        
        Boolean ret = null ;
        
        try {
            ret = new Boolean(
                ConvertParam.convertBoolean( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータを数値(Short)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setShort( Short o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertShort( o.shortValue() ) ;
            this.type = JRcParameterType.TYPE_SHORT ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを数値(Short)で取得.
     * <BR><BR>
     * @return Short パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Short getShort()
        throws ConvertException {
        
        Short ret = null ;
        
        try {
            ret = new Short(
                ConvertParam.convertShort( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータを数値(Integer)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setInteger( Integer o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertInt( o.intValue() ) ;
            this.type = JRcParameterType.TYPE_INTEGER ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを数値(Integer)で取得.
     * <BR><BR>
     * @return Integer パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Integer getInteger()
        throws ConvertException {
        
        Integer ret = null ;
        
        try {
            ret = new Integer(
                ConvertParam.convertInt( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータを数値(Long)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setLong( Long o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertLong( o.longValue() ) ;
            this.type = JRcParameterType.TYPE_LONG ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを数値(Long)で取得.
     * <BR><BR>
     * @return Long パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Long getLong()
        throws ConvertException {
        
        Long ret = null ;
        
        try {
            ret = new Long(
                ConvertParam.convertLong( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータを浮動少数値(Float)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setFloat( Float o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertFloat( o.floatValue() ) ;
            this.type = JRcParameterType.TYPE_FLOAT ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを浮動少数値(Float)で取得.
     * <BR><BR>
     * @return Float パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Float getFloat()
        throws ConvertException {
        
        Float ret = null ;
        
        try {
            ret = new Float(
                ConvertParam.convertFloat( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータを浮動少数値(Double)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setDouble( Double o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertDouble( o.doubleValue() ) ;
            this.type = JRcParameterType.TYPE_DOUBLE ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを浮動少数値(Double)で取得.
     * <BR><BR>
     * @return Double パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Double getDouble()
        throws ConvertException {
        
        Double ret = null ;
        
        try {
            ret = new Double(
                ConvertParam.convertDouble( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータを浮動少数値(Decimal)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setDecimal( BigDecimal o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertDecimal( o ) ;
            this.type = JRcParameterType.TYPE_DECIMAL ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータを浮動少数値(Decimal)で取得.
     * <BR><BR>
     * @return BigDecimal パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public BigDecimal getDecimal()
        throws ConvertException {
        
        BigDecimal ret = null ;
        
        try {
            ret = ConvertParam.convertDecimal( 0,this.value ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で設定.
     * <BR><BR>
     * @param o 設定対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setTimestamp( Date o )
        throws InputException {
        
        if( o == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            this.value = ConvertParam.convertLong( o.getTime() ) ;
            this.type = JRcParameterType.TYPE_TIMESTAMP ;
        } catch ( Exception e ) {
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で取得.
     * <BR><BR>
     * @return Date パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Date getTimestamp()
        throws ConvertException {
        
        Date ret = null ;
        
        try {
            ret = new Date(
                ConvertParam.convertLong( 0,this.value ) ) ;
        } catch( Exception e ) {
            throw new ConvertException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * パラメータタイプを取得.
     * <BR><BR>
     * @return int パラメータタイプが返されます.<BR>
     *             [JRcParameterType#TYPE_BINARY]が返された場合(byte[])情報です.<BR>
     *             [JRcParameterType#TYPE_STRING]が返された場合(String)情報です.<BR>
     *             [JRcParameterType#TYPE_BOOLEAN]が返された場合(Boolean)情報です.<BR>
     *             [JRcParameterType#TYPE_SHORT]が返された場合(Short)情報です.<BR>
     *             [JRcParameterType#TYPE_INTEGER]が返された場合(Integer)情報です.<BR>
     *             [JRcParameterType#TYPE_LONG]が返された場合(Long)情報です.<BR>
     *             [JRcParameterType#TYPE_FLOAT]が返された場合(Float)情報です.<BR>
     *             [JRcParameterType#TYPE_DOUBLE]が返された場合(Double)情報です.<BR>
     *             [JRcParameterType#TYPE_DECIMAL]が返された場合(BigDecimal)情報です.<BR>
     *             [JRcParameterType#TYPE_TIMESTAMP]が返された場合(Date)情報です.
     */
    public int getType() {
        return this.type ;
    }
    
    /**
     * パラメータ内容をOutputStreamに設定.
     * <BR><BR>
     * パラメータ内容をOutputStreamに設定します.
     * <BR>
     * @param outputStream 対象のOutputStreamを設定します.
     * @exception IOException I/O例外.
     */
    public void exportOutputStream( OutputStream outputStream )
        throws IOException {
        
        int len ;
        
        if( this.value == null || ( len = this.value.length ) <= 0 ) {
            return ;
        }
        
        outputStream.write( this.type & 0x000000ff ) ;
        outputStream.write( ConvertParam.convertInt( len ) ) ;
        outputStream.write( this.value ) ;
        
    }
    
    /**
     * パラメータ内容をバイナリリソースに設定.
     * <BR><BR>
     * パラメータ内容をバイナリリソースに設定します.
     * <BR>
     * @param pnt 設定開始位置を設定します.
     * @param binary 対象のバイナリリソースを設定します.
     * @return int 次の設定開始位置を示すポイントが返されます.<BR>
     *             [-1]が返された場合、処理に失敗しました.
     */
    public int exportResource( int pnt,BinResource binary ) {
        
        int len ;
        int ret = -1 ;
        
        if( pnt < 0 || binary == null ) {
            return -1 ;
        }
        
        if( this.value == null || ( len = this.value.length ) <= 0 ) {
            return pnt ;
        }
        
        binary.set( pnt,( this.type & 0x000000ff ) ) ;
        pnt += 1 ;
        ConvertResourceParam.convertInt( binary,pnt,len ) ;
        pnt += 4 ;
        binary.setBinary( pnt,this.value ) ;
        ret = pnt + len ;
        
        return ret ;
        
    }
    
    /**
     * バイナリリソース内容をパラメータに設定.
     * <BR><BR>
     * バイナリリソース内容をパラメータに設定します.
     * <BR>
     * @param pnt 取得開始位置を設定します.
     * @param binary 取得対象のバイナリリソースを設定します.
     * @return int 次の取得開始位置を示すポイントが返されます.<BR>
     *             [-1]が返された場合、処理に失敗しました.
     */
    public int importResource( int pnt,BinResource binary ) {
        
        int len ;
        int paramType ;
        int ret = -1 ;
        
        byte[] paramValue = null ;
        
        if( binary == null || pnt < 0 || pnt >= binary.size() ) {
            return -1 ;
        }
        
        try {
            
            // パラメータタイプを取得.
            paramType = ( int )( binary.get( pnt ) & 0x000000ff ) ;
            pnt += 1 ;
            
            // パラメータデータ長を取得.
            len = ConvertResourceParam.convertInt( pnt,binary ) ;
            pnt += 4 ;
            
            // パラメータデータ長が不正な場合.
            if( len <= 0 || len + pnt > binary.size() ) {
                ret = -1 ;
            }
            else {
                
                // パラメータデータを取得.
                paramValue = binary.getBinary( pnt,len ) ;
                
                // 正常な場合はこのオブジェクトに登録.
                this.type = paramType ;
                this.value = paramValue ;
                
                ret = pnt + len ;
                
            }
            
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
}

