/*
 * @(#)JRcConnectCommon.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer ;

import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.Socket;
import java.net.SocketException;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.BufferedBinResource;
import com.JRcServer.commons.resource.Resource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ConvertParam;

/**
 * JRcServer共通処理.
 *  
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcServerBase 1.00
 */
public class JRcConnectCommon {
    
    private JRcConnectCommon() {
    }
    
    /**
     * バッファサイズ.
     */
    public static final int BUFFER_LENGTH = 512 ;
    
    /**
     * ソケット受信タイムアウト.
     */
    public static final int RECV_TIMEOUT = 1000 ;
    
    /**
     * 基本バッファサイズ.
     */
    public static final int BASE_BUFFER = 131072 ;
    
    /**
     * ヘッダデータポイント.
     */
    private static final int HEADER_POINT = JRcProtocolDef.HEADER_LENGTH + 4 ;
    
    /**
     * ソケットオプションをセット.
     * <BR><BR>
     * ソケットオプションを設定します.
     * <BR>
     * @param socket 対象のソケット情報を設定します.
     * @exception SocketException ソケット例外.
     */
    public static final void setSocketOption( Socket socket )
        throws SocketException {
        
        socket.setSendBufferSize( BASE_BUFFER ) ;
        socket.setReceiveBufferSize( BASE_BUFFER ) ;
        socket.setSoTimeout( RECV_TIMEOUT ) ;
        socket.setSoLinger( true,1 ) ;
        socket.setKeepAlive( true ) ;
        socket.setTcpNoDelay( true ) ;
        socket.setReuseAddress( true ) ;
        
    }
    
    /**
     * データ受信処理.
     * <BR><BR>
     * データ受信処理を行います.
     * <BR>
     * @param receiveTime 対象の受信ヘッダタイプを設定します.<BR>
     *                    [true]が設定された場合は、送信ヘッダ条件の内容として取得します.<BR>
     *                    [false]が設定された場合は、受信ヘッダ条件の内容として取得します.
     * @param resType 対象のリソースタイプを設定します.
     * @param sync 同期オブジェクトを設定します.
     * @param stream 対象の入力ストリームを設定します.
     * @param timeout 受信タイムアウトを設定します.
     * @return BinResource 受信情報が返されます.<BR>
     *                     [null]が返された場合、接続先がクローズされた可能性があります.
     * @exception JRCommandException JRCommand例外.
     * @exception ConnectTimeoutException コネクションタイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public static final BinResource receive(
        boolean rcvType,ResourceType resType,Synchronized sync,InputStream stream,int timeout )
        throws JRCommandException,ConnectTimeoutException,ExecutionException {
        
        int data ;
        int cnt ;
        int headerLen ;
        long time ;
        boolean dataFlg = false ;
        boolean flg = false ;
        
        byte[] header = null ;
        BufferedBinResource buf = null ;
        BinResource ret = null ;
        
        try {
            
            // タイムアウトをセット.
            if( timeout != -1 ) {
                time = System.currentTimeMillis() + ( long )timeout ;
            }
            else {
                time = Long.MAX_VALUE ;
            }
            
            // 電文を受信する内容をセット.
            ret = Resource.createBinResource( resType,1 ) ;
            buf = new BufferedBinResource( ret ) ;
            
            // ヘッダ情報を生成.
            header = new byte[ HEADER_POINT ] ;
            
            // 受信情報待ち.
            for( cnt = 0,flg = false,headerLen = -1 ;; ) {
                
                // データ受信が終了した場合.
                if( flg == true ) {
                    break ;
                }
                
                try {
                    
                    // データ受信処理.
                    for( dataFlg = false ;; ) {
                        
                        // スレッド停止チェック.
                        if( sync != null && sync.isUse() == false ) {
                            throw new ExecutionException(
                                "スレッドは停止しました",
                                ExecutionException.LEVEL_STOP
                            ) ;
                        }
                        
                        // １バイト読み込み.
                        if( ( data = stream.read() ) < 0 ) {
                            
                            if( cnt <= 0 ) {
                                return null ;
                            }
                            
                            buf.flush() ;
                            flg = true ;
                            
                            break ;
                            
                        }
                        
                        // 電文ヘッダ内容受け取り条件の場合.
                        if( cnt <= HEADER_POINT ) {
                            
                            // ヘッダ長と一致する場合.
                            if( cnt == HEADER_POINT ) {
                                
                                // チェック指定が送信ヘッダである場合.
                                if( rcvType == true ) {
                                    
                                    // 送信ヘッダが不正.
                                    if( 
                                        header[ 0 ] != JRcProtocolDef.SEND_HEADER[ 0 ] ||
                                        header[ 1 ] != JRcProtocolDef.SEND_HEADER[ 1 ] ||
                                        header[ 2 ] != JRcProtocolDef.SEND_HEADER[ 2 ] ||
                                        header[ 3 ] != JRcProtocolDef.SEND_HEADER[ 3 ] ||
                                        header[ 4 ] != JRcProtocolDef.SEND_HEADER[ 4 ] ||
                                        header[ 5 ] != JRcProtocolDef.SEND_HEADER[ 5 ]
                                    ) {
                                        throw new JRCommandException(
                                            JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                                            "不正な電文を受信しました"
                                        ) ;
                                    }
                                    
                                }
                                else {
                                        
                                    // 受信ヘッダが不正.
                                    if( 
                                        header[ 0 ] != JRcProtocolDef.RCV_HEADER[ 0 ] ||
                                        header[ 1 ] != JRcProtocolDef.RCV_HEADER[ 1 ] ||
                                        header[ 2 ] != JRcProtocolDef.RCV_HEADER[ 2 ] ||
                                        header[ 3 ] != JRcProtocolDef.RCV_HEADER[ 3 ] ||
                                        header[ 4 ] != JRcProtocolDef.RCV_HEADER[ 4 ] ||
                                        header[ 5 ] != JRcProtocolDef.RCV_HEADER[ 5 ]
                                    ) {
                                        throw new JRCommandException(
                                            JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                                            "不正な電文を受信しました"
                                        ) ;
                                    }
                                    
                                }
                                
                                // ヘッダ長を取得.
                                headerLen = ConvertParam.convertInt(
                                    JRcProtocolDef.HEADER_LENGTH,header ) ;
                                
                            }
                            else{
                                
                                // ヘッダ情報をセット.
                                header[ cnt ] = ( byte )( data & 0x000000ff ) ;
                                
                            }
                            
                        }
                        
                        //System.out.println( "data[" + cnt + "]:0x" + Integer.toHexString( data ) ) ;
                        
                        // データセット.
                        buf.set( cnt,data ) ;
                        cnt ++ ;
                        dataFlg = true ;
                        
                        // 取得情報長が存在する場合.
                        if( headerLen != -1 && headerLen <= cnt ) {
                            
                            buf.flush() ;
                            flg = true ;
                            break ;
                            
                        }
                        
                    }
                    
                } catch( JRCommandException je ) {
                    throw je ;
                } catch( InterruptedIOException ii ) {
                    
                    // １度以上データを受信した場合の後
                    // タイムアウト例外が発生した場合.
                    if( dataFlg == true ) {
                        
                        // タイムアウト値を再設定.
                        if( timeout != -1 ) {
                            time = System.currentTimeMillis() + ( long )timeout ;
                        }
                        
                    }
                    // １度もデータを受信できていない場合.
                    else {
                        headerLen = -1 ;
                    }
                    
                } catch( ExecutionException ee ) {
                    throw ee ;
                } catch( Exception e ) {
                    // その他例外の場合は、スレッド破棄.
                    throw new ExecutionException(
                        e,ExecutionException.LEVEL_STOP
                    ) ;
                }
                
                // タイムアウトを検知した場合.
                if( time <= System.currentTimeMillis() ) {
                    throw new ConnectTimeoutException(
                        "受信タイムアウトを検知しました"
                    ) ;
                }
                
            }
            
        } catch( JRCommandException je ) {
            throw je ;
        } catch( InputException in ) {
            // その他例外の場合は、スレッド破棄.
            throw new ExecutionException(
                in,ExecutionException.LEVEL_STOP
            ) ;
        } finally {
            
            if( buf != null ) {
                buf.clear() ;
            }
            
            buf = null ;
            
        }
        
        return ret ;
        
    }
    
}

