/*
 * @(#)JRcRemoteControlServer
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server.control ;

import java.net.InetAddress;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.net.UdpProtocol;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.UtilCom;

/**
 * JRcServerリモートコントロールサーバ.
 *  
 * @version 2006/09/13
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
public class JRcRemoteControlServer extends Thread {
    
    /**
     * 受信タイムアウト値.
     */
    private static final int TIMEOUT = 1 ;
    
    /**
     * 受信先情報.
     */
    private final ConnectAddress receiveAddress = new ConnectAddress() ;
    
    /**
     * UDPプロトコル.
     */
    private UdpProtocol udp = null ;
    
    
    /**
     * オープンアドレス.
     */
    private String bindAddress = "127.0.0.1" ;
    
    /**
     * オープンポート番号.
     */
    private int bindPort = -1 ;
    
    /**
     * リモートステータス情報.
     */
    private volatile int state = JRcRemoteControlDef.DEF_REMOTE ;
    
    /**
     * スレッドストップ処理.
     */
    private volatile boolean exitThreadFlag = true ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public JRcRemoteControlServer() {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報監視モードセット.
     * <BR><BR>
     * 情報監視モードを設定します.
     * また、このとき実施される監視領域は、[localhost:32108]となります.
     * <BR>
     * @param port 受け取りポート番号を設定します.
     * @exception InputException 入力例外.
     */
    public final void create() {
        try {
            this.create( null,JRcRemoteControlDef.DEF_PORT ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 情報監視モードセット.
     * <BR><BR>
     * 情報監視モードを設定します.
     * また、このとき実施される監視領域は、[localhost]となります.
     * <BR>
     * @param port 受け取りポート番号を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( int port )
        throws InputException {
        this.create( null,port ) ;
    }
    
    /**
     * 情報監視モードセット.
     * <BR><BR>
     * 情報監視モードを設定します.
     * また、このとき実施される監視領域は、[localhost]となります.
     * <BR>
     * @param addr バインドアドレスを設定します.
     * @param port 受け取りポート番号を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( String addr,int port )
        throws InputException {
        
        if( addr == null || port <= 0 || port >= 65535 ){
            if( port <= 0 || port >= 65535 ){
                throw new InputException( "引数が不正です" ) ;
            }
            addr = "127.0.0.1" ;
        }
        
        this.clear() ;
        sync.create() ;
        
        try {
            
            synchronized( sync.get() ) {
                
                exitThreadFlag = false ;
                
                bindAddress = addr ;
                bindPort = port ;
                udp = new UdpProtocol() ;
                
                this.start() ;
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 情報監視モードクリア.
     * <BR><BR>
     * 情報監視モードをクリアします.
     */
    public final void clear() {
        
        try {
            
            synchronized( sync.get() ) {
                
                exitThreadFlag = true ;
                
                try{
                    udp.close() ;
                }catch( Exception e ){
                }
                
                exitThreadFlag = true ;
                receiveAddress.clear() ;
                bindAddress = "127.0.0.1" ;
                bindPort = -1 ;
                udp = null ;
                
            }
            
        } catch( Exception e ) {
        }
        
        exitThreadFlag = true ;
        receiveAddress.clear() ;
        bindAddress = "127.0.0.1" ;
        bindPort = -1 ;
        udp = null ;
        
    }
    
    /**
     * 受信ステータス取得.
     * <BR><BR>
     * 受信ステータス取得します.
     * この処理により、[JRcContorolServer]より
     * 指示された情報が渡されます.
     * <BR>
     * @return int 受信ステータス内容が渡されます.
     */
    public final int getStatus() {
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                
                ret = this.state ;
                
            }
        } catch( Exception e ) {
            ret = JRcRemoteControlDef.REMOTE_SHUTDOWN ;
        }
        
        return ret ;
    }
    
    /**
     * 受信時コネクション情報取得.
     * <BR><BR>
     * 受信字コネクション情報を取得します.
     * <BR>
     * @return ConnectAddress 受信コネクションアドレスが返されます.
     */
    public final ConnectAddress getAddress() {
        
        ConnectAddress ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = new ConnectAddress( receiveAddress ) ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド動作チェック.
     * <BR><BR>
     * スレッドが動作しているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.
     */
    public final boolean isThread() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = true ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実施.
     */
    public final void run() {
        
        int state ;
        
        try{
            
            for( ;; ){
                
                UtilCom.idleTime() ;
                
                synchronized( sync.get() ) {
                    
                    if( exitThreadFlag == true ) {
                        break ;
                    }
                    
                    state = this.getRemoteState() ;
                    this.state = state ;
                    
                    if(
                        state == JRcRemoteControlDef.REMOTE_ERROR ||
                        state == JRcRemoteControlDef.REMOTE_SHUTDOWN ||
                        exitThreadFlag == true
                    )
                    {
                        break ;
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
        }
        
    }
    
    /**
     * リモートステータス情報を取得.
     */
    private final int getRemoteState() {
        int ret ;
        byte[] info = null ;
        
        ret = 0 ;
        
        try{
            
            if( udp.isOpen() == false ){
                
                udp.openTo(
                    UdpProtocol.NOT_MODE,8192,bindPort,
                    InetAddress.getByName( bindAddress ) ) ;
                
            }
            
            info = udp.receive( receiveAddress,JRcRemoteControlServer.TIMEOUT ) ;
            ret = JRcRemoteControlDef.convertBinaryByRemoteMode( info ) ;
            
        }catch( ConnectTimeoutException tm ){
            ret = JRcRemoteControlDef.REMOTE_TIMEOUT ;
        }catch( Exception t ){
            ret = JRcRemoteControlDef.REMOTE_ERROR ;
        }finally{
            info = null ;
        }
        
        return ret ;
    }
    
    
    
}

