/*
 * @(#)JRcRemoteControlClient
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server.control ;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.CommunicateException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.net.UdpProtocol;

/**
 * JRcServerリモートコントロールクライアント.
 *  
 * @version 2006/09/13
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
public class JRcRemoteControlClient
{
    
    /**
     * リモート用コネクション.
     */
    private UdpProtocol udp = null ;
    
    /**
     * コネクション先アドレス用.
     */
    private ConnectAddress bindAddress = null ;
    
    /**
     * コンストラクタ.
     */
    public JRcRemoteControlClient() {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Throwable 例外処理が返されます.
     */
    protected final void finalize() throws Throwable {
        
        try{
            this.clear() ;
        }catch( Throwable t ){
        }
        
    }
    
    /**
     * 制御モードセット.
     * <BR><BR>
     * 制御を行うための情報を設定します.
     * また、このとき監視対象となる範囲は、[localhost:32108]となります.
     * <BR>
     * @exception InputException 入力例外.
     * @exception CommunicateException コネクション例外.
     */
    public final void create()
        throws InputException,CommunicateException {
        
        this.create( "127.0.0.1",JRcRemoteControlDef.DEF_PORT ) ;
        
    }
    
    /**
     * 制御モードセット.
     * <BR><BR>
     * 制御を行うための情報を設定します.
     * また、このとき監視対象となる範囲は、[localhost]となります.
     * <BR>
     * @param port コネクション先ポート番号を指定します.
     * @exception InputException 入力例外.
     * @exception CommunicateException コネクション例外.
     */
    public final void create( int port )
        throws InputException,CommunicateException {
        
        this.create( "127.0.0.1",port ) ;
        
    }
    
    /**
     * 制御モードセット.
     * <BR><BR>
     * 制御を行うための情報を設定します.
     * また、このとき監視対象となる範囲は、[localhost:32108]となります.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception CommunicateException コネクション例外.
     */
    public final void create( String addr )
        throws InputException,CommunicateException {
        
        this.create( addr,JRcRemoteControlDef.DEF_PORT ) ;
        
    }
    
    /**
     * 制御モードセット.
     * <BR><BR>
     * 制御を行うための情報を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port コネクション先ポート番号を指定します.
     * @exception InputException 入力例外.
     * @exception CommunicateException コネクション例外.
     */
    public final void create( String addr,int port )
        throws InputException,CommunicateException {
        
        try{
            
            // 接続先情報の設定.
            bindAddress = new ConnectAddress(
                addr,port
            ) ;
            
            // コネクション処理.
            udp = new UdpProtocol() ;
            udp.open( 4096 ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( CommunicateException cm ){
            throw cm ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear() {
        
        try{
            
            udp.close() ;
            bindAddress.clear() ;
            
        }catch( Throwable t ){
        }
        
        udp = null ;
        bindAddress = null ;
    }
    
    /**
     * シャットダウン情報送信.
     * <BR><BR>
     * シャットダウン情報を送信します.
     * <BR>
     * @return boolean シャットダウン結果が返されます.<BR>
     *                 [true]が返された場合、シャットダウン送信は成功です.
     *                 [false]が返された場合、シャットダウン送信は失敗です.
     */
    public final boolean remoteShutdown() {
        return this.remoteSend( JRcRemoteControlDef.REMOTE_SHUTDOWN ) ;
    }
    
    /**
     * 指定情報送信.
     */
    private final boolean remoteSend( int send ) {
        
        boolean ret ;
        
        byte[] code = null ;
        
        try{
            
            code = JRcRemoteControlDef.convertRemoteModeByBinary( send ) ;
            
            udp.send(
                code,
                bindAddress.getAddress(),
                bindAddress.getPort()
            ) ;
            
            ret = true ;
            
        }catch( BaseException be ){
            
            System.out.println( be.readStackTrace() ) ;
            ret = false ;
            
        }catch( Throwable t ){
            
            t.printStackTrace() ;
            ret = false ;
            
        }finally{
            
            code = null ;
            
        }
        
        return ret ;
        
    }
}
