/*
 * @(#)JRcSessionThread.java
 *
 * Copyright (c) 2006 KBMJ, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.DeleteSessionTrigger;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.UtilCom;

/**
 * セッションタイムアウト監視スレッド.
 *
 * @version 2006/09/07
 * @author  Masahito Suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcSessionThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( JRcSessionThread.class ) ;
    
    /**
     * １スレッド待機時間.
     */
    private static final long WAIT_TIME = 5000L ;
    
    
    
    /**
     * ループスレッド.
     */
    private LoopThread threadObject = null ;
    
    /**
     * セッションマネージャ.
     */
    private JRcSessionManagerImple manager = null ;
    
    /**
     * コールバックメソッド.
     */
    private DeleteSessionTrigger callback = null ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    private JRcSessionThread() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定して、スレッドを生成します.
     * <BR>
     * @param manager 対象のセッションマネージャを設定します.
     * @param callback 対象のコールバックオブジェクトを設定します.
     * @exception AccessException アクセス例外.
     */
    public JRcSessionThread( JRcSessionManagerImple manager,DeleteSessionTrigger callback )
        throws AccessException
    {
        if( manager == null || manager.isSessionManager() == false ) {
            throw new AccessException( "セッションマネージャは不正です" ) ;
        }
        
        sync.create() ;
        
        try{
            this.manager = manager ;
            this.callback = callback ;
            threadObject = new LoopThread() ;
            threadObject.create( this ) ;
            threadObject.startThread() ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception
    {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy()
    {
        
        try{
            synchronized( sync.get() ){
                
                try{
                    this.threadObject.clear() ;
                }catch( Exception e ){
                }
                
                this.threadObject = null ;
                this.manager = null ;
                this.callback = null ;
                
            }
        }catch( Exception ee ){
        }
        
        this.threadObject = null ;
        this.manager = null ;
        this.callback = null ;
        
        sync.clear() ;
        
    }
    
    /**
     * スレッドが有効であるかチェック.
     * <BR><BR>
     * スレッドが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isThread() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = true ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外.
     */
    protected void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int len ;
        
        JRcSessionImple imple = null ;
        
        try{
            
            UtilCom.idleSleep( WAIT_TIME ) ;
            
            synchronized( sync.get() ) {
                len = this.manager.size() ;
            }
            
            for( i = 0 ; i < len ; i ++ ) {
                
                synchronized( sync.get() ) {
                    
                    imple = ( JRcSessionImple )this.manager.getSession( i ) ;
                    
                    if( imple.getTimeoutRemainder() <= 0L ) {
                        
                        if( this.callback != null ) {
                            
                            try {
                                
                                callback.callback(
                                    imple.getApplicationName(),
                                    imple.getId()
                                ) ;
                                
                            } catch( Exception ee ) {
                                LOG.error( "JRcSessionThread [" +
                                    callback.getClass().getName() +
                                    "] - error.",ee ) ;
                            }
                            
                        }
                        
                        try {
                            LOG.info(
                                "## [remove-session]: applicationName:" +
                                imple.getApplicationName() +
                                " sessionID:" +
                                imple.getId()
                            ) ;
                            manager.removeSession( imple ) ;
                        } catch( Exception ee ) {
                        }
                        
                        len -- ;
                        i -- ;
                        
                    }
                    
                }
                
                imple = null ;
                UtilCom.idleTime() ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( Exception e ){
            LOG.error( "JRcSessionThread - error.",e ) ;
        }finally{
            imple = null ;
        }
        
    }
    
}

