/*
 * @(#)JRcSessionManager.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import com.JRcServer.JRcSession;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.serialize.SerializeUtil;
import com.JRcServer.commons.util.BigTable;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * JRcServerセッションマネージャ.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcSessionManagerImple implements JRcSessionManager,SerializableSession {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            JRcSessionManager.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 最大生成リトライ数.
     */
    private static final int MAX_RETRY_CREATE = 999 ;
    
    /**
     * セッションオブジェクト格納情報.
     */
    private ObjectArray array = null ;
    
    /**
     * セッションIDでセッション群を管理するテーブル.
     */
    private transient BigTable table = null ;
    
    /**
     * セッションタイムアウト.
     */
    private long sessionTimeout = -1L ;
    
    /**
     * セッション終了フラグ.
     */
    private transient boolean exitSessionFlag = false ;
    
    
    /**
     * コンストラクタ.
     */
    public JRcSessionManagerImple() {
        this( JRcSession.DEF_SESSION_TIMEOUT ) ;
    }
    
    /**
     * コンストラクタ.
     */
    public JRcSessionManagerImple( long sessionTimeout ) {
        
        if( sessionTimeout <= JRcSession.MIN_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MIN_SESSION_TIMEOUT ;
        }
        if( sessionTimeout >= JRcSession.MAX_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MAX_SESSION_TIMEOUT ;
        }
        
        this.array = new ObjectArray() ;
        this.table = new BigTable() ;
        this.sessionTimeout = sessionTimeout ;
        this.exitSessionFlag = false ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクト内容を破棄します.
     */
    public synchronized void destroy() {
        
        this.array = null ;
        this.table = null ;
        this.sessionTimeout = -1L ;
        this.exitSessionFlag = true ;
        
    }
    
    /**
     * セッション初期化処理.
     * <BR><BR>
     * セッション初期化処理を行います.<BR>
     * この処理はオブジェクトがシリアライズから復帰した後に実施する
     * 必要があります.
     * <BR>
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、セッション情報は正しく再構成されました.<BR>
     *                 [false]が返された場合、セッション情報は正しく再構成されませんでした.
     */
    public synchronized boolean initSession() {
        
        int i ;
        int len ;
        boolean flg ;
        
        JRcSessionImple imple = null ;
        
        if( array != null && ( len = array.size() ) > 0 ) {
            
            table = new BigTable() ;
            
            for( i = 0 ; i < len ; i ++ ) {
                
                flg = false ;
                
                imple = ( JRcSessionImple )array.get( i ) ;
                
                if( imple != null && imple.initSession() == true ) {
                    
                    try {
                        
                        imple.setSessionTimeout( sessionTimeout ) ;
                        table.add( imple.getId(),imple ) ;
                        flg = true ;
                        
                    } catch( Exception e ) {
                    }
                    
                }
                
                if ( flg == false ) {
                    array.remove( i ) ;
                    i -- ;
                }
                
            }
            
        }
        else if( array == null ) {
            
            array = new ObjectArray() ;
            table = new BigTable() ;
            
        }
        
        this.exitSessionFlag = false ;
        
        return true ;
        
    }
    
    /**
     * セッション終了処理.
     * <BR><BR>
     * セッション終了処理を実施します.<BR>
     * この処理はオブジェクトがシリアライズする前に実施する
     * 必要があります.
     * <BR>
     * @return boolean シリアライズが可能であるか返されます.<BR>
     *                 [true]が返された場合、シリアライズ可能です.<BR>
     *                 [false]が返された場合、シリアライズ不可能です.
     */
    public synchronized boolean exitSession() {
        
        int i ;
        int len ;
        int cnt = 0 ;
        boolean flg = false ;
        
        JRcSessionImple imple = null ;
        this.exitSessionFlag = true ;
        
        if( array != null && ( len = array.size() ) > 0 ) {
            
            for( i = 0,cnt = 0 ; i < len ; i ++ ) {
                
                flg = false ;
                
                if( ( imple = ( JRcSessionImple )array.get( i ) ) != null ) {
                    
                    if( imple.exitSession() == true ) {
                        
                        flg = true ;
                        cnt ++ ;
                        
                    }
                    
                }
                
                if( flg == false ) {
                    
                    array.remove( i ) ;
                    i -- ;
                    len -- ;
                    
                }
                
            }
            
        }
        
        return ( cnt <= 0 ) ? false : true ;
        
    }
    
    /**
     * 新しいセッションオブジェクトを取得.
     * <BR><BR>
     * 新しいセッションオブジェクトを取得します.
     * <BR>
     * @param applicationName 対象のアプリケーション名を設定します.
     * @return JRcSession セッションオブジェクトが返されます.
     */
    public synchronized JRcSession createSession( String applicationName ) {
        
        int i ;
        
        long id = -1L ;
        JRcSessionImple ret = null ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return null ;
        }
        
        if(
            applicationName == null ||
            ( applicationName = applicationName.trim().toLowerCase() ).length() <= 0
        ) {
            return null ;
        }
        
        try {
            
            for( i = 0 ; i < MAX_RETRY_CREATE ; i ++ ) {
                
                id = ( long )(
                    ( ( long )UtilCom.random( Integer.MAX_VALUE ) & 0x00000000ffffffffL ) |
                    ( ( ( long )UtilCom.random( Integer.MAX_VALUE ) & 0x000000007fffffffL ) << 32L )
                ) ;
                
                if( table.isData( id ) == false ) {
                    break ;
                }
                
                id = -1L ;
                UtilCom.cpuCreate() ;
                
            }
            
            if( id == -1L ) {
                ret = null ;
            }
            else {
                
                ret = new JRcSessionImple(
                    applicationName,id,sessionTimeout
                ) ;
                
                table.add( id,ret ) ;
                array.add( ret ) ;
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * セッションオブジェクトを削除.
     * <BR><BR>
     * セッションオブジェクトを削除します.
     * <BR>
     * @param applicationName 削除対象のアプリケーション名を設定します.
     * @param sessionID 削除対象のセッションIDを設定します.
     */
    public synchronized void removeSession( String applicationName,long sessionID ) {
        
        int i ;
        int len ;
        
        JRcSessionImple imple = null ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return ;
        }
        
        if(
            applicationName == null ||
            ( applicationName = applicationName.trim().toLowerCase() ).length() <= 0 ||
            sessionID < 0L
        ) {
            return ;
        }
        
        if(
            ( imple = ( JRcSessionImple )table.get( sessionID ) ) != null &&
            applicationName.equals( imple.getApplicationName() ) == true
        ) {
            
            len = array.size() ;
            
            for( i = 0 ; i < len ; i ++ ) {
                
                if( imple == array.get( i ) ) {
                    array.remove( i ) ;
                    table.remove( sessionID ) ;
                    break ;
                }
                
            }
            
        }
        
    }
    
    /**
     * セッションオブジェクトを削除.
     * <BR><BR>
     * セッションオブジェクトを削除します.
     * <BR>
     * @param session 削除対象のセッションオブジェクトを設定します.
     */
    public synchronized void removeSession( JRcSession session ) {
        
        if( session != null ) {
            
            this.removeSession(
                session.getApplicationName(),session.getId() ) ;
            
        }
        
    }
    
    /**
     * セッションオブジェクトを取得.
     * <BR><BR>
     * セッションオブジェクトを取得します.
     * <BR>
     * @param applicationName 対象のアプリケーション名を設定します.
     * @param sessionID 対象のセッションIDを設定します.
     * @return JRcSession セッションオブジェクトが返されます.
     */
    public synchronized JRcSession getSession( String applicationName,long sessionID ) {
        
        JRcSessionImple ret = null ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return null ;
        }
        
        if(
            applicationName == null ||
            ( applicationName = applicationName.trim().toLowerCase() ).length() <= 0 ||
            sessionID < 0L
        ) {
            return null ;
        }
        
        if(
            ( ret = ( JRcSessionImple )table.get( sessionID ) ) == null ||
            applicationName.equals( ret.getApplicationName() ) == false
        ) {
            
            ret = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象項番からセッションオブジェクトを取得.
     * <BR><BR>
     * 対象の項番からセッションオブジェクトを取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return JRcSession セッションオブジェクトが返されます.
     */
    public synchronized JRcSession getSession( int no ) {
        
        JRcSession ret = null ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return null ;
        }
        
        try {
            ret = ( JRcSession )array.get( no ) ;
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * セッション情報数を取得.
     * <BR><BR>
     * 管理されているセッション数を取得します.
     * <BR>
     * @return int セッション情報数が返されます.
     */
    public synchronized int size() {
        
        int ret = 0 ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return 0 ;
        }
        
        try {
            ret = array.size() ;
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * セッションID群を取得.
     * <BR><BR>
     * 管理されているセッションID群を取得します.
     * <BR>
     * @return long[] セッションID群が返されます.
     */
    public synchronized long[] getSessionIDs() {
        
        long[] ret = null ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return null ;
        }
        
        try {
            ret = table.getNumbers() ;
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * セッションタイムアウト値を変更.
     * <BR><BR>
     * 現在のセッションタイムアウト値を変更します.
     * <BR>
     * @param sessionTimeout 変更対象のセッションタイムアウト値を設定します.
     */
    public synchronized void setSessionTimeout( long sessionTimeout ) {
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return ;
        }
        
        if( sessionTimeout <= JRcSession.MIN_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MIN_SESSION_TIMEOUT ;
        }
        if( sessionTimeout >= JRcSession.MAX_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MAX_SESSION_TIMEOUT ;
        }
        
        this.sessionTimeout = sessionTimeout ;
        
    }
    
    /**
     * 設定されているセッションタイムアウト値を取得.
     * <BR><BR>
     * 設定されているセッションタイムアウト値を取得します.
     * <BR>
     * @return long 設定されているセッションタイムアウト値が返されます.
     */
    public synchronized long getSessionTimeout() {
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return -1L ;
        }
        
        return this.sessionTimeout ;
        
    }
    
    /**
     * セッションIDが存在するかチェック.
     * <BR><BR>
     * 指定したセッションIDが管理情報に存在するかチェックします.
     * <BR>
     * @param sessionID 対象のセッションIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、セッションIDは存在します.<BR>
     *                 [false]が返された場合、セッションIDは存在しません.
     */
    public synchronized boolean isSessionID( long sessionID ) {
        
        boolean ret = false ;
        
        if( exitSessionFlag == true || array == null || table == null ) {
            return false ;
        }
        
        try {
            ret = table.isData( sessionID ) ;
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * セッション管理マネージャが有効かチェック.
     * <BR><BR>
     * セッション管理マネージャが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、セッション管理マネージャは有効です.<BR>
     *                 [false]が返された場合、セッション管理マネージャは無効です.
     */
    public synchronized boolean isSessionManager() {
        
        if( array == null || table == null ) {
            return false ;
        }
        
        return true ;
        
    }
    
    /**
     * オブジェクトを文字列に変換.
     * <BR><BR>
     * オブジェクトを文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public synchronized String toString() {
        
        int i ;
        int len ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( array != null && ( len = array.size() ) > 0 ) {
            
            buf = new StringBuffer() ;
            
            for( i = 0 ; i < len ; i ++ ) {
                
                buf.append( " {" ).
                append( (i+1) ).
                append( "}=" ).
                append( array.get( i ) ).
                append( BaseDef.ENTER_SEPARATOR ) ;
                
            }
            
            ret = buf.toString() ;
            
        }
        else {
            ret = " not " ;
        }
        
        return ret ;
        
    }
    
}

