/*
 * @(#)JRcSessionImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import java.io.Serializable;

import com.JRcServer.JRcSession;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.DateTimeFormat;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * JRcServerセッション実体.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcSessionImple implements JRcSession,SerializableSession {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            JRcSessionImple.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    
    
    /**
     * セッション利用名.
     */
    private String applicationName = null ;
    
    /**
     * セッションID.
     */
    private long sessionID = -1L ;
    
    /**
     * セッション生成日付.
     */
    private long createSessionTime = -1L ;
    
    /**
     * セッション最終アクセス日付.
     */
    private transient long lastAccessSessionTime = -1L ;
    
    /**
     * セッションタイムアウト値.
     */
    private long sessionTimeout = -1L ;
    
    
    /**
     * セッション情報格納情報.
     */
    private ObjectArray sessionArray = null ;
    
    /**
     * セッション格納テーブル.
     */
    private transient CharTable sessionTable = null ;
    
    /**
     * セッション終了フラグ.
     */
    private transient boolean exitSessionFlag = false ;
    
    
    /**
     * コンストラクタ.
     */
    private JRcSessionImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 新しくセッションオブジェクトを生成します.
     * <BR>
     * @param applicationName アプリケーション名を設定します.
     * @param sessionID 対象のセッションIDを設定します.
     * @param sessionTimeout 対象のセッションタイムアウト値を設定します.
     * @exception InputException 入力例外.
     */
    public JRcSessionImple( String applicationName,long sessionID,long sessionTimeout )
        throws InputException {
        
        long time ;
        
        if(
            applicationName == null ||
            ( applicationName = applicationName.trim().toLowerCase() ).length() <= 0 ||
            sessionID < 0L ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( sessionTimeout <= JRcSession.MIN_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MIN_SESSION_TIMEOUT ;
        }
        if( sessionTimeout >= JRcSession.MAX_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MAX_SESSION_TIMEOUT ;
        }
        
        this.applicationName = applicationName ;
        this.sessionID = sessionID ;
        this.sessionTimeout = sessionTimeout ;
        
        time = System.currentTimeMillis() ;
        this.createSessionTime = time ;
        this.lastAccessSessionTime = time ;
        
        this.sessionArray = new ObjectArray() ;
        this.sessionTable = new CharTable() ;
        this.exitSessionFlag = false ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public synchronized void destroy() {
        
        if( sessionArray != null ) {
            sessionArray.clear() ;
        }
        if( sessionTable != null ) {
            sessionTable.clear() ;
        }
        
        sessionArray = null ;
        sessionTable = null ;
        
        applicationName = null ;
        sessionID = -1L ;
        createSessionTime = -1L ;
        lastAccessSessionTime = -1L ;
        sessionTimeout = -1L ;
        
        exitSessionFlag = true ;
        
    }
    
    /**
     * セッション初期化処理.
     * <BR><BR>
     * セッション初期化処理を行います.<BR>
     * この処理はオブジェクトがシリアライズから復帰した後に実施する
     * 必要があります.
     * <BR>
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、セッション情報は正しく再構成されました.<BR>
     *                 [false]が返された場合、セッション情報は正しく再構成されませんでした.
     */
    public synchronized boolean initSession() {
        
        int i ;
        int len ;
        boolean flg ;
        
        String key = null ;
        Object value = null ;
        SessionValue val = null ;
        
        if( this.sessionID < 0L ) {
            this.exitSessionFlag = true ;
            return false ;
        }
        
        this.lastAccessSessionTime = System.currentTimeMillis() ;
        
        if( sessionArray != null && ( len = sessionArray.size() ) > 0 ) {
            
            sessionTable = new CharTable() ;
            
            for( i = 0 ; i < len ; i ++ ) {
                
                flg = false ;
                
                if( ( val = ( SessionValue )sessionArray.get( i ) ) != null ) {
                    
                    key = val.getKey() ;
                    value = val.getValue() ;
                    
                    if( key != null && key.length() > 0 && value != null ) {
                        try {
                            this.sessionTable.add( key,value ) ;
                            flg = true ;
                        } catch( Exception e ) {
                            flg = false ;
                        }
                    }
                    
                    if( flg == false ) {
                        sessionArray.remove( i ) ;
                        i -- ;
                        len -- ;
                    }
                    
                }
                
            }
            
        }
        else if( sessionArray == null ) {
            sessionArray = new ObjectArray() ;
            sessionTable = new CharTable() ;
        }
        
        this.exitSessionFlag = false ;
        this.lastAccessSessionTime = System.currentTimeMillis() ;
        
        return true ;
        
    }
    
    /**
     * セッション終了処理.
     * <BR><BR>
     * セッション終了処理を実施します.<BR>
     * この処理はオブジェクトがシリアライズする前に実施する
     * 必要があります.
     * <BR>
     * @return boolean シリアライズが可能であるか返されます.<BR>
     *                 [true]が返された場合、シリアライズ可能です.<BR>
     *                 [false]が返された場合、シリアライズ不可能です.
     */
    public synchronized boolean exitSession() {
        
        int i ;
        int len ;
        int cnt ;
        boolean flg ;
        
        String key = null ;
        SessionValue val = null ;
        
        this.exitSessionFlag = true ;
        cnt = 0 ;
        
        if( this.sessionArray != null &&
            ( len = this.sessionArray.size() ) > 0
        ) {
            
            for( i = 0 ; i < len ; i ++ ) {
                
                flg = false ;
                
                if(
                    ( val = ( SessionValue )this.sessionArray.get( i ) ) != null &&
                    val.checkSerializable() == true
                ) {
                    cnt ++ ;
                    flg = true ;
                }
                
                if( flg == false ) {
                    
                    this.sessionArray.remove( i ) ;
                    
                    if( val != null ) {
                        
                        key = val.getKey() ;
                        
                        if( this.sessionTable != null && key != null && key.length() > 0 ) {
                            
                            try {
                                this.sessionTable.remove( key ) ;
                            } catch( Exception e ) {
                            }
                            
                        }
                        
                    }
                    
                    i -- ;
                    len -- ;
                    
                }
                
            }
            
            this.lastAccessSessionTime = System.currentTimeMillis() ;
            
        }
        
        return ( cnt <= 0 ) ? false : true ;
        
    }
    
    /**
     * 要素を追加.
     * <BR><BR>
     * 要素を追加します.
     * <BR>
     * @param name 追加対象の要素名を設定します.
     * @param value 要素情報を設定します.
     * @exception InputException 引数が不正な場合.
     */
    public synchronized void putValue( String name,Object value )
        throws InputException {
        
        int i ;
        int len ;
        
        SessionValue val = null ;
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return ;
        }
        
        if (
            name == null || name.length() <= 0 ||
            value == null ) {
            throw new InputException( "引数は不正です(name:" + name + " value:" + value + ")" ) ;
        }
        
        if( sessionTable.isData( name ) == false ) {
            sessionTable.add( name,value ) ;
            sessionArray.add( new SessionValue( name,value ) ) ;
        }
        else {
            
            len = sessionArray.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                if( ( val = ( SessionValue )sessionArray.get( i ) ) != null ) {
                    
                    if( name.equals( val.getKey() ) == true ) {
                        val.setValue( value ) ;
                        sessionTable.add( name,value ) ;
                        break ;
                    }
                    
                }
                else {
                    
                    sessionArray.remove( i ) ;
                    i -- ;
                    len -- ;
                    
                }
            }
            
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
    }
    
    /**
     * 要素情報を削除.
     * <BR><BR>
     * 要素を削除します.
     * <BR>
     * @param name 削除対象の要素名を設定します.
     * @exception InputException 引数が不正な場合.
     */
    public synchronized void removeValue( String name )
        throws InputException {
        
        int i ;
        int len ;
        
        SessionValue val = null ;
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return ;
        }
        
        if ( name == null || name.length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( sessionTable.isData( name ) == true ) {
            
            sessionTable.remove( name ) ;
            len = sessionArray.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                if( ( val = ( SessionValue )sessionArray.get( i ) ) != null ) {
                    
                    if( name.equals( val.getKey() ) == true ) {
                        sessionArray.remove( i ) ;
                        break ;
                    }
                    
                }
                else {
                    
                    sessionArray.remove( i ) ;
                    i -- ;
                    len -- ;
                    
                }
            }
            
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
    }
    
    /**
     * 要素を取得.
     * <BR><BR>
     * 要素を取得します.
     * <BR>
     * @param name 取得対象の要素名を設定します.
     * @return Object 要素情報が返されます.
     */
    public synchronized Object getValue( String name ) {
        
        Object ret = null ;
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return null ;
        }
        
        if ( name == null || name.length() <= 0 ) {
            return null ;
        }
        
        if( sessionTable.isData( name ) == true ) {
            ret = sessionTable.get( name ) ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return ret ;
        
    }
    
    /**
     * 登録されている要素名群を取得.
     * <BR><BR>
     * 登録されている要素名群を取得します.
     * <BR>
     * @return String[] 登録されている要素名群が返されます.
     */
    public synchronized String[] getNames() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return null ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return sessionTable.getNames() ;
        
    }
    
    /**
     * 登録されている要素数を取得.
     * <BR><BR>
     * 登録されている要素数を取得します.
     * <BR>
     * @return int 登録されている要素数が返されます.
     */
    public synchronized int size() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return 0 ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return sessionTable.size() ;
        
    }
    
    /**
     * アプリケーション名を取得.
     * <BR><BR>
     * アプリケーション名を取得します.
     * <BR>
     * @return String アプリケーション名が返されます.
     */
    public synchronized String getApplicationName() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return null ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return applicationName ;
        
    }
    
    /**
     * セッションIDを取得.
     * <BR><BR>
     * セッションIDを取得します.
     * <BR>
     * @return long セッションIDが返されます.
     */
    public synchronized long getId() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return -1L ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return sessionID ;
        
    }
    
    /**
     * セッション生成時間を取得.
     * <BR><BR>
     * セッション生成の時間を取得します.
     * <BR>
     * @return long セッションが生成された時間が返されます.
     */
    public synchronized long getCreateTime() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return -1L ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return createSessionTime ;
        
    }
    
    /**
     * ラストアクセス時間を取得.
     * <BR><BR>
     * ラストアクセス時間を取得します.
     * <BR>
     * @return long ラストアクセス時間が返されます.
     */
    public synchronized long getLastAccessTime() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return -1L ;
        }
        
        return lastAccessSessionTime ;
        
    }
    
    /**
     * セッションタイムアウトを設定.
     * <BR><BR>
     * セッションタイムアウト値を設定します.
     * <BR>
     * @param sessionTimeout セッションタイムアウトを設定します.
     */
    public synchronized void setSessionTimeout( long sessionTimeout ) {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return ;
        }
        
        if( sessionTimeout <= JRcSession.MIN_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MIN_SESSION_TIMEOUT ;
        }
        if( sessionTimeout >= JRcSession.MAX_SESSION_TIMEOUT ) {
            sessionTimeout = JRcSession.MAX_SESSION_TIMEOUT ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        this.sessionTimeout = sessionTimeout ;
        
    }
    
    /**
     * セッションタイムアウトを取得.
     * <BR><BR>
     * セッションタイムアウトを取得します.
     * <BR>
     * @return long セッションのタイムアウト時間が返されます.
     */
    public synchronized long getSessionTimeout() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return -1L ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return sessionTimeout ;
        
    }
    
    /**
     * 最後にリクエストされた時間からセッションタイムアウトする
     * までの時間を取得.
     * <BR><BR>
     * 最後にリクエストされた時間からセッションタイムアウトする
     * までの時間を取得します.
     * <BR>
     * @return long 最後にリクエストされた時間からの
     *              セッションタイムアウト値が返されます.
     */
    public synchronized long getTimeoutRemainder() {
        
        if( exitSessionFlag == true || sessionTable == null ) {
            return -1L ;
        }
        
        return ( lastAccessSessionTime + sessionTimeout ) -
            System.currentTimeMillis() ;
        
    }
    
    /**
     * セッション情報が有効であるかチェック.
     * <BR><BR>
     * このセッション情報が有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、セッションは有効です.<BR>
     *                 [false]が返された場合、セッションは無効です.
     */
    public synchronized boolean isSession() {
        
        if( sessionTable == null || sessionArray == null || sessionID < 0L) {
            return false ;
        }
        
        lastAccessSessionTime = System.currentTimeMillis() ;
        
        return true ;
        
    }
    
    /**
     * オブジェクトを文字列に変換.
     * <BR><BR>
     * オブジェクトを文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public synchronized String toString() {
        
        int i ;
        int len ;
        
        DateTimeFormat fmt = null ;
        StringBuffer buf = null ;
        SessionValue val = null ;
        String ret = null ;
        
        if( sessionArray != null && ( len = sessionArray.size() ) > 0 ) {
            
            buf = new StringBuffer() ;
            fmt = new DateTimeFormat( "YYYY/MM/DD hh:mm:ss.SSSS" ) ;
            
            buf.append( " applicationName:" ).
            append( applicationName ).
            append( " sessionID:" ).
            append( sessionID ).
            append( " create:" ).
            append( fmt.getString( createSessionTime ) ).
            append( " lastAccess:" ).
            append( fmt.getString( lastAccessSessionTime ) ).
            append( " sessionTimeout:" ).
            append( sessionTimeout ) ;
            
            for( i = 0 ; i < len ; i ++ ) {
                
                val = ( SessionValue )sessionArray.get( i ) ;
                
                buf.append( " [" ).
                append( (i+1) ).
                append( "]:" ).
                append( val.toString() ) ;
                
            }
            
            ret = buf.toString() ;
            
        }
        else {
            ret = " not " ;
        }
        
        return ret ;
    }
    
}

/**
 * セッション内容管理データ.
 */
class SessionValue implements Serializable {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            SessionValue.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * キー名.
     */
    private String key = null ;
    
    /**
     * 要素情報.
     */
    private Object value = null ;
    
    /**
     * コンストラクタ.
     */
    public SessionValue() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param key キー情報を設定します.
     * @param value 要素情報を設定します.
     */
    public SessionValue( String key,Object value ) {
        this.key = key ;
        this.value = value ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        key = null ;
        value = null ;
    }
    
    /**
     * key を取得.
     * <BR><BR>
     * @return key が返されます.
     */
    public String getKey() {
        return key;
    }
    
    /**
     * key を設定.
     * <BR><BR>
     * @param key key を設定します.
     */
    public void setKey(String key) {
        this.key = key;
    }
    
    /**
     * value を取得.
     * <BR><BR>
     * @return value が返されます.
     */
    public Object getValue() {
        return value;
    }
    
    /**
     * value を設定.
     * <BR><BR>
     * @param value value を設定します.
     */
    public void setValue(Object value) {
        this.value = value;
    }
    
    /**
     * このオブジェクトがシリアライズ可能かチェック.
     * <BR><BR>
     * このオブジェクトがシリアライズ可能であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、シリアライズ可能です.
     *                 [false]が返された場合、シリアライズ不可能です.
     */
    protected boolean checkSerializable() {
        
        boolean ret = false ;
        
        if(
            this.key == null || this.key.length() <= 0 ||
            this.value == null
        ) {
            return false ;
        }
        
        try {
            
            if( UtilCom.convertObjectByBinary( ( Serializable )this.value ) != null ) {
                ret = true ;
            }
            else {
                ret = false ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * オブジェクトを文字列に変換.
     * <BR><BR>
     * オブジェクトを文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public String toString() {
        
        return new StringBuffer().
            append( " key:" ).
            append( key ).
            append( " value:" ).
            append( value ).
            toString() ;
            
    }
    
}

