/*
 * @(#)JRcServerConf.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import com.JRcServer.commons.conf.BaseConfig;
import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.ReadSyntaxException;
import com.JRcServer.commons.io.FileAccessException;

/**
 * JRcServerコンフィグ.
 *
 * @version 2006/09/13
 * @author  Masahito Suzuki
 * @since   JRcServerAPI 1.00
 */
public class JRcServerConf extends BaseConfig
{
    
    /**
     * コンフィグセクション : JRcServerコンフィグファイル.
     */
    private static final String SECTION_JRC_SERVER_BEAN_CONFIG = "jrcserver-config" ;
    
    /**
     * JRcServerコンフィグファイル : バインドポート.
     */
    private static final String BIND_PORT = "bind-port" ;
    
    /**
     * JRcServerコンフィグファイル : バインドアドレス.
     */
    private static final String BIND_ADDR = "bind-address" ;
    
    /**
     * JRcServerコンフィグファイル : キャッシュサイズ.
     */
    private static final String CACHE_SIZE = "cache-size" ;
    
    /**
     * JRcServerコンフィグファイル : セッションタイムアウト.
     */
    private static final String SESSION_TIMEOUT = "session-timeout" ;
    
    /**
     * JRcServerコンフィグファイル : セッションタイムアウトコールバックオブジェクト.
     */
    private static final String SESSION_CALLBACK = "session-callback" ;
    
    /**
     * JRcServerコンフィグファイル : 最大コネクション数.
     */
    private static final String MAX_CONNECTION = "max-connection" ;
    
    /**
     * JRcServerコンフィグファイル : 受信タイムアウト.
     */
    private static final String RECEIVE_TIMEOUT = "receive-timeout" ;
    
    /**
     * JRcServerコンフィグファイル : サーバ終了メッセージ受信ポート番号.
     */
    private static final String EXIT_RECEIVE_PORT = "exit-receive-port" ;
    
    
    
    /**
     * コンストラクタ.
     */
    public JRcServerConf() {
        super() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * MQ定義ファイルを指定して生成.
     * <BR><BR>
     * MQ定義ファイルを指定して生成します.
     * <BR>
     * @param name 生成対象のファイル名を設定します.
     * @exception ExecutionException 実行例外.
     */
    public void create( String name )
        throws ExecutionException {
        
        try{
            
            this.open( name ) ;
            
        }catch( BaseException be ){
            this.clear() ;
            throw new ExecutionException( be ) ;
        }catch( Exception e ){
            this.clear() ;
            throw new ExecutionException( e ) ;
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクト情報をクリアします.
     */
    public void clear() {
        
        super.close() ;
    }
    
    /**
     * バインドポートを取得.
     * <BR><BR>
     * バインドポートを取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[bind-port]情報を取得します.
     * <BR>
     * @return int バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getBindPort() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.BIND_PORT,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[bind-address]情報を取得します.
     * <BR>
     * @return String バインドアドレスが返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public String getBindAddress() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.BIND_ADDR,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * キャッシュサイズを取得.
     * <BR><BR>
     * キャッシュサイズを取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[cache-size]情報を取得します.
     * <BR>
     * @return int キャッシュサイズが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getCacheSize() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.CACHE_SIZE,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * セッションタイムアウトを取得.
     * <BR><BR>
     * セッションタイムアウトを取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[session-timeout]情報を取得します.
     * <BR>
     * @return long セッションタイムアウトが返されます.<BR>
     *              情報が存在しない場合[-1L]が返されます.
     */
    public long getSessionTimeout() {
        
        long ret = -1L ;
        
        try{
            
            ret = Long.parseLong(
                this.getString(
                    JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                    JRcServerConf.SESSION_TIMEOUT,
                    0
                )
            ) ;
            
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * セッションタイムアウトコールバックオブジェクトを取得.
     * <BR><BR>
     * セッションタイムアウトコールバックオブジェクトを取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[session-callback]情報を取得します.
     * <BR>
     * @return String セッションタイムアウトコールバックオブジェクトが返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public String getSessionCallback() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.SESSION_CALLBACK,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 最大コネクション数を取得.
     * <BR><BR>
     * 最大コネクション数を取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[max-connection]情報を取得します.
     * <BR>
     * @return int 最大コネクション数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getMaxConnection() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.MAX_CONNECTION,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信タイムアウトを取得.
     * <BR><BR>
     * 受信タイムアウトを取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[receive-timeout]情報を取得します.
     * <BR>
     * @return int 受信タイムアウトが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getReceiveTimeout() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.RECEIVE_TIMEOUT,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    
    /**
     * サーバ終了メッセージ受信ポート番号を取得.
     * <BR><BR>
     * サーバ終了メッセージ受信ポート番号を取得します.<BR>
     * この情報はセクション名[jrcserver-config]以下に
     * 設定されている[exit-receive-port]情報を取得します.
     * <BR>
     * @return int 受信タイムアウトが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getExitReceivePort() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG,
                JRcServerConf.EXIT_RECEIVE_PORT,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 対象の情報を文字列で取得.
     * <BR><BR>
     * 対象の情報を文字列で取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return String 対象の条件をStringで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    public String get( String section,String key,int no ) {
        
        String ret = null ;
        
        try{
            
            ret = this.getString( section,key,no ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象条件数を取得.
     * <BR><BR>
     * 対象の条件の定義数を取得します.
     * <BR>
     * @return int 定義数が返されます.
     */
    public int size( String section,String key ) {
        return this.getLength( section,key ) ;
    }
    
    /**
     * 定義ファイルで必要なセクション名群が有効であるかチェック.
     * <BR><BR>
     * 定義ファイルで必要なセクション名群が有効であるかチェックします.
     * <BR>
     * @return boolean 有効であるかチェックします.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、有効ではありません.
     */
    public boolean isUseSection() {
        return (
            this.isSection( JRcServerConf.SECTION_JRC_SERVER_BEAN_CONFIG ) == false
        ) ? false : true ;
    }
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象のセクション名が存在するかチェックします.
     * @param section 対象のセクション名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、セクション名は存在します.<BR>
     *                 [false]が返された場合、セクション名は存在しません.
     */
    public boolean isSection( String section ) {
        return super.isSection( section ) ;
    }
    
    
    
    
    /**
     * 対象コンフィグ情報を取得.
     * <BR><BR>
     * 対象のコンフィグ情報を取得します.
     * <BR>
     * @param name 対象のコンフィグ名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    protected final void newInit( String name )
        throws InputException,FileAccessException,ReadSyntaxException {
        
        try{
            synchronized( m_sync.get() ){
                m_conf.create( name ) ;
                m_name = name ;
                m_isRead = true ;
            }
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.clear() ;
            throw fa ;
        }catch( ReadSyntaxException rs ){
            this.clear() ;
            throw rs ;
        }
    }
    
}
