/*
 * @(#)JRcResponseImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;

import com.JRcServer.JRcResponse;
import com.JRcServer.JRcResponseBean;
import com.JRcServer.JRcResultDef;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.BinResourceOutputStream;

/**
 * JRcServerレスポンス実装.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcResponseImple implements JRcResponse {
    
    /**
     * レスポンスBean.
     */
    private JRcResponseBean bean = null ;
    
    /**
     * OutputStream.
     */
    private OutputStream outputStream = null ;
    
    /**
     * PrintWriter.
     */
    private PrintWriter printWriter = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    private JRcResponseImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * Beanを設定してオブジェクトを生成します.
     * <BR>
     * @param bean 対象のBeanを設定します.
     * @exception InputException 入力例外.
     */
    public JRcResponseImple( JRcResponseBean bean )
        throws InputException {
        
        if( bean == null || bean.isBean() == false ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.bean = bean ;
        this.outputStream = new BufferedOutputStream(
            new BinResourceOutputStream( bean.getBinary() )
        ) ;
        this.printWriter = null ;
        this.bean.setResult( JRcResultDef.SUCCESS ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public synchronized void destroy() {
        
        if( this.printWriter != null ) {
            try {
                this.printWriter.flush() ;
                this.printWriter.close() ;
            } catch( Exception e ) {
            }
        }
        
        if( this.outputStream != null ) {
            try {
                this.outputStream.flush() ;
                this.outputStream.close() ;
            } catch( Exception e ) {
            }
        }
        
        this.bean = null ;
        this.outputStream = null ;
        this.printWriter = null ;
        
    }
    
    /**
     * データフラッシュ.
     * <BR><BR>
     * キャッシュデータをフラッシュします.
     */
    public synchronized void flush() {
        
        if( this.printWriter != null ) {
            try {
                this.printWriter.flush() ;
                this.printWriter.close() ;
            } catch( Exception e ) {
            }
        }
        
        if( this.outputStream != null ) {
            try {
                this.outputStream.flush() ;
            } catch( Exception e ) {
            }
        }
        
        this.printWriter = null ;
        
    }
    
    /**
     * セッションIDを設定.
     */
    public synchronized void setSessionID( long sessionID ) {
        
        if(
            this.bean == null || this.bean.isBean() == false ||
            sessionID < 0L
        ) {
            return ;
        }
        
        this.bean.setSessionID( sessionID ) ;
        
    }
    
    /**
     * セッションIDを取得.
     * <BR><BR>
     * セッションIDを取得します.
     * <BR>
     * @return long セッションIDが返されます.
     */
    public synchronized long getSessionID() {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return -1L ;
        }
        
        return bean.getSessionID() ;
        
    }
    
    /**
     * キャラクタセットを設定.
     * <BR><BR>
     * キャラクタセットを設定します.
     * <BR>
     * @param charset 対象のキャラクタセットを設定します.
     */
    public synchronized void setCharset( String charset ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        if( charset == null || ( charset = charset.trim() ).length() <= 0 ) {
            return ;
        }
        
        this.bean.setCharset( charset ) ;
        
    }
    
    /**
     * キャラクタセットを取得.
     * <BR><BR>
     * キャラクタセットを取得します.
     * <BR>
     * @return String キャラクタセットが返されます.
     */
    public synchronized String getCharset() {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return null ;
        }
        
        return this.bean.getCharset() ;
        
    }
    
    /**
     * Mimeタイプを設定.
     * <BR><BR>
     * Mimeタイプを設定します.
     * <BR>
     * @param mimeType 対象のMimeタイプを設定します.
     */
    public void setMimeType( String mimeType ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        if( mimeType == null || ( mimeType = mimeType.trim() ).length() <= 0 ) {
            return ;
        }
        
        this.bean.setMimeType( mimeType ) ;
        
    }
    
    /**
     * Mimeタイプを取得.
     * <BR><BR>
     * Mimeタイプを取得します.
     * <BR>
     * @return String Mimeタイプが返されます.
     */
    public String getMimeType() {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return null ;
        }
        
        return this.bean.getMimeType() ;
        
    }
    
    /**
     * レスポンスにバイナリデータを出力する際に
     * 使用するOutputStreamを取得.
     * <BR><BR>
     * レスポンスにバイナリデータを出力する際に
     * 使用するOutputStreamを取得します.
     * <BR>
     * @return OutputStream バイナリデータに出力する
     *                      OutputStreamが返されます.
     * @exception IOException I/O例外.
     */
    public synchronized OutputStream getOutputStream() throws IOException {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return null ;
        }
        
        return this.outputStream ;
        
    }
    
    /**
     * 文字データをクライアントに送り返すのに
     * 使用するPrintWriterオブジェクトを取得.
     * <BR><BR>
     * 文字データをクライアントに送り返すのに
     * 使用するPrintWriterオブジェクトを取得します.
     * <BR>
     * @return PrintWriter クライアントに文字データを送り返すことが
     *                     できるPrintWriterオブジェクトが返されます.
     * @exception IOException I/O例外.
     */
    public synchronized PrintWriter getWriter() throws IOException {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return null ;
        }
        
        if( this.printWriter == null ) {
            this.printWriter = new PrintWriter(
                new OutputStreamWriter(
                    this.outputStream,this.bean.getCharset()
                ),true
            ) ;
        }
        
        this.bean.setMode( true ) ;
        this.bean.lock() ;
        
        return this.printWriter ;
        
    }
    
    /**
     * データモードを取得.
     * <BR><BR>
     * 確定しているデータモードを取得します.
     * <BR>
     * @return boolean 確定しているデータモードが返されます.<BR>
     *                 [true]が返された場合、文字情報として送られます.<BR>
     *                 [false]が返された場合、バイナリ情報として送られます.
     */
    public synchronized boolean isMode() {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return false ;
        }
        
        return this.bean.isMode() ;
        
    }
    
    /**
     * 正常戻り値を設定.
     * <BR><BR>
     * 正常戻り値を設定します.
     * <BR>
     * @param result 正常戻り値を設定します.
     */
    public synchronized void setSuccess( int result ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        result = ( result & (~JRcResultDef.STATE_MASK) ) | JRcResultDef.SUCCESS ;
        this.bean.setResult( result ) ;
        
    }
    
    /**
     * 警告戻り値を設定.
     * <BR><BR>
     * 警告戻り値を設定します.
     * <BR>
     * @param result 警告戻り値を設定します.
     */
    public synchronized void setWarning( int result ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        result = ( result & (~JRcResultDef.STATE_MASK) ) | JRcResultDef.WARNING ;
        this.bean.setResult( result ) ;
        
    }
    
    /**
     * 異常戻り値を設定.
     * <BR><BR>
     * 異常戻り値を設定します.
     * <BR>
     * @param result 異常戻り値を設定します.
     */
    public synchronized void setError( int result ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        result = ( result & (~JRcResultDef.STATE_MASK) ) | JRcResultDef.ERROR ;
        this.bean.setResult( result ) ;
        
    }
    
    /**
     * 異常戻り値を設定.
     * <BR><BR>
     * 異常戻り値を設定します.
     * <BR>
     * @param result 異常戻り値を設定します.
     * @param message 異常メッセージを設定します.
     */
    public synchronized void setError( int result,String message ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        if( message != null && message.length() > 0 ) {
            this.bean.setResultMessage( message ) ;
        }
        
        this.setError( result ) ;
        
    }
    
    /**
     * 復旧不能異常戻り値を設定.
     * <BR><BR>
     * 復旧不能異常戻り値を設定します.
     * <BR>
     * @param result 復旧不能異常戻り値を設定します.
     */
    public synchronized void setFatalError( int result ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        result = ( result & (~JRcResultDef.STATE_MASK) ) | JRcResultDef.FATAL ;
        this.bean.setResult( result ) ;
        
    }
    
    /**
     * 復旧不能異常戻り値を設定.
     * <BR><BR>
     * 復旧不能異常戻り値を設定します.
     * <BR>
     * @param result 復旧不能異常戻り値を設定します.
     * @param message 復旧不能異常メッセージを設定します.
     */
    public synchronized void setFatalError( int result,String message ) {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return ;
        }
        
        if( message != null && message.length() > 0 ) {
            this.bean.setResultMessage( message ) ;
        }
        
        this.setFatalError( result ) ;
        
    }
    
    /**
     * 設定されている戻り値を取得.
     * <BR><BR>
     * 設定されている戻り値を取得します.
     * <BR>
     * @return int 設定されている戻り値が返されます.
     */
    public synchronized int getResult() {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return -1 ;
        }
        
        return this.bean.getResult() ;
        
    }
    
    /**
     * 設定されている戻りメッセージを取得.
     * <BR><BR>
     * 設定されている戻りメッセージを取得します.
     * <BR>
     * @return String 設定されている戻りメッセージが返されます.
     */
    public synchronized String getResultMessage() {
        
        if( this.bean == null || this.bean.isBean() == false ) {
            return null ;
        }
        
        return this.bean.getResultMessage() ;
        
    }
    
}

