/*
 * @(#)JRcRequestImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.util.Date;

import com.JRcServer.JRcParameter;
import com.JRcServer.JRcParameterType;
import com.JRcServer.JRcRequest;
import com.JRcServer.JRcSession;
import com.JRcServer.commons.conv.Base64;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ConvertException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.DateTimeFormat;

/**
 * JRcServerリクエスト実体.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcRequestImple implements JRcRequest {
    
    /**
     * リクエストに対するレスポンスオブジェクト.
     */
    private JRcResponseImple response = null ;
    
    /**
     * セッションオブジェクト.
     */
    private JRcSession session = null ;
    
    /**
     * リモートIPアドレス.
     */
    private InetAddress remoteAddress = null ;
    
    /**
     * リモートポート番号.
     */
    private int remotePort = -1 ;
    
    /**
     * サーバIPアドレス.
     */
    private InetAddress serverAddress = null ;
    
    /**
     * サーバポート番号.
     */
    private int serverPort = -1 ;
    
    /**
     * コマンド名.
     */
    private String commandName = null ;
    
    /**
     * アプリケーション名.
     */
    private String applicationName = null ;
    
    /**
     * パラメータ管理.
     */
    private CharTable table = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    private JRcRequestImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを設定します.
     * <BR>
     * @param response このリクエストに対するレスポンスオブジェクトを
     *                 設定します.
     * @param applicationName このリクエストを送ってきたクライアント
     *                        アプリケーション名を設定します.
     * @param session セッションオブジェクトを設定します.
     * @param remoteAddr リモートアドレスを設定します.
     * @param remotePort リモートポートを設定します.
     * @param serverAddr サーバアドレスを設定します.
     * @param serverPort サーバポートを設定します.
     * @param commandName コマンド名を設定します.
     * @param table パラメータ内容を設定します.
     * @exception InputException 入力例外.
     */
    public JRcRequestImple(
        JRcResponseImple response,
        String applicationName,
        JRcSessionImple session,
        InetAddress remoteAddr,int remotePort,
        InetAddress serverAddr,int serverPort,
        String commandName,CharTable table )
        throws InputException {
        
        if(
            response == null ||
            applicationName == null ||
            ( applicationName = applicationName.trim().toLowerCase() ).length() <= 0 ||
            remoteAddr == null ||
            remotePort < 0 || remotePort > 65535 ||
            serverAddr == null ||
            serverPort < 0 || serverPort > 65535 ||
            commandName == null || commandName.length() <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.response = response ;
        this.applicationName = applicationName ;
        this.remoteAddress = remoteAddr ;
        this.remotePort = remotePort ;
        this.serverAddress = serverAddr ;
        this.serverPort = serverPort ;
        this.commandName = commandName ;
        
        if(
            session == null ||
            session.isSession() == false ||
            session.getId() == -1L
        ) {
            this.session = null ;
            response.setSessionID( -1L ) ;
        }
        else {
            this.session = session ;
            response.setSessionID( session.getId() ) ;
        }
        
        if( table == null || table.size() <= 0 ) {
            this.table = null ;
        }
        else {
            this.table = table ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * 対象のオブジェクトを破棄します.
     */
    public synchronized void destroy() {
        
        this.response = null ;
        this.applicationName = null ;
        this.remoteAddress = null ;
        this.remotePort = -1 ;
        this.serverAddress = null ;
        this.serverPort = -1 ;
        this.commandName = null ;
        this.session = null ;
        this.table = null ;
        
    }
    
    
    /**
     * リモートIPアドレスを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントのIPアドレスを取得します.
     * <BR>
     * @return InetAddress リモートIPアドレスが返されます.
     */
    public synchronized InetAddress getRemoteInetAddress() {
        return remoteAddress ;
    }
    
    /**
     * リモートポート番号.
     * <BR><BR>
     * リクエストを送ってきたクライアントのIPアドレスを取得します.
     * <BR>
     * @return int リモートポート番号が返されます.
     */
    public synchronized int getRemotePort() {
        return remotePort ;
    }
    
    /**
     * サーバIPアドレスを取得.
     * <BR><BR>
     * このサーバのIPアドレスを取得します.
     * <BR>
     * @return InetAddress サーバIPアドレスが返されます.
     */
    public synchronized InetAddress getInetAddress() {
        return serverAddress ;
    }
    
    /**
     * サーバポート番号を取得.
     * <BR><BR>
     * このサーバのポート番号を取得します.
     * <BR>
     * @return int サーバポート番号が返されます.
     */
    public synchronized int getPort() {
        return serverPort ;
    }
    
    /**
     * コマンド名を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのコマンド名を取得します.
     * <BR>
     * @return String コマンド名が返されます.
     */
    public synchronized String getCommandName() {
        return commandName ;
    }
    
    /**
     * パラメータを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return JRcParameter パラメータ情報が返されます.
     */
    public synchronized JRcParameter getParameter( String key ) {
        
        if( table != null ) {
            return ( JRcParameter )table.get( convertKey( key ) ) ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータをバイナリで取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return byte[] パラメータ情報が返されます.
     */
    public synchronized byte[] getParameterToBinary( String key ) {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getBinary() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return String パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized String getParameterToString( String key )
        throws ConvertException {
        
        int type ;
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            type = param.getType() ;
            
            try {
                switch( type ){
                    case JRcParameterType.TYPE_BINARY :
                        return Base64.encode( param.getBinary() ) ;
                    case JRcParameterType.TYPE_STRING :
                        return param.getString() ;
                    case JRcParameterType.TYPE_BOOLEAN :
                        return param.getBoolean().toString() ;
                    case JRcParameterType.TYPE_SHORT :
                        return param.getShort().toString() ;
                    case JRcParameterType.TYPE_INTEGER :
                        return param.getInteger().toString() ;
                    case JRcParameterType.TYPE_LONG :
                        return param.getLong().toString() ;
                    case JRcParameterType.TYPE_FLOAT :
                        return param.getFloat().toString() ;
                    case JRcParameterType.TYPE_DOUBLE :
                        return param.getDouble().toString() ;
                    case JRcParameterType.TYPE_DECIMAL :
                        return param.getDecimal().toString() ;
                    case JRcParameterType.TYPE_TIMESTAMP :
                        DateTimeFormat fmt = new DateTimeFormat( "YYYY/MM/DD hh:mm:ss.SSSS" ) ;
                        return fmt.getString( param.getLong().longValue() ) ;
                }
            } catch( ConvertException ce ) {
                throw ce ;
            } catch( Exception e ) {
                throw new ConvertException( e ) ;
            }
        }
        
        return null ;
        
    }
    
    /**
     * パラメータをフラグで取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Boolean パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Boolean getParameterToBoolean( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getBoolean() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを数値(Short)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Short パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Short getParameterToShort( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getShort() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを数値(Integer)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Integer パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Integer getParameterToInteger( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getInteger() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを数値(Long)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Long パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Long getParameterToLong( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getLong() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを浮動少数値(Float)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Float パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Float getParameterToFloat( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getFloat() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを浮動少数値(Double)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Double パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Double getParameterToDouble( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getDouble() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを浮動少数値(Decimal)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return BigDecimal パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized BigDecimal getParameterToDecimal( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getDecimal() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Date パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Date getParameterToTimestamp( String key )
        throws ConvertException {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getTimestamp() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータタイプを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータタイプが返されます.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return int パラメータタイプが返されます.<BR>
     *             [JRcParameterType#TYPE_BINARY]が返された場合(byte[])情報です.<BR>
     *             [JRcParameterType#TYPE_STRING]が返された場合(String)情報です.<BR>
     *             [JRcParameterType#TYPE_BOOLEAN]が返された場合(Boolean)情報です.<BR>
     *             [JRcParameterType#TYPE_SHORT]が返された場合(Short)情報です.<BR>
     *             [JRcParameterType#TYPE_INTEGER]が返された場合(Integer)情報です.<BR>
     *             [JRcParameterType#TYPE_LONG]が返された場合(Long)情報です.<BR>
     *             [JRcParameterType#TYPE_FLOAT]が返された場合(Float)情報です.<BR>
     *             [JRcParameterType#TYPE_DOUBLE]が返された場合(Double)情報です.<BR>
     *             [JRcParameterType#TYPE_DECIMAL]が返された場合(BigDecimal)情報です.<BR>
     *             [JRcParameterType#TYPE_TIMESTAMP]が返された場合(Date)情報です.<BR>
     *             パラメータタイプが定義されていない場合かキー内容が存在しない場合、
     *             [-1]が返されます.
     */
    public synchronized int getParameterType( String key ) {
        
        JRcParameter param = null ;
        
        if( ( param = this.getParameter( convertKey( key ) ) ) != null ) {
            return param.getType() ;
        }
        
        return -1 ;
        
    }
    
    /**
     * パラメータキー名一覧を取得.
     * <BR><BR>
     * パラメータキー名一覧を取得します.
     * <BR>
     * @return String[] パラメータキー名一覧が返されます.
     */
    public synchronized String[] getParameterKeys() {
        
        if( table != null ) {
            return table.getNames() ;
        }
        
        return null ;
        
    }
    
    /**
     * パラメータ数を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ数が返されます.
     * <BR>
     * @return int 設定されているパラメータ数が返されます.
     */
    public synchronized int getParameterSize() {
        
        if( table != null ) {
            return table.size() ;
        }
        
        return 0 ;
        
    }
    
    /**
     * パラメータ名が存在するかチェック.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ名が
     * 存在するかチェックします.
     * <BR>
     * @param key チェック対象Key名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象Key名はパラメータに存在します.<BR>
     *                 [false]が返された場合、対象Key名はパラメータに存在しません.
     */
    public synchronized boolean isParameterName( String key ) {
        
        if( table != null ) {
            return table.isData( key ) ;
        }
        
        return false ;
        
    }
    
    /**
     * アプリケーション名を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのアプリケーション名を取得します.
     * <BR>
     * @return String アプリケーション名が返されます.
     */
    public synchronized String getApplicationName() {
        
        return applicationName ;
        
    }
    
    /**
     * セッション情報を取得.
     * <BR><BR>
     * このリクエストに関連付けられている現在のセッション情報を取得します.
     * <BR>
     * @return JRcSession セッション情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public synchronized JRcSession getSession()
        throws AccessException {
        
        long id ;
        
        JRcSessionImple imple = null ;
        JRcBaseManager man = null ;
        JRcSessionManagerImple sman = null ;
        
        if( session == null ) {
            
            if( ( man = JRcManagerFactory.getJRcBaseManager() ) != null ) {
                if(
                    ( sman = ( JRcSessionManagerImple )man.getSessionManager() ) == null ||
                    sman.isSessionManager() == false
                ) {
                    throw new AccessException( "セッションマネージャが不正です" ) ;
                }
                
                imple = ( JRcSessionImple )sman.createSession( applicationName ) ;
                
                if( imple == null || imple.isSession() == false ) {
                    response.setSessionID( -1L ) ;
                    throw new AccessException( "セッション情報の取得に失敗しました" ) ;
                }
            }
            else {
                response.setSessionID( -1L ) ;
                throw new AccessException(
                    "JRcServerは不正です[" +
                    JRcManagerFactory.class.getName() +
                    "が無効]" ) ;
            }
            
            if( imple != null ) {
                id = imple.getId() ;
                response.setSessionID( id ) ;
            }
            
            session = imple ;
            
        }
        
        return session ;
        
    }
    
    /**
     * 指定キー名を小文字変換.
     */
    private static final String convertKey( String key ) {
        if(
            key == null ||
            ( key = key.trim().toLowerCase() ).length() <= 0
        ) {
            return null ;
        }
        
        return key ;
    }
}

