/*
 * @(#)JRcManagerFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import com.JRcServer.JRcBaseDefineBean;
import com.JRcServer.commons.beans.BaseBean;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.sys.NamingManager;

/**
 * JRcServerマネージャファクトリ.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
public class JRcManagerFactory {
    
    /**
     * JRcServerAPIバージョン.
     */
    public static final String VERSION = "JRcServerAPI-1.00" ;
    
    /**
     * JRcServerネーミング登録名 : 基本マネージャ.
     */
    public static final String JRC_BASE_MANAGER = "jrcserver@base.manager.factory" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * 初期化フラグオブジェクト.
     */
    private static volatile boolean INIT_FLAG = false ;
    
    
    /**
     * 基本マネージャを登録.
     * <BR><BR>
     * 基本マネージャを登録します.
     * <BR>
     * @param bean 登録対象の基本マネージャBeanを設定します.
     * @exception AccessException アクセス例外.
     */
    protected static final void initJRcBeanManager( BaseBean bean )
        throws AccessException {
        
        JRcBaseManagerImple imple = null ;
        
        if(
            bean == null ||
            ( bean instanceof JRcBaseDefineBean ) == false ||
            ( ( JRcBaseDefineBean )bean ).getBaseDirectory() == null ||
            ( ( JRcBaseDefineBean )bean ).getBaseDirectory().length() <= 0
        ) {
            throw new AccessException( "指定された起動Beanは不正です" ) ;
        }
        
        synchronized( SYNC ) {
            
            if( INIT_FLAG == false ) {
                
                imple = ( JRcBaseManagerImple )NamingManager.get(
                    JRC_BASE_MANAGER
                ) ;
                
                if( imple != null ) {
                    imple.destroy() ;
                }
                
                imple = new JRcBaseManagerImple() ;
                imple.init( bean ) ;
                
                NamingManager.add( JRC_BASE_MANAGER,imple ) ;
                
                INIT_FLAG = true ;
                
            }
            
        }
        
    }
    
    /**
     * 基本マネージャを破棄.
     * <BR><BR>
     * 基本マネージャを破棄します.
     */
    protected static final void destroyJRcBeanManager() {
        
        JRcBaseManagerImple imple = null ;
        
        synchronized( SYNC ) {
            
            if( INIT_FLAG == true ) {
                
                imple = ( JRcBaseManagerImple )NamingManager.get(
                    JRC_BASE_MANAGER ) ;
                
                if( imple != null ) {
                    
                    NamingManager.remove( JRC_BASE_MANAGER ) ;
                    imple.destroy() ;
                    
                }
                
                INIT_FLAG = false ;
                
            }
            
        }
        
    }
    
    /**
     * 基本マネージャを取得.
     * <BR><BR>
     * 基本マネージャを取得します.
     * <BR>
     * @return JRcBaseManager 基本マネージャが返されます.
     */
    public static final JRcBaseManager getJRcBaseManager() {
        
        JRcBaseManager ret = null ;
        
        synchronized( SYNC ) {
            
            ret = ( JRcBaseManager )NamingManager.get(
                JRC_BASE_MANAGER ) ;
                
        }
        
        return ret ;
        
    }
    
    /**
     * 基本マネージャ初期化状態取得.
     * <BR><BR>
     * 基本マネージャ初期化状態を取得します.
     * <BR>
     * @return boolean 基本マネージャ初期化状態が返されます.
     */
    public static final boolean isJRcBaseManager() {
        
        boolean ret = false ;
        
        synchronized( SYNC ) {
            ret = INIT_FLAG ;
        }
        
        return ret ;
        
    }
    
    /**
     * JRcServerAPIバージョンを取得.
     * <BR><BR>
     * JRcServerAPIバージョンを取得します.
     * <BR>
     * @return String JRcServerAPIバージョン情報が返されます.
     */
    public static final String getVersion() {
        
        return VERSION ;
        
    }
    
    /**
     * コンストラクタ.
     */
    private JRcManagerFactory() {
        
    }
}

