/*
 * @(#)JRcConnectServerImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.JRcConnectCommon;
import com.JRcServer.JRcService;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.IdManager;
import com.JRcServer.commons.util.NumberTable;

/**
 * JRcServerサーバネットワークオブジェクト.
 * <BR><BR>
 * JRcServerネットワークサーバオブジェクトです.
 *  
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcConnectServerImple
    extends ExecutionThread
    implements JRcConnectServer {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( JRcConnectServerImple.class ) ;
    
    /**
     * アクセプト待ちタイムアウト.
     */
    private static final int ACCEPT_TIMEOUT = 50 ;
    
    /**
     * 基本バッファサイズ.
     */
    private static final int BASE_BUFFER = JRcConnectCommon.BASE_BUFFER ;
    
    /**
     * JRcServerサービス.
     */
    private JRcService service = null ;
    
    /**
     * コネクション制限数.
     */
    private int maxConnectCount = -1 ;
    
    /**
     * サーバソケット.
     */
    private ServerSocket serverSocket = null ;
    
    /**
     * コネクションタイムアウト.
     */
    private int timeout = -1 ;
    
    /**
     * コネクションネットワーク管理テーブル.
     */
    private NumberTable table = null ;
    
    /**
     * ソケット管理テーブル.
     */
    private IdManager idman = null ;
    
    /**
     * ループスレッド.
     */
    private LoopThread threadObject = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    private JRcConnectServerImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを設定します.
     * <BR>
     * @param service JRcServiceを設定します.
     * @param port バインド先のポート番号を設定します.
     * @param timeout コネクションタイムアウト値を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public JRcConnectServerImple( JRcService service,int port,int timeout )
        throws InputException,AccessException {
        
        this( service,port,null,timeout,-1 ) ;
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを設定します.
     * <BR>
     * @param service JRcServiceを設定します.
     * @param port バインド先のポート番号を設定します.
     * @param timeout コネクションタイムアウト値を設定します.
     * @param maxConnect コネクション制限値を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public JRcConnectServerImple( JRcService service,int port,int timeout,int maxConnect )
        throws InputException,AccessException {
        
        this( service,port,null,timeout,maxConnect ) ;
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを設定します.
     * <BR>
     * @param service JRcServiceを設定します.
     * @param port バインド先のポート番号を設定します.
     * @param bindAddress バインド先のアドレスを設定します.
     * @param timeout コネクションタイムアウト値を設定します.
     * @param maxConnect コネクション制限値を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public JRcConnectServerImple( JRcService service,int port,InetAddress bindAddress,int timeout,int maxConnect )
        throws InputException,AccessException {
        
        ServerSocket ss = null ;
        
        if(
            service == null ||
            port < 0 || port > 65535
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        maxConnect = ( maxConnect <= 0 ) ? -1 : maxConnect ;
        timeout = ( timeout <= 0 ) ? -1 : timeout ;
        sync.create() ;
        
        try {
            
            ss = new ServerSocket() ;
            
            ss.setReuseAddress( true ) ;
            ss.setSoTimeout( ACCEPT_TIMEOUT ) ;
            ss.setReceiveBufferSize( BASE_BUFFER ) ;
            
            if( bindAddress == null ) {
                ss.bind( new InetSocketAddress( port ),maxConnect ) ;
            }
            else {
                ss.bind( new InetSocketAddress( bindAddress,port ),maxConnect ) ;
            }
            
            this.service = service ;
            this.maxConnectCount = maxConnect ;
            this.serverSocket = ss ;
            this.timeout = timeout ;
            this.table = new NumberTable() ;
            this.idman = new IdManager() ;
            this.threadObject = new LoopThread() ;
            this.threadObject.create( this ) ;
            this.threadObject.startThread() ;
            
        } catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクトを破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        int i ;
        int len ;
        int[] nums = null ;
        JRcConnect net = null ;
        
        try {
            synchronized( sync.get() ) {
                
                try {
                    this.serverSocket.close() ;
                } catch( Exception e ) {
                }
                
                this.serverSocket = null ;
                
                try {
                    this.threadObject.clear() ;
                } catch( Exception e ) {
                }
                
                this.threadObject = null ;
                
                if( this.table != null ) {
                    nums = this.table.getNumbers() ;
                    if( nums != null && ( len = nums.length ) > 0 ) {
                        for( i = 0 ; i < len ; i ++ ) {
                            
                            net = ( JRcConnect )this.table.get( nums[ i ] ) ;
                            if( net != null ) {
                                net.destroy() ;
                            }
                            
                        }
                        this.table.clear() ;
                    }
                }
                
                if( this.idman != null ) {
                    this.idman.clear() ;
                }
                
                this.idman = null ;
                
                this.service = null ;
                this.maxConnectCount = -1 ;
                this.timeout = -1 ;
                
            }
            
        } catch( Exception e ) {
        }
        
        this.service = null ;
        this.maxConnectCount = -1 ;
        this.serverSocket = null ;
        this.timeout = -1 ;
        this.table = null ;
        this.idman = null ;
        this.threadObject = null ;
        
        sync.clear() ;
        
    }
    
    /**
     * 新しいIDを発行.
     * <BR><BR>
     * 新しいIDを発行します.
     * <BR>
     * @return int 新しいID情報が発行されます.
     */
    protected int getID() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                
                ret = this.idman.getID() ;
                
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定IDでソケット管理情報に追加.
     * <BR><BR>
     * 指定IDでソケット管理情報に追加します.
     * <BR>
     * @param id 対象のIDを設定します.
     * @param connect 対象のコネクションオブジェクトを設定します.
     */
    protected void addConnect( int id,JRcConnect connect ) {
        
        if( id < 0 || connect == null ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                this.table.add( id,connect ) ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 指定IDのソケット管理情報を削除.
     * <BR><BR>
     * 指定IDのソケット管理情報を削除します.
     * <BR>
     * @param id 削除対象のソケット情報を削除します.
     */
    protected void removeConnect( int id ) {
        
        try {
            synchronized( sync.get() ) {
                
                try {
                    this.table.remove( id ) ;
                } catch( Exception ee ) {
                }
                
                try {
                    this.idman.removeID( id ) ;
                } catch( Exception ee ) {
                }
                
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * サービスを取得.
     * <BR><BR>
     * サービスオブジェクトが返されます.
     * <BR>
     * @return JRcService サービスオブジェクトが返されます.
     */
    protected JRcService getService() {
        
        JRcService ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.service ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在コネクション中の数を取得.
     * <BR><BR>
     * 現在コネクション中の数が返されます.
     * <BR>
     * @return int 現在コネクション中の数が返されます.
     */
    public int getConnectionCount() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.table.size() ;
            }
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * バインド先アドレスを取得.
     * <BR><BR>
     * バインド先アドレスを取得します.
     * <BR>
     * @return InetAddress バインド先アドレスが返されます.
     */
    public InetAddress getBindAddress() {
        
        InetAddress ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.serverSocket.getInetAddress() ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バインド先ポート番号を取得.
     * <BR><BR>
     * バインド先ポート番号を取得します.
     * <BR>
     * @return int バインド先ポート番号が返されます.
     */
    public int getBindPort() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.serverSocket.getLocalPort() ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * コネクション制限数を取得.
     * <BR><BR>
     * コネクション制限数を取得します.
     * <BR>
     * @return int コネクション制限数が返されます.
     */
    public int getMaxConnectCount() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.maxConnectCount ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * コネクションタイムアウト値を取得.
     * <BR><BR>
     * コネクションタイムアウト値を取得します.
     * <BR>
     * @return int コネクションタイムアウト値が返されます.
     */
    public int getConnectTimeout() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.timeout ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトが返されます.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    protected Synchronized getSync() {
        return sync ;
    }
    
    /**
     * サーバオブジェクトが有効であるかチェック.
     * <BR><BR>
     * サーバオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isServer() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.serverSocket.isClosed() ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外.
     */
    protected void execution( Object obj )
        throws ExecutionException
    {
        int timeout ;
        
        Socket socket = null ;
        
        try{
            
            synchronized( sync.get() ) {
                
                // コネクションタイムアウト値を取得.
                timeout = this.timeout ;
                
                // アクセプト.
                try {
                    socket = this.serverSocket.accept() ;
                } catch( SocketTimeoutException st ) {
                    socket = null ;
                } catch( Exception e ) {
                    socket = null ;
                    throw new ExecutionException(
                        e,ExecutionException.LEVEL_STOP
                    ) ;
                }
                
            }
            
            // ソケットが存在しない場合.
            if( socket != null ) {
                
                // 新しいソケットを作成して登録.
                JRcConnectCommon.setSocketOption( socket ) ;
                new JRcConnect( this,socket,timeout ) ;
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( Exception e ){
            LOG.error( "JRcConnectServer - error.",e ) ;
        }finally{
            socket = null ;
        }
        
    }
    
}

