/*
 * @(#)JRcConnect.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.Socket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.JRCommand;
import com.JRcServer.JRCommandException;
import com.JRcServer.JRcConnectCommon;
import com.JRcServer.JRcErrorCodeDef;
import com.JRcServer.JRcParameter;
import com.JRcServer.JRcProtocolDef;
import com.JRcServer.JRcResponseBean;
import com.JRcServer.JRcService;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.BinResourceOutputStream;
import com.JRcServer.commons.resource.BufferedBinResource;
import com.JRcServer.commons.resource.ConvertResourceParam;
import com.JRcServer.commons.resource.Resource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.ConvertParam;

/**
 * JRcServerネットワーク.
 *  
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcConnect extends ExecutionThread {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( JRcConnect.class ) ;
    
    /**
     * 送信バッファ長.
     */
    private static final int SEND_BUFFER = JRcConnectCommon.BUFFER_LENGTH ;
    
    /**
     * ソケットID.
     */
    private int id = -1 ;
    
    /**
     * 受信タイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * ソケット.
     */
    private Socket socket = null ;
    
    /**
     * サーバネット.
     */
    private JRcConnectServerImple server = null ;
    
    /**
     * ループスレッド.
     */
    private LoopThread threadObject = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    private JRcConnect() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ソケットオブジェクトを設定してコネクションネットワークを生成します.
     * <BR>
     * @param server サーバネットを設定します.
     * @param socket 対象のソケットを設定します.
     * @param timeout 受信タイムアウト値を設定します.
     * @exception AccessException アクセス例外.
     */
    public JRcConnect( JRcConnectServerImple server,Socket socket,int timeout )
        throws AccessException {
        
        int id ;
        boolean err = false ;
        
        sync.create() ;
        
        try {
            
            synchronized( server.getSync().get() ) {
                
                if( ( id = server.getID() ) == -1 ) {
                    throw new AccessException(
                        "ソケットID項番の発行に失敗しました[" +
                        server + "]" ) ;
                }
                
                this.id = id ;
                this.timeout = timeout ;
                this.socket = socket ;
                this.server = server ;
                
                server.addConnect( id,this ) ;
                
            }
            
            this.threadObject = new LoopThread() ;
            this.threadObject.create( this ) ;
            this.threadObject.startThread() ;
            
            LOG.info(
                new StringBuffer().
                append( "### コネクション[addr:" ).
                append( socket.getInetAddress().getHostAddress() ).
                append( " port:" ).
                append( socket.getPort() ).
                append( "]が確立しました" ).
                toString()
            ) ;
            
        } catch( AccessException ae ) {
            err = true ;
            throw ae ;
        } catch( Exception e ) {
            err = true ;
            throw new AccessException( e ) ;
        } finally {
            
            if( err == true ) {
                try {
                    socket.close() ;
                } catch( Exception e ) {
                }
                this.destroy() ;
            }
            
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクトを破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        try {
            synchronized( sync.get() ) {
                
                try {
                    this.socket.close() ;
                } catch( Exception e ) {
                }
                
                this.socket = null ;
                
                try {
                    this.threadObject.clear() ;
                } catch( Exception e ) {
                }
                
                this.threadObject = null ;
                
                if( this.server != null ) {
                    if( id != -1 ) {
                        
                        this.server.removeConnect( this.id ) ;
                        
                    }
                }
                
                this.id = -1 ;
                this.timeout = -1 ;
                this.socket = null ;
                this.server = null ;
                this.threadObject = null ;
                
            }
            
        } catch( Exception e ) {
        }
        
        this.id = -1 ;
        this.timeout = -1 ;
        this.socket = null ;
        this.server = null ;
        this.threadObject = null ;
        
        sync.clear() ;
        
    }
    
    /**
     * ソケットをクローズ.
     * <BR><BR>
     * ソケット情報をクローズします.
     */
    public void socketClose() {
        
        try {
            synchronized( sync.get() ) {
                this.socket.shutdownInput() ;
                this.socket.shutdownOutput() ;
                this.socket.close() ;
            }
            this.socket = null ;
        } catch( Exception ee ) {
        }
        
    }
    
    /**
     * ソケットIDを取得.
     * <BR><BR>
     * ソケットIDを取得します.
     * <BR>
     * @return int ソケットIDが返されます.
     */
    public int getSocketID() {
        
        int ret ;
        
        try {
            synchronized( sync.get() ) {
                
                ret = this.id ;
                
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * スレッドが有効であるかチェック.
     * <BR><BR>
     * スレッドが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isThread() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = this.socket.isClosed() ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外.
     */
    protected void execution( Object obj )
        throws ExecutionException
    {
        int timeout ;
        int port = -1 ;
        int sPort = -1 ;
        boolean exitFlg = false ;
        boolean timeoutFlg = false ;
        
        JRcService service = null ;
        JRcRequestImple request = null ;
        JRcResponseImple response = null ;
        JRcResponseBean bean = null ;
        BinResource bin = null ;
        JRcCache cache = null ;
        ResourceType resType = null ;
        InputStream inputStream = null ;
        OutputStream outputStream = null ;
        InetAddress addr = null ;
        InetAddress sAddr = null ;
        
        try {
            
            // リソースタイプを取得.
            if( ( cache = JRcManagerFactory.getJRcBaseManager().getCache() ) == null ) {
                resType = null ;
            }
            else {
                resType = cache.getResourceType() ;
            }
            
            // 初期処理.
            synchronized( sync.get() ) {
                
                if( this.socket.isClosed() == true ) {
                    throw new ExecutionException(
                        "コネクションは切断されています",
                        ExecutionException.LEVEL_STOP
                    ) ;
                }
                
                timeout = this.timeout ;
                
                inputStream = new BufferedInputStream(
                    this.socket.getInputStream()
                ) ;
                //inputStream = this.socket.getInputStream() ;
                
            }
            
            // データ受信処理.
            bin = JRcConnectCommon.receive( false,resType,sync,inputStream,timeout ) ;
            if( bin == null ) {
                
                throw new ExecutionException(
                    "ソケットはクローズされました",
                    ExecutionException.LEVEL_STOP
                ) ;
                
            }
            
            inputStream = null ;
            
            // 受信後初期化処理.
            synchronized( sync.get() ) {
                
                service = this.server.getService() ;
                
                addr = this.socket.getInetAddress() ;
                port = this.socket.getPort() ;
                sAddr = this.socket.getLocalAddress() ;
                sPort = this.socket.getLocalPort() ;
                
            }
            
            try {
                
                // レスポンスを生成.
                bean = new JRcResponseBean() ;
                bean.setBinary( Resource.createBinResource( resType,1 ) ) ;
                response = new JRcResponseImple( bean ) ;
                
                // 受信データ解析.
                request = this.getAnalysis( bin,response,addr,port,sAddr,sPort ) ;
                bin = null ;
                
                // サービス呼び出し処理.
                this.executionService( service,request,response ) ;
                response.flush() ;
                request = null ;
                response = null ;
                
                // 送信用OutputStreamを生成.
                try {
                    synchronized( sync.get() ) {
                        if( this.socket.isClosed() == false ) {
                            outputStream = new BufferedOutputStream(
                                this.socket.getOutputStream()
                            ) ;
                            //outputStream = this.socket.getOutputStream() ;
                        }
                        else {
                            outputStream = null ;
                        }
                    }
                    
                    if( outputStream != null ) {
                        // 送信処理.
                        this.sendData( resType,bean,outputStream,addr,port ) ;
                        outputStream.flush() ;
                        outputStream = null ;
                    }
                    
                } catch( Exception ee ) {
                }
                
            } catch ( JRCommandException ce ) {
                
                try {
                    // 送信用OutputStreamを生成.
                    synchronized( sync.get() ) {
                        if( this.socket.isClosed() == false ) {
                            outputStream = new BufferedOutputStream(
                                this.socket.getOutputStream()
                            ) ;
                            //outputStream = this.socket.getOutputStream() ;
                        }
                        else {
                            outputStream = null ;
                        }
                    }
                    
                    if( outputStream != null ) {
                        // エラー内容を送信.
                        this.sendErrorData( resType,ce,outputStream,addr,port ) ;
                        outputStream.flush() ;
                        outputStream = null ;
                    }
                    
                } catch( Exception ee ) {
                }
                
            }
            
        } catch( ConnectTimeoutException tm ) {
            exitFlg = true ;
            timeoutFlg = true ;
            throw new ExecutionException(
                tm,ExecutionException.LEVEL_STOP
            ) ;
        } catch( NullPointerException nul ) {
            exitFlg = true ;
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        } catch( ExecutionException ee ) {
            exitFlg = true ;
            throw ee ;
        } catch( Exception e ){
            LOG.error( "JRcConnect - error.",e ) ;
        } finally {
            
            // コネクション破棄の場合.
            if( exitFlg == true ) {
                
                if( timeoutFlg == true ) {
                    LOG.warn(
                        "タイムアウト例外によりコネクションを破棄します[addr:" +
                        addr.getHostAddress() + " port:" +
                        port + "]" ) ;
                }
                else {
                    LOG.error(
                        "接続先から切断されました[addr:" +
                        addr.getHostAddress() + " port:" +
                        port + "]" ) ;
                }
                
                try {
                    synchronized( sync.get() ) {
                        this.server.removeConnect( this.id ) ;
                    }
                } catch( Exception ee ) {
                }
                
                this.socketClose() ;
                
            }
            
            service = null ;
            request = null ;
            response = null ;
            bean = null ;
            bin = null ;
            cache = null ;
            resType = null ;
            inputStream = null ;
            outputStream = null ;
            addr = null ;
            
        }
        
    }
    
    /**
     * 受信データを解析.
     */
    private final JRcRequestImple getAnalysis(
        BinResource bin,JRcResponseImple response,
        InetAddress remoteAddr,int remotePort,
        InetAddress serverAddr,int serverPort
    ) throws JRCommandException {
        
        int i ;
        int len ;
        int paramType ;
        int paramLen ;
        int pnt ;
        
        long sessionID = -1L ;
        
        byte[] hd = null ;
        byte[] paramBin = null ;
        JRcSessionImple session = null ;
        JRcSessionManager sessionMan = null ;
        JRcParameter param = null ;
        String applicationName = null ;
        String commandName = null ;
        String paramName = null ;
        CharTable parameters = null ;
        JRcRequestImple ret = null ;
        
        try {
            
            // 最低プロトコル長チェック.
            if( bin.size() <= JRcProtocolDef.MIN_RCV_PROTOCOL_LENGTH ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                    "受信プロトコルは不正です(" + bin.size() + ")"
                ) ;
            }
            
            hd = bin.getBinary( 0,JRcProtocolDef.RCV_HEADER_LENGTH ) ;
            
            // プロトコルチェック.
            if(
                hd[ 0 ] != JRcProtocolDef.RCV_HEADER[ 0 ] ||
                hd[ 1 ] != JRcProtocolDef.RCV_HEADER[ 1 ] ||
                hd[ 2 ] != JRcProtocolDef.RCV_HEADER[ 2 ] ||
                hd[ 3 ] != JRcProtocolDef.RCV_HEADER[ 3 ] ||
                hd[ 4 ] != JRcProtocolDef.RCV_HEADER[ 4 ] ||
                hd[ 5 ] != JRcProtocolDef.RCV_HEADER[ 5 ]
            ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                    "受信プロトコルヘッダは不正です"
                ) ;
            }
            
            hd = null ;
            pnt = JRcProtocolDef.RCV_HEADER_LENGTH + 4 ;
            
            // セッションID.
            sessionID = ConvertResourceParam.convertLong( pnt,bin ) ;
            pnt += 8 ;
            
            // アプリケーション名長.
            len = ConvertResourceParam.convertInt( pnt,bin ) ;
            pnt += 4 ;
            
            if( len <= 0 ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_APPLICATION_NAME,
                    "受信データにアプリケーション名は存在しません"
                ) ;
            }
            else if( len + pnt > bin.size() ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                    "受信のアプリケーション名は不正です"
                ) ;
            }
            
            // アプリケーション名
            applicationName = ConvertResourceParam.convertString(
                pnt,len,bin,BaseDef.UTF8 ) ;
            pnt += len ;
            
            // コマンド名長.
            len = ConvertResourceParam.convertInt( pnt,bin ) ;
            pnt += 4 ;
            
            if( len <= 0 ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_APPLICATION_NAME,
                    "受信データにコマンド名は存在しません"
                ) ;
            }
            else if( len + pnt > bin.size() ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                    "受信のコマンド名は不正です"
                ) ;
            }
            
            // コマンド名
            commandName = ConvertResourceParam.convertString(
                pnt,len,bin,BaseDef.UTF8 ) ;
            pnt += len ;
            
            // パラメータ数を取得.
            len = ConvertResourceParam.convertInt( pnt,bin ) ;
            pnt += 4 ;
            
            // パラメータが存在する場合.
            if( len > 0 ) {
                
                // パラメータ受け皿の情報を生成.
                parameters = new CharTable() ;
                    
                // パラメータ数に対するパラメータを取得.
                for( i = 0 ; i < len ; i ++ ) {
                    
                    // パラメータ名長を取得.
                    paramLen = ConvertResourceParam.convertInt( pnt,bin ) ;
                    pnt += 4 ;
                    
                    // パラメータ名が存在しない場合は、読み飛ばす.
                    if( paramLen <= 0 ) {
                        pnt += 4 + 1 ;
                        continue ;
                    }
                    else if( paramLen + pnt > bin.size() ) {
                        throw new JRCommandException(
                            JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                            "受信パラメータ名(" + (i+1) + ")は不正です"
                        ) ;
                    }
                    
                    // パラメータ名を取得.
                    paramName = ConvertResourceParam.convertString(
                        pnt,paramLen,bin,BaseDef.UTF8 ) ;
                    pnt += paramLen ;
                    
                    // パラメータタイプを取得.
                    paramType = ( int )( bin.get( pnt ) & 0x000000ff ) ;
                    pnt += 1 ;
                    
                    // パラメータデータ長を取得.
                    paramLen = ConvertResourceParam.convertInt( pnt,bin ) ;
                    pnt += 4 ;
                    
                    if( paramLen <= 0 ) {
                        throw new JRCommandException(
                            JRcErrorCodeDef.ERROR_NOT_PARAMETER_DATA,
                            "受信パラメータデータ長(no:" + (i+1) +
                            " name:" + paramName + ")は不正です"
                        ) ;
                    }
                    else if( paramLen + pnt > bin.size() ) {
                        throw new JRCommandException(
                            JRcErrorCodeDef.ERROR_NOT_PROTOCOL,
                            "受信パラメータデータ(no:" + (i+1) +
                            " name:" + paramName + ")は不正です"
                        ) ;
                    }
                    
                    // パラメータデータを取得.
                    paramBin = bin.getBinary( pnt,paramLen ) ;
                    pnt += paramLen ;
                    
                    // パラメータをセット.
                    param = new JRcParameter( paramType,paramBin ) ;
                    
                    // パラメータ受け皿にパラメータをセット.
                    parameters.add( paramName,param ) ;
                    
                    param = null ;
                    paramName = null ;
                    paramBin = null ;
                    
                }
                
            }
            
            // セッションオブジェクトを取得.
            sessionMan = JRcManagerFactory.getJRcBaseManager().getSessionManager() ;
            session = ( JRcSessionImple )sessionMan.getSession( applicationName,sessionID ) ;
            
            // 情報を生成.
            ret = new JRcRequestImple(
                response,applicationName,session,
                remoteAddr,remotePort,
                serverAddr,serverPort,
                commandName,parameters
            ) ;
            
        } catch( JRCommandException jc ) {
            
            this.shutdownSocket() ;
            
            LOG.error( "受信データは不正です[addr:" +
                remoteAddr.getHostAddress() + " port:" +
                remotePort + "]",jc ) ;
            
            throw jc ;
            
        } catch( Exception e ) {
            
            this.shutdownSocket() ;
            
            LOG.error( "受信データ解析処理中にエラーが発生しました[addr:" +
                remoteAddr.getHostAddress() + " port:" +
                remotePort + "]",e ) ;
            
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_NOT_PROTOCOL,e ) ;
            
        } finally {
            
            paramBin = null ;
            session = null ;
            sessionMan = null ;
            param = null ;
            applicationName = null ;
            commandName = null ;
            paramName = null ;
            parameters = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * サービス実行.
     */
    private final void executionService(
        JRcService service,JRcRequestImple request,JRcResponseImple response )
        throws JRCommandException {
        
        int i ;
        int len ;
        boolean flg = false ;
        
        String[] names = null ;
        String serviceName = null ;
        JRCommand command = null ;
        
        serviceName = request.getCommandName() ;
        
        try {
            
            if(
                serviceName == null ||
                ( serviceName = serviceName.trim().toLowerCase() ).length() <= 0
            ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_NOT_APPLICATION_NAME,
                    "受信データにサービス名は存在しません"
                ) ;
            }
            
            // サービス名がリストの場合.
            if( JRcService.COMMON_SERVICE_LIST.equals( serviceName.toLowerCase() ) == true ) {
                
                PrintWriter w = null ;
                
                names = service.getServiceNames() ;
                if( names != null && ( len = names.length ) > 0 ) {
                    
                    w = response.getWriter() ;
                    
                    for( i = 0 ; i < len ; i ++ ) {
                        
                        if( i != 0 ) {
                            w.print( " " ) ;
                        }
                        w.print( names[ i ] ) ;
                        
                    }
                    response.setSuccess( 0 ) ;
                    
                }
                else {
                    
                    throw new JRCommandException(
                        JRcErrorCodeDef.WARNING_ZERO_SERVICE,
                        "登録されているサービスは存在しません"
                    ) ;
                    
                }
                
            }
            // サービス名がサーババージョンの場合.
            else if( JRcService.COMMON_SERVER_VERSION.equals( serviceName.toLowerCase() ) == true ) {
                
                PrintWriter w = null ;
                
                w = response.getWriter() ;
                w.print( service.getServerVersion() ) ;
                response.setSuccess( 0 ) ;
                
            }
            // サービス名がヘルプの場合.
            else if( JRcService.COMMON_SERVICE_HELP.equals( serviceName.toLowerCase() ) == true ) {
                
                PrintWriter w = null ;
                String helpService = null ;
                
                helpService = request.getParameterToString( JRcService.COMMON_SERVICE_PARAM_HELP ) ;
                if( helpService == null || ( helpService = helpService.trim() ).length() <= 0 ) {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_NOT_SERVICE,
                        "ヘルプ表示を行うサービス名は指定されていないか不正です"
                    ) ;
                }
                
                names = service.getServiceNames() ;
                if( names != null && ( len = names.length ) > 0 ) {
                    
                    for( i = 0,flg = false ; i < len ; i ++ ) {
                        if( helpService.equals( names[ i ] ) == true ) {
                            flg = true ;
                            break ;
                        }
                        
                    }
                    
                    if( flg == false ) {
                        throw new JRCommandException(
                            JRcErrorCodeDef.ERROR_NOT_SERVICE,
                            "HELP対象条件名[" + helpService +
                            "]のサービスは存在しません"
                        ) ;
                    }
                    
                }
                else {
                    
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_NOT_SERVICE,
                        "HELP対象条件名[" + helpService +
                        "]のサービスは存在しません"
                    ) ;
                    
                }
                
                command = service.getService( helpService ) ;
                
                if( command == null ) {
                    
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_NOT_SERVICE,
                        "ヘルプ表示を行うサービス名[" +
                            helpService + "]は利用できません"
                    ) ;
                    
                }
                
                w = response.getWriter() ;
                command.getHelp( w ) ;
                response.setSuccess( 0 ) ;
                
            }
            // 登録サービスの場合.
            else {
                
                names = service.getServiceNames() ;
                if( names != null && ( len = names.length ) > 0 ) {
                    for( i = 0,flg = false ; i < len ; i ++ ) {
                        if( serviceName.equals( names[ i ] ) == true ) {
                            flg = true ;
                            break ;
                        }
                        
                    }
                    
                    if( flg == false ) {
                        throw new JRCommandException(
                            JRcErrorCodeDef.ERROR_NOT_SERVICE,
                            "対象名[" + serviceName +
                            "]のサービスは存在しません"
                        ) ;
                    }
                    
                }
                else {
                    
                    throw new JRCommandException(
                        JRcErrorCodeDef.WARNING_ZERO_SERVICE,
                        "登録されているサービスは存在しません(call:" +
                        serviceName + ")"
                    ) ;
                    
                }
                
                command = service.getService( serviceName ) ;
                
                if( command == null ) {
                    
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_NOT_SERVICE,
                        "指定されたサービス名[" +
                            serviceName + "]は利用できません"
                    ) ;
                    
                }
                
                // コマンド実行.
                command.execution( request,response ) ;
                
                LOG.info(
                    new StringBuffer().
                    append( "## コマンド:" ).
                    append( serviceName ).
                    append( " を実行しました[addr:" ).
                    append( request.getRemoteInetAddress().getHostAddress() ).
                    append( " port:" ).
                    append( request.getRemotePort() ).
                    append( "]" ).
                    toString()
                ) ;
                
            }
            
            // レスポンスをフラッシュして、破棄.
            response.flush() ;
            response.destroy() ;
            response = null ;
            
        } catch( JRCommandException jc ) {
            
            LOG.error( "コマンド処理中にエラーが発生[addr:" +
                request.getRemoteInetAddress().getHostAddress() + " port:" +
                request.getRemotePort() + "]",jc ) ;
            
            throw jc ;
            
        } catch( Exception e ) {
            
            LOG.error( "コマンド処理中にエラーが発生[addr:" +
                request.getRemoteInetAddress().getHostAddress() + " port:" +
                request.getRemotePort() + "]",e ) ;
            
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_EXECUTION_COMMAND,e ) ;
            
        } finally {
            serviceName = null ;
            command = null ;
        }
        
    }
    
    /**
     * データ送信処理.
     */
    private final void sendData(
        ResourceType resType,JRcResponseBean bean,OutputStream stream,
        InetAddress addr,int port )
        throws ExecutionException {
        
        int len ;
        int cnt ;
        int data ;
        
        byte[] buf = null ;
        String charset = null ;
        String mimeType = null ;
        String message = null ;
        BinResource bin = null ;
        BinResource sendBin = null ;
        BufferedBinResource bufBin = null ;
        OutputStream sendStream = null ;
        
        try {
            
            sendBin = Resource.createBinResource( resType,1 ) ;
            sendStream = new BufferedOutputStream(
                new BinResourceOutputStream(
                    sendBin
                )
            ) ;
            
            // データモード(0:false 1:true).
            sendStream.write( ( ( bean.isMode() == true ) ? 1 : 0 ) ) ;
            
            // キャラクタセット長.
            charset = bean.getCharset() ;
            
            // 対象データモードがバイナリの場合.
            if( bean.isMode() == false ) {
                
                // ISO_8859_1の文字コードで設定.
                charset = BaseDef.ISO_8859_1 ;
                
            }
            
            if(
                charset == null ||
                ( charset = charset.trim() ).length() <= 0
            ) {
                // ISO_8859_1の文字コードで設定.
                buf = BaseDef.ISO_8859_1.getBytes( BaseDef.UTF8 ) ;
            }
            else {
                buf = charset.getBytes( BaseDef.UTF8 ) ;
            }
            
            len = buf.length ;
            
            sendStream.write( ConvertParam.convertInt( len ) ) ;
            
            // キャラクタセット.
            if( len > 0 ) {
                sendStream.write( buf ) ;
                buf = null ;
            }
            
            // Mimeタイプ長.
            mimeType = bean.getMimeType() ;
            
            if(
                mimeType == null ||
                ( mimeType = mimeType.trim() ).length() <= 0
            ) {
                // [text/plain]で設定.
                buf = JRcResponseBean.DEFAULT_MIME_TYPE.getBytes( BaseDef.UTF8 ) ;
            }
            else {
                buf = mimeType.getBytes( BaseDef.UTF8 ) ;
            }
            
            len = buf.length ;
            
            sendStream.write( ConvertParam.convertInt( len ) ) ;
            
            // Mimeタイプ.
            if( len > 0 ) {
                sendStream.write( buf ) ;
                buf = null ;
            }
            
            // 戻り値.
            sendStream.write( ConvertParam.convertInt( bean.getResult() ) ) ;
            
            // 戻り値メッセージ長.
            message = bean.getResultMessage() ;
            if(
                message == null ||
                ( message = message.trim() ).length() <= 0
            ) {
                bin = null ;
                len = 0 ;
            }
            else {
                buf = message.getBytes( BaseDef.UTF8 ) ;
                len = buf.length ;
            }
            
            sendStream.write( ConvertParam.convertInt( len ) ) ;
            
            // 戻り値メッセージ.
            if( len > 0 ) {
                sendStream.write( buf ) ;
                bin = null ;
            }
            
            // セッションID.
            sendStream.write( ConvertParam.convertLong( bean.getSessionID() ) ) ;
            
            // データ長.
            bin = bean.getBinary() ;
            if( bin == null || ( len = bin.size() ) <= 0 ) {
                len = 0 ;
            }
            
            sendStream.write( ConvertParam.convertInt( len ) ) ;
            
            // データ.
            bufBin = new BufferedBinResource( bin ) ;
            
            // データ書き込み.
            for( cnt = 0 ;; ) {
                if( ( data = bufBin.get( cnt ) ) < 0 ) {
                    break ;
                }
                
                sendStream.write( data ) ;
                cnt ++ ;
            }
            
            sendStream.flush() ;
            
            sendStream.close() ;
            sendStream = null ;
            
            buf = new byte[ SEND_BUFFER ] ;
            
            // 送信条件に対して、プロトコルヘッダと情報長を設定する.
            len = JRcProtocolDef.SEND_HEADER_LENGTH + 4 + sendBin.size() ;
            
            // プロトコルヘッダ.
            stream.write( JRcProtocolDef.SEND_HEADER ) ;
            
            // 情報長.
            stream.write( ConvertParam.convertInt( len ) ) ;
            
            // プロトコルデータ.
            bufBin = new BufferedBinResource( sendBin ) ;
            
            // 生成したプロトコル内容を設定.
            for( cnt = 0 ;; ) {
                
                if( ( data = bufBin.get( cnt ) ) < 0 ) {
                    break ;
                }
                stream.write( data ) ;
                cnt ++ ;
                
            }
            
            stream.flush() ;
            
            bufBin.clear() ;
            bufBin = null ;
            
        } catch( Exception e ) {
            LOG.error( "送信処理中が発生[addr:" +
                addr.getHostAddress() + " port:" +
                port + "]",e ) ;
            throw new ExecutionException(
                e,ExecutionException.LEVEL_STOP
            ) ;
        } finally {
            
            if( bufBin != null ) {
                try {
                    bufBin.clear() ;
                } catch ( Exception e ) {
                }
            }
            
            if( sendStream != null ) {
                try {
                    sendStream.close() ;
                } catch( Exception ee ) {
                }
            }
            
            if( bean != null ) {
                bean.destroy() ;
            }
            
            buf = null ;
            charset = null ;
            mimeType = null ;
            message = null ;
            bin = null ;
            sendBin = null ;
            sendStream = null ;
            bufBin = null ;
            
        }
        
    }
    
    /**
     * エラーデータを送信.
     */
    private final void sendErrorData(
        ResourceType resType,JRCommandException exp,OutputStream stream,
        InetAddress addr,int port
    )
        throws ExecutionException {
        
        int len ;
        int cnt ;
        int data ;
        
        String message = null ;
        byte[] bin = null ;
        BufferedBinResource bufBin = null ;
        BinResource sendBin = null ;
        OutputStream sendStream = null ;
        
        try {
            
            sendBin = Resource.createBinResource( resType,1 ) ;
            sendStream = new BufferedOutputStream(
                new BinResourceOutputStream(
                    sendBin
                )
            ) ;
            
            // データモード(0:false 1:true).
            sendStream.write( 1 ) ;
            
            // キャラクタセット長.
            bin = BaseDef.UTF8.getBytes( BaseDef.UTF8 ) ;
            sendStream.write( ConvertParam.convertInt( bin.length ) ) ;
            
            // キャラクタセット.
            sendStream.write( bin ) ;
            bin = null ;
            
            // Mimeタイプ長.
            bin = JRcResponseBean.DEFAULT_MIME_TYPE.getBytes( BaseDef.UTF8 ) ;
            sendStream.write( ConvertParam.convertInt( bin.length ) ) ;
            
            // Mimeタイプ.
            sendStream.write( bin ) ;
            bin = null ;
            
            // 戻り値.
            sendStream.write( ConvertParam.convertInt( exp.getErrorCode() ) ) ;
            
            // 戻り値メッセージ長.
            message = exp.getMessage() ;
            if(
                message == null ||
                ( len = ( message = message.trim() ).length() ) <= 0
            ) {
                bin = null ;
                len = 0 ;
            }
            else {
                bin = message.getBytes( BaseDef.UTF8 ) ;
                len = bin.length ;
            }
            
            sendStream.write( ConvertParam.convertInt( len ) ) ;
            
            // 戻り値メッセージ.
            if( len > 0 ) {
                sendStream.write( bin ) ;
                bin = null ;
            }
            
            // セッションID.
            sendStream.write( ConvertParam.convertLong( -1L ) ) ;
            
            // データ長.
            sendStream.write( ConvertParam.convertInt( 0 ) ) ;
            
            sendStream.flush() ;
            sendStream.close() ;
            sendStream = null ;
            
            bin = new byte[ SEND_BUFFER ] ;
            
            // 送信条件に対して、プロトコルヘッダと情報長を設定する.
            len = JRcProtocolDef.SEND_HEADER_LENGTH + 4 + sendBin.size() ;
            
            // プロトコルヘッダ.
            stream.write( JRcProtocolDef.SEND_HEADER ) ;
            
            // 情報長.
            stream.write( ConvertParam.convertInt( len ) ) ;
            
            // プロトコルデータ.
            bufBin = new BufferedBinResource( sendBin ) ;
            
            // 生成したプロトコル内容を設定.
            for( cnt = 0 ;; ) {
                
                if( ( data = bufBin.get( cnt ) ) < 0 ) {
                    break ;
                }
                stream.write( data ) ;
                cnt ++ ;
                
            }
            
            stream.flush() ;
            
            bufBin.clear() ;
            bufBin = null ;
            
        } catch( Exception e ) {
            LOG.error( "エラーメッセージ送信処理中にエラーが発生[addr:" +
                addr.getHostAddress() + " port:" +
                port + "]",e ) ;
            throw new ExecutionException(
                e,ExecutionException.LEVEL_STOP
            ) ;
        } finally {
            
            if( bufBin != null ) {
                try {
                    bufBin.clear() ;
                } catch ( Exception e ) {
                }
            }
            
            if( sendStream != null ) {
                try {
                    sendStream.close() ;
                } catch( Exception ee ) {
                }
            }
            
            message = null ;
            bin = null ;
            bufBin = null ;
            sendBin = null ;
            sendStream = null ;
            
        }
        
    }
    
    /**
     * コネクションソケットを強制クローズ.
     */
    private final void shutdownSocket() {
        
        try {
            synchronized( sync.get() ) {
                this.socket.close() ;
            }
        } catch( Exception e ) {
        }
        
        this.socket = null ;
    }
    
}

