/*
 * @(#)JRcCacheImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.resource.cache.CacheIO;
import com.JRcServer.commons.resource.cache.CacheManager;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.UtilCom;

/**
 * FastMQキャッシュオブジェクト.
 * <BR><BR>
 * FastMQ内における１つのキャッシュ情報を持つオブジェクトです.
 *  
 * @version 2006/09/03
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcCacheImple implements JRcCache
{
    
    /**
     * キャッシュ名.
     */
    private String name = null ;
    
    /**
     * キャッシュID.
     */
    private long id = -1L ;
    
    /**
     * ファイル名.
     */
    private String fileName = null ;
    
    /**
     * キャッシュセクタ数.
     */
    private int cacheSize = -1 ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType resourceType = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private JRcCacheImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象条件を設定してキャッシュ情報を生成します.
     * <BR>
     * @param dir 対象のキャッシュディレクトリ名を設定します.
     * @param name 対象のキャッシュファイル名を設定します.
     * @param size キャッシュセクタ数を設定します.<BR>
     *             設定可能な最小値は[CacheDef.MIN_SECTOR]です.<BR>
     *             設定可能な最大値は[CacheDef.MAX_SECTOR]です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public JRcCacheImple( String dir,String name,int size )
        throws InputException,AccessException {
        
        CacheIO io = null ;
        String fileName = null ;
        
        if(
            dir == null || dir.length() <= 0 ||
            name == null || name.length() <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        sync.create() ;
        
        try {
            
            if( IOCom.isDirExists( dir ) == false ) {
                IOCom.mkdirs( dir ) ;
            }
            
            fileName = new StringBuffer().
                append( dir ).
                append( BaseDef.FILE_SEPARATOR ).
                append( name ).
                toString() ;
            
            name = JRcCacheImple.getCacheName( fileName ) ;
            CacheManager.putNamingManagerByCacheIO( name,fileName,size ) ;
            io = CacheManager.getNamingManagerByCacheIO( name ) ;
            
            synchronized( sync.get() ) {
                
                this.name = name ;
                this.id = io.getUniqueID() ;
                this.fileName = fileName ;
                this.cacheSize = io.getMaxSector() ;
                this.resourceType = new ResourceType(
                    ResourceType.RESOURCE_TYPE_CACHE,
                    null,
                    this.id
                ) ;
                
            }
            
        } catch( AccessException ae ) {
            this.destroy() ;
            throw ae ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * チャネルオブジェクトを破棄します.
     */
    public void destroy() {
        
        try {
            synchronized( sync.get() ) {
                CacheManager.delete( id ) ;
                name = null ;
                id = -1L ;
                fileName = null ;
                cacheSize = -1 ;
                resourceType = null ;
            }
            
        } catch( Exception e ) {
            
        }
        
        sync.clear() ;
        name = null ;
        id = -1L ;
        fileName = null ;
        cacheSize = -1 ;
        resourceType = null ;
        
    }
    
    /**
     * キャッシュ名を取得.
     * <BR><BR>
     * キャッシュ名を取得します.
     * <BR>
     * @return String キャッシュ名を取得します.
     */
    public String getName() {
        
        String ret = null ;
        
        try {
            
            synchronized( sync.get() ) {
                ret = name ;
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュIDを取得.
     * <BR><BR>
     * キャッシュIDを取得します.
     * <BR>
     * @return long キャッシュIDが返されます.
     */
    public long getID() {
        
        long ret = -1L ;
        
        try {
            
            synchronized( sync.get() ) {
                ret = id ;
            }
            
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュファイル名を取得.
     * <BR><BR>
     * キャッシュファイル名を取得します.
     * <BR>
     * @return String キャッシュファイル名が返されます.
     */
    public String getFileName() {
        
        String ret = null ;
        
        try {
            
            synchronized( sync.get() ) {
                ret = fileName ;
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュセクタサイズを取得.
     * <BR><BR>
     * キャッシュセクタサイズを取得します.
     * <BR>
     * @return int キャッシュセクタサイズを取得します.
     */
    public int getCacheSize() {
        
        int ret = -1 ;
        
        try {
            
            synchronized( sync.get() ) {
                ret = cacheSize ;
            }
            
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在利用中のキャッシュセクタサイズを取得.
     * <BR><BR>
     * 現在利用中のキャッシュセクタサイズを取得します.
     * <BR>
     * @return int 現在利用中のキャッシュセクタサイズが返されます.
     */
    public int getUseCacheSize() {
        
        int ret = -1 ;
        CacheIO io = null ;
        
        try {
            
            synchronized( sync.get() ) {
                io = CacheManager.getNamingManagerByCacheIO( name ) ;
                ret = io.getUseSector() ;
            }
            
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 利用比率[%]を取得.
     * <BR><BR>
     * 利用比率[%]を取得します.
     * <BR>
     * @return int 利用比率[0-100]%をが整数で返されます.<BR>
     *             仮に70%の利用状況の場合[70]の整数が返されます.
     */
    public int getCachePersent() {
        
        double dbl ;
        int ret = -1 ;
        CacheIO io = null ;
        
        try {
            
            synchronized( sync.get() ) {
                io = CacheManager.getNamingManagerByCacheIO( name ) ;
                dbl = ( double )(
                    ( ( double )io.getUseSector() ) / ( ( double )cacheSize )
                ) ;
                ret = ( int )( dbl * 100f ) ;
            }
            
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュリソースタイプを取得.
     * <BR><BR>
     * キャッシュリソースタイプを取得します.
     * <BR>
     * @return ResourceType キャッシュリソースタイプが返されます.
     */
    public ResourceType getResourceType() {
        
        ResourceType ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = resourceType ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュI/Oオブジェクトを取得.
     * <BR><BR>
     * キャッシュI/Oオブジェクトを取得します.
     * <BR>
     * @return CacheIO キャッシュI/Oオブジェクトが返されます.
     */
    public CacheIO getCacheIO() {
        
        CacheIO ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = CacheManager.getNamingManagerByCacheIO( name ) ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュオブジェクトが有効かチェック.
     * <BR><BR>
     * キャッシュオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isCache() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = ( name != null && name.length() <= 0 ) ?
                    true : false ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュ名を生成.
     */
    private static final String getCacheName( String fileName ) {
        
        String ret = null ;
        
        try {
            ret = UtilCom.changeString( fileName,BaseDef.YEN_CODE,"~" ) ;
            ret = UtilCom.changeString( ret,BaseDef.SLASH_CODE,"~" ) ;
        } catch( Exception e ) {
            ret = fileName ;
        }
        
        return ret ;
        
    }
    
}

