/*
 * @(#)JRcBaseManagerImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.DeleteSessionTrigger;
import com.JRcServer.JRcBaseDefineBean;
import com.JRcServer.JRcService;
import com.JRcServer.JRcStateDef;
import com.JRcServer.commons.beans.BaseBean;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.resource.cache.CacheDef;
import com.JRcServer.commons.serialize.SerializeUtil;
import com.JRcServer.commons.sys.Initializer;

/**
 * JRcServer基本マネージャ.
 * <BR><BR>
 * JRcServer基本マネージャクラス.
 *  
 * @version 2006/09/07
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
class JRcBaseManagerImple implements Initializer,JRcBaseManager {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( JRcBaseManagerImple.class ) ;
    
    /**
     * デフォルトキャッシュサイズ.
     */
    private static final int DEF_CACHE_SIZE = JRcCache.DEFAULT_CHACHE_LENGTH ;
    
    /**
     * JRcServerシンボル.
     */
    private static final String JRC_SERVER_SIMBOL = "#JRC_SV@" + JRcManagerFactory.VERSION ;
    
    /**
     * JRcServer定義ディレクトリ名.
     */
    private static final String JRC_SERVER_DIR = "jrcserver" ;
    
    /**
     * セッションシリアライズファイル名.
     */
    private static final String SERIALIZABLE_NAME = ".serializable" ;
    
    /**
     * キャッシュファイル名.
     */
    private static final String CACHE_NAME = ".cache" ;
    
    
    
    /**
     * キャッシュ.
     */
    private JRcCacheImple cache = null ;
    
    /**
     * セッションマネージャ.
     */
    private JRcSessionManagerImple sessionManager = null ;
    
    /**
     * 基本ディレクトリ名.
     */
    private String baseDir = null ;
    
    /**
     * バインドアドレス.
     */
    private InetAddress bindAddress = null ;
    
    /**
     * バインドポート番号.
     */
    private int bindPort = -1 ;
    
    /**
     * セッション削除時のコールバックメソッド.
     */
    private DeleteSessionTrigger callback = null ;
    
    /**
     * JRcServerステータス.
     */
    private int state = JRcStateDef.STATE_DOWN ;
    
    /**
     * セッションタイムアウト監視スレッド.
     */
    private JRcSessionThread sessionThread = null ;
    
    /**
     * 通信オブジェクト.
     */
    private JRcConnectServerImple connectServer = null ;
    
    /**
     * サービス管理オブジェクト.
     */
    private JRcService service = null ;
    
    /**
     * オプションオブジェクト.
     */
    private Object option = null ;
    
    /**
     * 初期化フラグ.
     */
    private boolean initFlag = false ;
    
    /**
     * コンストラクタ.
     */
    public JRcBaseManagerImple() {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.<BR>
     * また、このメソッドの場合、デフォルト値での初期化処理になります.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public synchronized void init() throws AccessException {
        throw new AccessException(
            JRcBaseManagerImple.class.getName() +
            " ではこのメソッド[init()]は対応していません" ) ;
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.
     * <BR>
     * @param bean 設定値が設定されているBeanオブジェクトを設定します.
     * @exception AccessException アクセス例外.
     */
    public synchronized void init( BaseBean bean ) throws AccessException {
        
        boolean sessionFlg = false ;
        int cacheSize = -1 ;
        JRcBaseDefineBean jrcBean = null ;
        String serializableFile = null ;
        
        if( initFlag == true ) {
            return ;
        }
        
        if(
            bean == null ||
            ( bean instanceof JRcBaseDefineBean ) == false ||
            ( jrcBean = ( JRcBaseDefineBean )bean ) == null ||
            jrcBean.getBaseDirectory() == null ||
            jrcBean.getBaseDirectory().length() <= 0 ||
            jrcBean.getBindPort() < 0 || jrcBean.getBindPort() > 65535 ||
            jrcBean.getService() == null
        ) {
            throw new AccessException( "指定された定義内容は不正です" ) ;
        }
        
        initFlag = true ;
        
        try {
            
            // バインドアドレスを取得.
            bindAddress = jrcBean.getBindAddress() ;
            
            if( bindAddress != null ) {
                LOG.info( "# bindAddress:" + bindAddress.getHostAddress() ) ;
            }
            else {
                LOG.info( "# bindAddress:DEFAULT" ) ;
            }
            
            // バインドポートを取得.
            bindPort = jrcBean.getBindPort() ;
            LOG.info( "# bindPort:" + bindPort ) ;
            
            // 基本ディレクトリ名生成.
            this.baseDir = new StringBuffer().
                append( jrcBean.getBaseDirectory() ).
                append( BaseDef.FILE_SEPARATOR ).
                append( JRC_SERVER_DIR ).
                toString() ;
            
            LOG.info( "# baseDirectory:" + baseDir ) ;
            
            // シリアライズファイル名生成.
            serializableFile = new StringBuffer().
                append( this.baseDir ).
                append( BaseDef.FILE_SEPARATOR ).
                append( SERIALIZABLE_NAME ).
                toString() ;
                
            // キャッシュ生成.
            if( ( cacheSize = jrcBean.getCacheSize() ) >= 0 ) {
                if( cacheSize == 0 ) {
                    cacheSize = DEF_CACHE_SIZE ;
                }
                this.cache = new JRcCacheImple(
                    baseDir,CACHE_NAME,cacheSize
                ) ;
                
                LOG.info( "# cacheSize(" + JRcCacheImple.class.getName() + "):" + cacheSize ) ;
            }
            else {
                LOG.info( "# cacheSize(Memory)" ) ;
            }
            
            // 以前にセッションがシリアライズされている場合.
            if( IOCom.isFileExists( serializableFile ) == true ) {
                
                try {
                    // セッションをシリアライズから復帰.
                    this.sessionManager = ( JRcSessionManagerImple )SerializeUtil.getCb32Serialize(
                        true,serializableFile+JRC_SERVER_SIMBOL,serializableFile
                    ) ;
                    
                    // セッション初期処理.
                    this.sessionManager.initSession() ;
                    
                    // セッションタイムアウトがセットされている場合.
                    if( jrcBean.getSessionTimeout() > 0L ){
                        this.sessionManager.setSessionTimeout(
                            jrcBean.getSessionTimeout()
                        ) ;
                    }
                    
                    LOG.info( "# シリアライズからセッションを復帰" ) ;
                    //LOG.info( "session:" + BaseDef.ENTER_SEPARATOR + this.sessionManager.toString() ) ;
                    sessionFlg = true ;
                    
                } catch( Exception ee ) {
                    sessionFlg = false ;
                    LOG.error( "# シリアライズ復帰に失敗",ee ) ;
                }
                
            }
            
            // 新しくセッション生成を行う場合.
            if( sessionFlg == false ) {
                
                if( IOCom.isFileExists( serializableFile ) == true ) {
                    IOCom.deleteFile( serializableFile ) ;
                }
                
                // セッションタイムアウトがセットされていない場合.
                if( jrcBean.getSessionTimeout() > 0L ){
                    this.sessionManager = new JRcSessionManagerImple() ;
                    LOG.info( "# sessionTimeout:DEFAULT" ) ;
                }
                // セッションタイムアウト値がセットされている場合.
                else {
                    this.sessionManager = new JRcSessionManagerImple(
                        jrcBean.getSessionTimeout()
                    ) ;
                    LOG.info( "# sessionTimeout:" + jrcBean.getSessionTimeout() ) ;
                }
                
            }
            
            // コールバックメソッドを設定.
            if( jrcBean.getCallback() != null ) {
                this.callback = jrcBean.getCallback() ;
                LOG.info( "# callback:" + this.callback.getClass().getName() ) ;
            }
            else {
                this.callback = null ;
                LOG.info( "# callback:NOT" ) ;
            }
            
            // サービスを設定.
            this.service = jrcBean.getService() ;
            
            // オプションオブジェクトを設定.
            this.option = jrcBean.getOption() ;
            
            // セッション管理スレッドを生成.
            this.sessionThread = new JRcSessionThread(
                this.sessionManager,this.callback ) ;
            
            // 通信オブジェクトを生成.
            this.connectServer = new JRcConnectServerImple(
                jrcBean.getService(),
                jrcBean.getBindPort(),
                jrcBean.getBindAddress(),
                jrcBean.getReceiveTimeout(),
                jrcBean.getMaxConnect()
            ) ;
            
            // ステータスを開始にセット.
            this.state = JRcStateDef.STATE_SUCCESS ;
            
        } catch( AccessException ae ) {
            this.destroy() ;
            throw ae ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実施します.<BR>
     * また、この処理を実施した場合、再び Initializer.init() を
     * 呼び出さないとオブジェクトの利用が出来なくなります.
     */
    public synchronized void destroy() {
        
        String serializableFile = null ;
        
        if( initFlag == false ) {
            return ;
        }
        
        // サーバコネクションが存在する場合.
        if(
            this.connectServer != null ||
            this.connectServer.isServer() == true
        ) {
            try {
                this.connectServer.destroy() ;
            } catch( Exception e ) {
            }
            this.connectServer = null ;
        }
        
        // セッションスレッドが存在する場合.
        if( this.sessionThread != null ) {
            try {
                this.sessionThread.destroy() ;
            } catch( Exception e ) {
            }
            this.sessionThread = null ;
        }
        
        try {
            
            // セッションマネージャが有効な場合.
            if(
                this.sessionManager != null &&
                this.sessionManager.isSessionManager() == true
            ) {
                
                // シリアライズファイル名生成.
                serializableFile = new StringBuffer().
                    append( this.baseDir ).
                    append( BaseDef.FILE_SEPARATOR ).
                    append( SERIALIZABLE_NAME ).
                    toString() ;
                
                // セッション情報をシリアライズ化.
                if( this.sessionManager.exitSession() == false ) {
                    
                    LOG.info( "# [Shutdown] : session - delete" ) ;
                    
                    // シリアライズ化できない場合.
                    IOCom.deleteFile( serializableFile ) ;
                    
                }
                // セッション情報シリアライズ化成功の場合.
                else {
                    
                    SerializeUtil.putCb32Serialize(
                        true,serializableFile+JRC_SERVER_SIMBOL,serializableFile,
                        this.sessionManager
                    ) ;
                    
                    LOG.info( "# [Shutdown]: session - serializable" ) ;
                    
                }
                
            }
            else {
                if( IOCom.isFileExists( serializableFile ) == true ) {
                    IOCom.deleteFile( serializableFile ) ;
                }
                LOG.info( "# [Shutdown] : session - not save." ) ;
            }
            
            this.sessionManager.destroy() ;
            
        } catch( Exception e ) {
            
        }
        
        this.sessionManager = null ;
        
        try {
            
            // キャッシュファイルが有効な場合.
            if( this.cache != null && this.cache.isCache() == true ) {
                
                this.cache.destroy() ;
                
            }
            
        } catch( Exception e ) {
            
        }
        
        this.cache = null ;
        this.baseDir = null ;
        this.callback = null ;
        this.option = null ;
        this.connectServer = null ;
        this.service = null ;
        this.state = JRcStateDef.STATE_DOWN ;
        
        initFlag = false ;
        
    }
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @return JRcCache キャッシュオブジェクトが返されます.
     */
    public synchronized JRcCache getCache() {
        
        if( ( this.state & JRcStateDef.STATE_MASK_ERROR ) == JRcStateDef.STATE_MASK_ERROR ) {
            return null ;
        }
        
        return this.cache ;
        
    }
    
    /**
     * セッションマネージャオブジェクトを取得.
     * <BR><BR>
     * セッションマネージャオブジェクトを取得します.
     * <BR>
     * @return JRcSessionManager セッションマネージャオブジェクトが返されます.
     */
    public synchronized JRcSessionManager getSessionManager() {
        
        if( ( this.state & JRcStateDef.STATE_MASK_ERROR ) == JRcStateDef.STATE_MASK_ERROR ) {
            return null ;
        }
        
        return this.sessionManager ;
        
    }
    
    /**
     * 基本ディレクトリ名を取得.
     * <BR><BR>
     * 基本ディレクトリ名を取得します.
     * <BR>
     * @return String 基本ディレクトリ名が返されます.
     */
    public synchronized String getBaseDirectory() {
        
        if( ( this.state & JRcStateDef.STATE_MASK_ERROR ) == JRcStateDef.STATE_MASK_ERROR ) {
            return null ;
        }
        
        return this.baseDir ;
        
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     */
    public synchronized InetAddress getInetAddress() {
        
        if( ( this.state & JRcStateDef.STATE_MASK_ERROR ) == JRcStateDef.STATE_MASK_ERROR ) {
            return null ;
        }
        
        return this.bindAddress ;
        
    }
    
    /**
     * バインドポート番号を取得.
     * <BR><BR>
     * バインドポート番号を取得します.
     * <BR>
     * @return int バインドポート番号が返されます.
     */
    public synchronized int getBindPort() {
        
        if( ( this.state & JRcStateDef.STATE_MASK_ERROR ) == JRcStateDef.STATE_MASK_ERROR ) {
            return -1 ;
        }
        
        return this.bindPort ;
        
    }
    
    /**
     * コールバックオブジェクトを取得.
     * <BR><BR>
     * セッション削除時に呼び出されるコールバックオブジェクトを取得します.
     * <BR>
     * @return DeleteSessionTrigger コールバックオブジェクトが返されます.
     */
    public synchronized DeleteSessionTrigger getCallback() {
        
        return this.callback ;
        
    }
    
    /**
     * オプションオブジェクトを取得.
     * <BR><BR>
     * 設定されているオプションオブジェクトを取得します.
     * <BR>
     * @return Object オプションオブジェクトが返されます.
     */
    public synchronized Object getOption() {
        
        return this.option ;
        
    }
    
    /**
     * コネクションサーバオブジェクトを取得.
     * <BR><BR>
     * コネクションサーバオブジェクトを取得します.
     * <BR>
     * @return JRcConnectServer コネクションサーバオブジェクトが返されます.
     */
    public synchronized JRcConnectServer getConnectServer() {
        
        return this.connectServer ;
        
    }
    
    /**
     * サービス管理オブジェクトを取得.
     * <BR><BR>
     * サービス管理オブジェクトを取得します.
     * <BR>
     * @return JRcService サービスオブジェクトが返されます.
     */
    public synchronized JRcService getServer() {
        
        return this.service ;
        
    }
    
    /**
     * ステータス情報を設定.
     * <BR><BR>
     * ステータス情報を設定します.
     * <BR>
     * @param state 対象のステータスを設定します.
     */
    public synchronized void setState( int state ) {
        
        if(
            ( state & 0xf0000000 ) == 0 ||
            ( state & 0xf0000000 ) == JRcStateDef.STATE_MASK_WARNING ||
            ( state & 0xf0000000 ) == JRcStateDef.STATE_MASK_ERROR
        ) {
            this.state = state ;
        }
        
    }
    
    /**
     * ステータス情報を取得.
     * <BR><BR>
     * ステータス情報を取得します.
     * <BR>
     * @return int ステータス情報が返されます.
     */
    public synchronized int getState() {
        
        return this.state ;
        
    }
    
}

