/*
 * @(#)InitJRcServer.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.server ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.JRcBaseDefineBean;
import com.JRcServer.JRcSession;
import com.JRcServer.commons.InitPackage;
import com.JRcServer.commons.beans.BaseBean;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.sys.Initializer;
import com.JRcServer.commons.util.UtilCom;

/**
 * JRcServer初期化処理.
 *  
 * @version 2006/09/13
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
public class InitJRcServer implements Initializer {
    
    /**
     * キャッシュサイズデフォルト値.
     */
    public static final int DEF_CHACHE_LENGTH = JRcCache.DEFAULT_CHACHE_LENGTH ;
    
    /**
     * バインドポートデフォルト値.
     */
    public static final int DEF_BIND_PORT = JRcConnectServer.DEFAULT_BIND_PORT ;
    
    /**
     * セッションタイムアウトデフォルト値.
     */
    public static final long DEF_SESSION_TIMEOUT = JRcSession.DEF_SESSION_TIMEOUT ;
    
    /**
     * デフォルト最大コネクション.
     */
    public static final int DEF_MAX_CONNECTION = JRcConnectServer.DEFAULT_MAX_CONNECTION ;
    
    /**
     * デフォルト受信タイムアウト.
     */
    public static final int DEF_RECEIVE_TIMEOUT = JRcConnectServer.DEFAULT_TIMEOUT ;
    
    
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( InitJRcServer.class ) ;
    
    /**
     * 同期用.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * 固有サービス初期化処理.
     */
    private Initializer uniqueServiceInit = null ;
    
    /**
     * コンストラクタ.
     */
    private InitJRcServer() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * JRcServerを初期化します.<BR>
     * また、固有のサービスを提供する場合、その条件を初期化するための
     * 初期化オブジェクトを設定します.
     * <BR>
     * @param uniqueObject 固有サービス初期化処理用オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public InitJRcServer( Initializer uniqueObject )
        throws InputException {
        
        synchronized( SYNC ) {
            if( uniqueObject == null ) {
                throw new InputException( "引数は不正です" ) ;
            }
            
            this.uniqueServiceInit = uniqueObject ;
        }
        
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.<BR>
     * また、このメソッドの場合、デフォルト値での初期化処理になります.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public void init() throws AccessException {
        throw new AccessException(
            InitJRcServer.class.getName() +
            " ではこのメソッド[init()]は対応していません" ) ;
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.
     * <BR>
     * @param bean 設定値が設定されているBeanオブジェクトを設定します.
     * @exception AccessException アクセス例外.
     */
    public void init( BaseBean bean ) throws AccessException {
        
        synchronized( SYNC ) {
            
            if(
                bean == null ||
                ( bean instanceof JRcBaseDefineBean ) == false ||
                ( ( JRcBaseDefineBean )bean ).getOption() == null
            ) {
                throw new AccessException( "指定された起動Beanは不正です" ) ;
            }
            
            if( JRcManagerFactory.isJRcBaseManager() == false ) {
                
                try {
                    
                    LOG.info( "## [JRcServer] init - start" ) ;
                    
                    // JRcCommons初期化処理.
                    new InitPackage().init() ;
                    
                    if( uniqueServiceInit != null ) {
                        
                        // 固有の初期化処理.
                        uniqueServiceInit.init( bean ) ;
                        
                    }
                    
                    // ファクトリにBeanを生成して初期化.
                    JRcManagerFactory.initJRcBeanManager( bean ) ;
                    
                    LOG.info( "## [JRcServer] init - end:(success)" ) ;
                    
                } catch( AccessException ae ) {
                    
                    this.halt( this.uniqueServiceInit ) ;
                    LOG.error( "## [JRcServer] init - end:(error)" ) ;
                    throw ae ;
                    
                }
                
            }
            
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実施します.<BR>
     * また、この処理を実施した場合、再び Initializer.init() を
     * 呼び出さないとオブジェクトの利用が出来なくなります.
     */
    public synchronized void destroy() {
        
        synchronized( SYNC ) {
            
            this.destroy( this.uniqueServiceInit ) ;
            
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実施します.<BR>
     * また、この処理を実施した場合、再び Initializer.init() を
     * 呼び出さないとオブジェクトの利用が出来なくなります.
     * <BR>
     * @param uniqueObject 固有サービス初期化処理用オブジェクトを設定します.
     */
    public synchronized void destroy( Initializer uniqueServiceInit ) {
        
        synchronized( SYNC ) {
            
            if( JRcManagerFactory.isJRcBaseManager() == true ) {
                
                this.halt( uniqueServiceInit ) ;
                
            }
            
        }
        
    }
    
    /**
     * 処理強制終了.
     */
    private final void halt( Initializer uniqueServiceInit ) {
        
        LOG.info( "## [JRcServer] destroy - start" ) ;
        
        if( uniqueServiceInit != null ) {
            uniqueServiceInit.destroy() ;
        }
        
        JRcManagerFactory.destroyJRcBeanManager() ;
        
        LOG.info( "## [JRcServer] destroy - end" ) ;
        
        UtilCom.sleep( 500 ) ;
        
        new InitPackage().destroy() ;
        
    }
}

