/*
 * @(#)JRcRequest.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer ;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.util.Date;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ConvertException;

/**
 * JRcServerリクエストインターフェイス.
 *  
 * @version 2006/09/06
 * @author  masahito suzuki
 * @since   JRcServerAPI 1.00
 */
public interface JRcRequest {
    
    /**
     * リモートIPアドレスを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントのIPアドレスを取得します.
     * <BR>
     * @return InetAddress リモートIPアドレスが返されます.
     */
    public InetAddress getRemoteInetAddress() ;
    
    /**
     * リモートポート番号.
     * <BR><BR>
     * リクエストを送ってきたクライアントのIPアドレスを取得します.
     * <BR>
     * @return int リモートポート番号が返されます.
     */
    public int getRemotePort() ;
    
    /**
     * サーバIPアドレスを取得.
     * <BR><BR>
     * このサーバのIPアドレスを取得します.
     * <BR>
     * @return InetAddress サーバIPアドレスが返されます.
     */
    public InetAddress getInetAddress() ;
    
    /**
     * サーバポート番号を取得.
     * <BR><BR>
     * このサーバのポート番号を取得します.
     * <BR>
     * @return int サーバポート番号が返されます.
     */
    public int getPort() ;
    
    /**
     * コマンド名を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのコマンド名を取得します.
     * <BR>
     * @return String コマンド名が返されます.
     */
    public String getCommandName() ;
    
    /**
     * パラメータを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return JRcParameter パラメータ情報が返されます.
     */
    public JRcParameter getParameter( String key ) ;
    
    /**
     * パラメータをバイナリで取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return byte[] パラメータ情報が返されます.
     */
    public byte[] getParameterToBinary( String key ) ;
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return String パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public String getParameterToString( String key )
        throws ConvertException ;
    
    /**
     * パラメータをフラグで取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Boolean パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Boolean getParameterToBoolean( String key )
        throws ConvertException ;
    
    /**
     * パラメータを数値(Short)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Short パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Short getParameterToShort( String key )
        throws ConvertException ;
    
    /**
     * パラメータを数値(Integer)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Integer パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Integer getParameterToInteger( String key )
        throws ConvertException ;
    
    /**
     * パラメータを数値(Long)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Long パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Long getParameterToLong( String key )
        throws ConvertException ;
    
    /**
     * パラメータを浮動少数値(Float)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Float パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Float getParameterToFloat( String key )
        throws ConvertException ;
    
    /**
     * パラメータを浮動少数値(Double)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Double パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Double getParameterToDouble( String key )
        throws ConvertException ;
    
    /**
     * パラメータを浮動少数値(Decimal)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return BigDecimal パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public BigDecimal getParameterToDecimal( String key )
        throws ConvertException ;
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータを取得します.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return Date パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Date getParameterToTimestamp( String key )
        throws ConvertException ;
    
    /**
     * パラメータタイプを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータタイプが返されます.
     * <BR>
     * @param key 取得対象のKey名を設定します.
     * @return int パラメータタイプが返されます.<BR>
     *             [JRcParameterType#TYPE_BINARY]が返された場合(byte[])情報です.<BR>
     *             [JRcParameterType#TYPE_STRING]が返された場合(String)情報です.<BR>
     *             [JRcParameterType#TYPE_BOOLEAN]が返された場合(Boolean)情報です.<BR>
     *             [JRcParameterType#TYPE_SHORT]が返された場合(Short)情報です.<BR>
     *             [JRcParameterType#TYPE_INTEGER]が返された場合(Integer)情報です.<BR>
     *             [JRcParameterType#TYPE_LONG]が返された場合(Long)情報です.<BR>
     *             [JRcParameterType#TYPE_FLOAT]が返された場合(Float)情報です.<BR>
     *             [JRcParameterType#TYPE_DOUBLE]が返された場合(Double)情報です.<BR>
     *             [JRcParameterType#TYPE_DECIMAL]が返された場合(BigDecimal)情報です.<BR>
     *             [JRcParameterType#TYPE_TIMESTAMP]が返された場合(Date)情報です.<BR>
     *             パラメータタイプが定義されていない場合かキー内容が存在しない場合、
     *             [-1]が返されます.
     */
    public int getParameterType( String key ) ;
    
    /**
     * パラメータキー名一覧を取得.
     * <BR><BR>
     * パラメータキー名一覧を取得します.
     * <BR>
     * @return String[] パラメータキー名一覧が返されます.
     */
    public String[] getParameterKeys() ;
    
    /**
     * パラメータ数を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ数が返されます.
     * <BR>
     * @return int 設定されているパラメータ数が返されます.
     */
    public int getParameterSize() ;
    
    /**
     * パラメータ名が存在するかチェック.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ名が
     * 存在するかチェックします.
     * <BR>
     * @param key チェック対象Key名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象Key名はパラメータに存在します.<BR>
     *                 [false]が返された場合、対象Key名はパラメータに存在しません.
     */
    public boolean isParameterName( String key ) ;
    
    /**
     * アプリケーション名を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのアプリケーション名を取得します.
     * <BR>
     * @return String アプリケーション名が返されます.
     */
    public String getApplicationName() ;
    
    /**
     * セッション情報を取得.
     * <BR><BR>
     * このリクエストに関連付けられている現在のセッション情報を取得します.
     * <BR>
     * @return JRcSession セッション情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public JRcSession getSession()
        throws AccessException ;
    
}

