/*
 * @(#)ZipBinary.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.zip;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;

/**
 * ZipBinaryオブジェクト.
 * <BR><BR>
 * 指定したバイナリ情報をZIPによる圧縮／解凍します.
 *
 * @version 1.00, 2004/09/20
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ZipBinary
{
    
    /**
     * バッファ長.
     */
    private static final int BUF_LENGTH = 1024 ;
    
    /**
     * ZIPEntry名.
     */
    private static final String NAME_ZIP_ENTRY = "binary" ;
    
    /**
     * シングルトン.
     */
    private static final ZipBinary SNGL = new ZipBinary() ;
    
    
    /**
     * コンストラクタ.
     */
    private ZipBinary()
    {
        
    }
    
    /**
     * オブジェクトを取得.
     * <BR><BR>
     * オブジェクトを取得します.
     * <BR>
     * @return ZipBinary オブジェクト情報が返されます.
     */
    public static final ZipBinary getInstance() {
        return SNGL ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByZip( byte[] binary )
        throws InputException,AccessException
    {
        if( binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByZip( binary,0,binary.length,Deflater.DEFAULT_COMPRESSION ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param length 圧縮情報長を設定します.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByZip( byte[] binary,int length )
        throws InputException,AccessException
    {
        if( binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByZip( binary,0,length,Deflater.DEFAULT_COMPRESSION ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param offset 圧縮情報開始位置を設定します.
     * @param length 圧縮情報長を設定します.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByZip( byte[] binary,int offset,int length )
        throws InputException,AccessException
    {
        if( binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByZip( binary,offset,length,Deflater.DEFAULT_COMPRESSION ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param offset 圧縮情報開始位置を設定します.
     * @param length 圧縮情報長を設定します.
     * @param level 圧縮レベルを設定します.<BR>
     *              設定可能な最小値は[0]です.<BR>
     *              設定可能な最大値は[9]です.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByZip( byte[] binary,int offset,int length,int level )
        throws InputException,AccessException
    {
        int len ;
        
        ByteArrayOutputStream buf = null ;
        ZipOutputStream zip = null ;
        ZipEntry entry = null ;
        
        byte[] ret = null ;
        
        if(
            binary == null || ( len = binary.length ) <= 0 ||
            offset < 0 || offset > len ||
            length <= 0 || length > len ||
            ( offset + length ) > len
            
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            buf = new ByteArrayOutputStream() ;
            zip = new ZipOutputStream( buf ) ;
            entry = new ZipEntry( ZipBinary.NAME_ZIP_ENTRY ) ;
            
            zip.setMethod( ZipOutputStream.DEFLATED ) ;
            zip.setLevel( level ) ;
            
            zip.putNextEntry( entry ) ;
            zip.write( binary,offset,length ) ;
            zip.closeEntry() ;
            
            zip.finish() ;
            
            ret = buf.toByteArray() ;
            
        }catch( IllegalArgumentException ia ){
            throw new InputException( ia ) ;
        }catch( ZipException ze ){
            throw new AccessException( ze ) ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }finally{
            
            try{
                zip.close() ;
            }catch( Exception t ){
            }
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            zip = null ;
            entry = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @return byte[] 解凍されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertZipByBinary( byte[] zbin )
        throws InputException,AccessException
    {
        if( zbin == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertZipByBinary( zbin,0,zbin.length ) ;
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @param length 解凍情報長を設定します.
     * @return byte[] 解凍されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertZipByBinary( byte[] zbin,int length )
        throws InputException,AccessException
    {
        if( zbin == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertZipByBinary( zbin,0,length ) ;
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @param offset 解凍情報開始位置を設定します.
     * @param length 解凍情報長を設定します.
     * @return byte[] 解凍されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertZipByBinary( byte[] zbin,int offset,int length )
        throws InputException,AccessException
    {
        int len ;
        
        ByteArrayInputStream bIn = null ;
        ByteArrayOutputStream bOut = null ;
        ZipInputStream zip = null ;
        byte[] tmp = null ;
        byte[] ret = null ;
        
        if(
            zbin == null || ( len = zbin.length ) <= 0 ||
            offset < 0 || offset > len ||
            length <= 0 || length > len ||
            ( offset + length ) > len
            
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            bIn = new ByteArrayInputStream( zbin,offset,length ) ;
            bOut = new ByteArrayOutputStream() ;
            zip = new ZipInputStream( bIn ) ;
            
            if( zip.getNextEntry() == null ){
                throw new AccessException( "対象のエントリー情報は存在しません" ) ;
            }
            
            tmp = new byte[ ZipBinary.BUF_LENGTH ] ;
            while( ( len = zip.read( tmp ) ) >= 0 ){
                bOut.write( tmp,0,len ) ;
            }
            
            ret = bOut.toByteArray() ;
            
        }catch( ZipException ze ){
            throw new AccessException( ze ) ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }catch( AccessException ac ){
            throw ac ;
        }finally{
            
            try{
                zip.close() ;
            }catch( Exception t ){
            }
            
            try{
                bIn.close() ;
            }catch( Exception t ){
            }
            
            try{
                bOut.close() ;
            }catch( Exception t ){
            }
            
            bIn = null ;
            bOut = null ;
            zip = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
}
