/*
 * @(#)ReadProperty.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.read;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.ReadSyntaxException;
import com.JRcServer.commons.util.ArrayTable;
import com.JRcServer.commons.util.ReadIndex;

/**
 * プロパティ情報読み込みクラス。
 * <BR><BR>
 * 指定した情報から、プロパティ情報を取得します。
 * <BR>
 *
 * @version 1.00, 2003/10/27
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ReadProperty
{
    /**
     * デフォルトキーコード.
     * <BR><BR>
     * デフォルトのキーコードとなる情報です.
     */
    private static final String DEFAULT_KEY = "default-key" ;
    
    /**
     * コメントコード.
     */
    private static final String COMMENT = "#" ;
    
    /**
     * ファイル情報生成チェック.
     */
    private boolean m_isNew = false ;
    
    /**
     * プロパティ格納.
     * <BR><BR>
     * プロパティ情報が格納されます.
     */
    protected final ArrayTable m_table = new ArrayTable() ;
    
    /**
     * コンストラクタ.
     */
    public ReadProperty()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * プロパティ情報生成。
     * <BR><BR>
     * 指定した情報からプロパティ情報を生成します。
     * <BR>
     * @param info プロパティ情報が格納されている
     *             ReadIndexオブジェクトを指定します.
     * @exception InputException 引数例外
     * @exception ReadSyntaxException 読み込み文法例外.
     */
    public final void loadPropery( ReadIndex info )
        throws InputException,ReadSyntaxException
    {
        ReadKey key = null ;
        
        this.clear() ;
        
        try{
            
            key = new ReadKey() ;
            key.create( info,ReadProperty.COMMENT ) ;
            
            m_isNew = ReadProperty.analysis( key,m_table ) ;
            
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( ReadSyntaxException rs ){
            this.clear() ;
            throw rs ;
        }finally{
            key.clear() ;
            key = null ;
        }
    }
    
    /**
     * 情報クリアー。
     * <BR><BR>
     * 情報をクリアーします。
     */
    public final void clear()
    {
        m_table.clear() ;
        m_isNew = false ;
    }
    
    /**
     * テーブル情報の取得.
     * <BR><BR>
     * テーブル情報の取得を行います.
     * <BR>
     * @return ArrayTable 格納されているテーブル情報が返されます.
     */
    public final ArrayTable getTable()
    {
        return m_table.getCloneObject() ;
    }
    
    /**
     * クローンオブジェクトの生成.
     * <BR><BR>
     * クローンオブジェクトの生成.
     * <BR>
     * @return ArrayTable クローンオブジェクトが返されます.
     */
    public final ReadProperty getClone()
    {
        ReadProperty ret = null ;
        
        ret = new ReadProperty() ;
        
        m_table.getCopyObject( ret.m_table ) ;
        
        return ret ;
    }
    
    /**
     * 生成チェック
     * <BR><BR>
     * プロパティ情報が生成されているかチェックします。
     * <BR>
     * @return boolean 情報が生成されているかチェック
     * 戻り値がtrueの場合、プロパティ情報は生成されています。
     * 戻り値がfalseの場合、プロパティ情報は生成されていません。
     */
    public final boolean isNew()
    {
        return m_isNew ;
    }
    
    /**
     * 解析処理.
     */
    private static final boolean analysis( ReadKey key,ArrayTable table )
        throws InputException,ReadSyntaxException
    {
        int i,nowNo ;
        int len ;
        int cnt ;
        boolean mode ;
        
        String tmp = null ;
        String keyName = null ;
        String dataTo = null ;
        
        if( key == null || table == null || ( len = key.size() ) == 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        // デフォルトキー情報のセット.
        keyName = ReadProperty.DEFAULT_KEY ;
        
        try{
            
            for( i = 0,mode = false,cnt = 0,nowNo = 0 ; i < len ; i ++ ){
                
                // 情報の取得.
                tmp = key.get( i ) ;
                
                // 情報が[=]の場合.
                if( tmp.equalsIgnoreCase( BaseDef.EQUAL ) == true ){
                    
                    mode = true ;
                    cnt = 0 ;
                    continue ;
                    
                }
                
                // モードがキー名の場合.
                if( mode == false ){
                    
                    // キー名が立て続けに設定される場合.
                    if( cnt >= 1 ){
                        throw new ReadSyntaxException(
                            "Key名(" + keyName + ")が重複して宣言されています(行:" +
                            key.getLine( nowNo ) +
                            " @@列 :" +
                            key.getSequence( nowNo ) +
                            ")::" + tmp + "." ) ;
                    }else{
                        cnt ++ ;
                    }
                    
                    keyName = null ;
                    keyName = new String( tmp ) ;
                    
                // モードがデータ情報の場合.
                }else{
                    
                    dataTo = null ;
                    dataTo = new String( tmp ) ;
                    
                    try{
                        table.add( keyName,dataTo ) ;
                    }catch( InputException in ){
                        throw new ReadSyntaxException(  
                            "対象のキー名(" + keyName +
                            ")及びデータ(" + dataTo +
                            ")が不正です(行:" +
                            key.getLine( nowNo ) +
                            " @@列 :" +
                            key.getSequence( nowNo ) +
                            ")" ) ;
                    }
                    mode = false ;
                    
                }
                
                nowNo ++ ;
            }
            
            return true ;
            
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }finally{
            tmp = null ;
            keyName = null ;
            dataTo = null ;
        }
    }
}
