/*
 * @(#)ReadIni.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.read;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.NotExistException;
import com.JRcServer.commons.exception.ReadSyntaxException;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ListDirectory;
import com.JRcServer.commons.util.ReadIndex;
import com.JRcServer.commons.util.UtilCom;

/**
 * INIファイル読み込み用オブジェクト.
 * <BR><BR>
 * INIファイルの読み込みをサポートします.
 *
 * @version 1.00, 2004/05/01
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ReadIni
{
    
    /**
     * デフォルトキーコード情報.
     */
    private static final String DEFAULT_KEY = "default" ;
    
    /**
     * セクションワード : 開始コード.
     */
    private static final char SECTION_ST = '[' ;
    
    /**
     * セクションワード : 終了コード.
     */
    private static final char SECTION_ED = ']' ;
    
    /**
     * コメントコード.
     */
    private static final String COMMENT_CD = ";#" ;
    
    /**
     * 次行リンクコード:string.
     */
    private static final String NEXT_LINE_CDSTR = "|" ;
    
    /**
     * 次行リンクコード.
     */
    private static final char NEXT_LINE_CD = ReadIni.NEXT_LINE_CDSTR.charAt(0) ;
    
    /**
     * キーコード参照値表現.
     */
    private static final char NEXT_VALUE = '=' ;
    
    /**
     * タブコード.
     */
    private static final char TAB_CD = '\t' ;
    
    /**
     * 半角スペースコード.
     */
    private static final char SPC_CD = ' ' ;
    
    /**
     * 全角スペースコード.
     */
    private static final char SPC2_CD = '　' ;
    
    /**
     * シングルコーテーション.
     */
    private static final char SNGL_COTE = '\'' ;
    
    /**
     * ダブルコーテーション.
     */
    private static final char DBL_COTE = '\"' ;
    
    
    /**
     * INI読み込みモード : 読み込み中.
     */
    private static final int MODE_NON = 0 ;
    
    /**
     * INI読み込みモード : セクション参照.
     */
    private static final int MODE_SECTION = 1 ;
    
    /**
     * INI読み込みモード : キーコード参照.
     */
    private static final int MODE_KEY = 2 ;
    
    /**
     * INI読み込みモード : キー参照コード参照.
     */
    private static final int MODE_VALUE = 3 ;
    
    
    
    /**
     * INIコーテーションモード : コーテーションなし.
     */
    private static final int MODE_COTE_NON = 0 ;
    
    /**
     * INIコーテーションモード : シングルコーテーション.
     */
    private static final int MODE_COTE_SINGLE = 1 ;
    
    /**
     * INIコーテーションモード : ダブルコーテーション.
     */
    private static final int MODE_COTE_DOUBLE = 2 ;
    
    
    /**
     * INIファイル管理仮想ディレクトリ.
     */
    private ListDirectory m_vdir = null ;
    
    /**
     * 書き込み先ディレクトリ位置.
     */
    private String m_pwd = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public ReadIni()
    {
        m_vdir = new ListDirectory() ;
        m_pwd = ListDirectory.PATH ;
    }
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * INIファイル管理仮想ディレクトリ情報を設定します.
     * <BR>
     * @param vdir 仮想ディレクトリ情報を設定します.
     * @param pwd 書き込み位置ディレクトリパスを設定します.
     * @exception InputException 入力例外.
     */
    public ReadIni( ListDirectory vdir,String pwd )
        throws InputException
    {
        
        if( vdir == null ){
            throw new InputException( "第一引数は不正です" ) ;
        }
        
        m_vdir = vdir ;
        m_pwd = pwd ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象のIndex情報から、情報を取得します.
     * <BR>
     * @param index INI情報取得対象のINDEX情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public final void create( ReadIndex index )
        throws InputException,ReadSyntaxException
    {
        
        try{
            
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                
                ReadIni.analisys_Ini( m_vdir,index,m_pwd ) ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException re ){
            throw re ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.<BR>
     * また、コンストラクタ[ReadIni( ListDirectory,String )]で
     * オブジェクトを生成している場合、この処理を呼び出すと、
     * 第一引数で渡した情報内容もがクリアされてしまいます.
     */
    public final void clear()
    {
        try{
            m_vdir.destroy() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 仮想ディレクトリ情報の取得.
     * <BR><BR>
     * 対象の仮想ディレクトリ情報を取得します.
     * <BR>
     * @return ListDirectory 管理されている仮想ディレクトリ情報が返されます.
     */
    public final ListDirectory getVirtualDirectory()
    {
        
        ListDirectory ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_vdir ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報を取得.
     * <BR><BR>
     * INI情報として、格納されているキーワードに対する
     * 情報を取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.<BR>
     *                [null]を設定した場合、セクション名を設定しないことを
     *                意味します.
     * @param key 対象のキーワード情報を設定します.
     * @param no 対象キーワードに対する取得項番を設定します.
     * @return String 格納されている情報が返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 情報非存在例外.
     */
    public final String get( String section,String key,int no )
        throws InputException,NotExistException
    {
        String ret = null ;
        
        if( key == null || no < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( section != null ){
                synchronized( m_sync.get() ){
                    m_vdir.upper( section ) ;
                    ret = ( String )m_vdir.getPause( key,no ) ;
                    m_vdir.lower() ;
                }
            }
            else{
                synchronized( m_sync.get() ){
                    ret = ( String )m_vdir.getPause( key,no ) ;
                }
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報を取得.
     * <BR><BR>
     * INI情報として、格納されているキーワードに対する
     * 全ての情報を取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.<BR>
     *                [null]を設定した場合、セクション名を設定しないことを
     *                意味します.
     * @param key 対象のキーワード情報を設定します.
     * @return String[] 格納されている情報が返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 情報非存在例外.
     */
    public final String[] getAll( String section,String key )
        throws InputException,NotExistException
    {
        int i ;
        int len ;
        
        String[] ret = null ;
        Object[] obj = null ;
        
        if( key == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( section != null ){
                synchronized( m_sync.get() ){
                    m_vdir.upper( section ) ;
                    obj = m_vdir.getAllPause( key ) ;
                    m_vdir.lower() ;
                }
            }
            else{
                synchronized( m_sync.get() ){
                    obj = m_vdir.getAllPause( key ) ;
                }
            }
            
            len = obj.length ;
            ret = new String[ len ] ;
            
            for( i = 0 ; i < len ; i ++ ){
                ret[ i ] = ( String )obj[ i ] ;
                obj[ i ] = null ;
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            obj = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * セクション名群の取得.
     * <BR><BR>
     * 対象のセクション名群を取得します.
     * <BR>
     * @return String[] 対象のセクション名群が返されます.
     */
    public final String[] getSections()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_vdir.getDirectorys() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション名数の取得.
     * <BR><BR>
     * 対象のセクション名数を取得します.
     * <BR>
     * @return int 対象のセクション名数が返されます.
     */
    public final int getSectionElements()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_vdir.getDirectoryCount() ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * キーワード名群の取得.
     * <BR><BR>
     * 対象のキーワード名群を取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.<BR>
     *                [null]を設定した場合、セクション名を設定しないことを
     *                意味します.
     * @return String[] 対象のキーワード名群が返されます.
     * @exception NotExistException 非存在例外.
     */
    public final String[] getKeywords( String section )
        throws NotExistException
    {
        String[] ret = null ;
        
        try{
            
            if( section != null ){
                synchronized( m_sync.get() ){
                    m_vdir.upper( section ) ;
                    ret = m_vdir.getPauses() ;
                    m_vdir.lower() ;
                }
            }
            else{
                synchronized( m_sync.get() ){
                    ret = m_vdir.getPauses() ;
                }
            }
            
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション内キーワード数の取得.
     * <BR><BR>
     * 対象のセクション内のキーワード数を取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.<BR>
     *                [null]を設定した場合、セクション名を設定しないことを
     *                意味します.
     * @return int 格納キーワード数が返されます.
     * @exception NotExistException 非存在例外.<BR>
     *                              この例外が返された場合、対象のセクション名は
     *                              存在しません.
     */
    public final int getKeywordElements( String section )
        throws NotExistException
    {
        int ret ;
        
        try{
            
            if( section != null ){
                synchronized( m_sync.get() ){
                    m_vdir.upper( section ) ;
                    ret = m_vdir.getPauseCount() ;
                    m_vdir.lower() ;
                }
            }
            else{
                synchronized( m_sync.get() ){
                    ret = m_vdir.getPauseCount() ;
                }
            }
            
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象キーワード名存在チェック.
     * <BR><BR>
     * 対象のキーワード名が存在するかチェックします.<BR>
     * また、存在する場合、その個数を返します.
     * <BR>
     * @param section 対象のセクション名を設定します.<BR>
     *                [null]を設定した場合、セクション名を設定しないことを
     *                意味します.
     * @param key 対象のキーワード名を設定します.
     * @return int 格納キーワード数が返されます.
     * @exception NotExistException 非存在例外.<BR>
     *                              この例外が返された場合、対象のセクション名は
     *                              存在しません.
     */
    public final int getElements( String section,String key )
        throws NotExistException
    {
        int ret ;
        
        try{
            
            if( section != null ){
                synchronized( m_sync.get() ){
                    m_vdir.upper( section ) ;
                    ret = m_vdir.getPauseElements( key ) ;
                    m_vdir.lower() ;
                }
            }
            else{
                synchronized( m_sync.get() ){
                    ret = m_vdir.getPauseElements( key ) ;
                }
            }
            
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象のセクション名が存在するかチェックします.
     * <BR>
     * @param section チェック対象セクション名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象セクション名は存在します.<BR>
     *                 [false]が返された場合、対象セクション名は存在しません.   
     */
    public final boolean isSection( String section )
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_vdir.isDirectory( section ) ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象キーワード名存在チェック.
     * <BR><BR>
     * 対象のキーワード名が存在するかチェックします.<BR>
     * また、存在する場合、その個数を返します.
     * <BR>
     * @param section 対象のセクション名を設定します.<BR>
     *                [null]を設定した場合、セクション名を設定しないことを
     *                意味します.
     * @param key 対象のキーワード名を設定します.
     * @return boolean キーワード存在結果が返されます.<BR>
     *                 [true]が返された場合、対象セクション名は存在します.<BR>
     *                 [false]が返された場合、対象セクション名は存在しません.   
     */
    public final boolean isKeyword( String section,String key )
    {
        int len ;
        boolean ret ;
        
        try{
            
            if( section != null ){
                synchronized( m_sync.get() ){
                    m_vdir.upper( section ) ;
                    len = m_vdir.getPauseElements( key ) ;
                    m_vdir.lower() ;
                }
                
            }
            else{
                synchronized( m_sync.get() ){
                    len = m_vdir.getPauseElements( key ) ;
                }
            }
            
            ret = ( len != 0 ) ? true : false ;
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * INIファイル解析.
     */
    private static final void analisys_Ini( ListDirectory out,ReadIndex index,String pwd )
        throws InputException,ReadSyntaxException
    {
        int i,j,k ;
        int len ;
        int lenJ ;
        int lenK ;
        int mode ;
        int mode_cote ;
        int line,count ;
        int oneLineNextCd ;
        boolean exit_line ;
        boolean link_line ;
        boolean lastLinkLine ;
        boolean nextcheck ;
        
        char nowWd ;
        
        char[] oneLines = null ;
        StringBuffer buf = null ;
        String nowSection = null ;
        String nowKey = null ;
        String nowWord = null ;
        String lastKey = null ;
        
        String tmp = null ;
        
        if(
            out == null ||
            index == null ||
            ( len = index.size() ) == ReadIndex.NOT_INDEX
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        line = 0 ;
        count = 0 ;
        mode_cote = ReadIni.MODE_COTE_NON ;
        
        try{
            
            // 格納ディレクトリ位置が指定されていない場合.
            if( pwd == null ){
                
                // ROOT領域として、セット.
                out.change( ListDirectory.PATH ) ;
                nowSection = null ;
                
            }
            // 格納ディレクトリ位置が指定されている場合.
            else{
                
                pwd = UtilCom.trimPlus( pwd ) ;
                
                if( out.isPath( pwd ) == false ){
                    throw new InputException(
                        "対象のパス情報(" + pwd +
                        ")は正しくないディレクトリ構成です"
                    ) ;
                }
                
                out.change( pwd ) ;
                nowSection = null ;
                
            }
            
            mode = ReadIni.MODE_NON ;
            lastKey = ReadIni.DEFAULT_KEY ;
            lastLinkLine = false ;
            
            // 全体読み込み.
            for( i = 0 ; i < len ; i ++ ){
                
                // 行番号を定義.
                line = i + 1 ;
                
                tmp = ( String )index.getIndex( i ) ;
                oneLineNextCd = tmp.indexOf( ReadIni.NEXT_LINE_CDSTR ) ;
                
                // 内容が存在しない場合.
                if( ( lenJ = tmp.length() ) <= 0 ){
                    continue ;
                }
                
                // １行情報の読み込み.
                oneLines = new char[ lenJ ] ;
                tmp.getChars( 0,lenJ,oneLines,0 ) ;
                
                mode_cote = MODE_COTE_NON ;
                exit_line = false ;
                link_line = false ;
                
                // １行情報解析.
                for( j = 0 ; j < lenJ ; j ++ ){
                    
                    // 列番号を定義.
                    count = j + 1 ;
                    
                    // 対象位置の１文字情報を取得.
                    nowWd = oneLines[ j ] ;
                    
                    // コーテーション括りでない場合の処理.
                    if( mode_cote == ReadIni.MODE_COTE_NON ){
                        
                        // コメントが存在する場合.
                        if( ReadIni.isCode( nowWd,ReadIni.COMMENT_CD ) == true ){
                            
                            link_line = false ;
                            break ;
                            
                        }
                        // 次行リンク指定の場合.
                        else if(
                            mode != ReadIni.MODE_SECTION &&
                            nowWd == ReadIni.NEXT_LINE_CD
                        )
                        {
                            link_line = true ;
                            break ;
                        }
                        
                    }
                    
                    // 前回の行情報と連結している場合.
                    if( lastLinkLine == true ){
                        
                        // コーテーション区切りの場合.
                        if(
                            mode == ReadIni.MODE_VALUE &&
                            (
                                nowWd == ReadIni.SNGL_COTE ||
                                nowWd == ReadIni.DBL_COTE
                            )
                        )
                        {
                            
                            mode_cote = ( nowWd == ReadIni.SNGL_COTE ) ?
                                ReadIni.MODE_COTE_SINGLE : ReadIni.MODE_COTE_DOUBLE ;
                            lastLinkLine = false ;
                            
                        }
                        // コーテーション区切り以外の場合.
                        // モード条件が存在しない場合.
                        else if(
                            mode == ReadIni.MODE_NON &&
                            (
                                ( nowWd >= 'a' && nowWd <= 'z' ) ||
                                ( nowWd >= 'A' && nowWd <= 'Z' ) ||
                                nowWd == ReadIni.SECTION_ST
                            )
                        )
                        {
                            
                            buf = new StringBuffer() ;
                            
                            // セクション情報開始の場合.
                            if( nowWd == ReadIni.SECTION_ST ){
                                mode = ReadIni.MODE_SECTION ;
                            }
                            // キーワード情報読み込み開始の場合.
                            else{
                                
                                mode = ReadIni.MODE_KEY ;
                                buf.append( nowWd ) ;
                                
                            }
                            
                            lastLinkLine = false ;
                            
                        }
                        // ワード情報検出の場合.
                        else if(
                            mode == ReadIni.MODE_VALUE &&
                            (
                                nowWd != ReadIni.TAB_CD &&
                                nowWd != ReadIni.SPC_CD
                            )
                        )
                        {
                            
                            buf.append( nowWd ) ;
                            lastLinkLine = false ;
                            
                        }
                        
                        continue ;
                        
                    }
                    
                    // 各モードセット.
                    switch( mode ){
                        
                        // 読み込み状態.
                        case ReadIni.MODE_NON :
                            
                            if( nowWd == ReadIni.SECTION_ST ){
                                
                                buf = new StringBuffer() ;
                                mode = ReadIni.MODE_SECTION ;
                                
                            }
                            // 文字列が英字の場合.
                            else if(
                                ( nowWd >= 'a' && nowWd <= 'z' ) ||
                                ( nowWd >= 'A' && nowWd <= 'Z' )
                            )
                            {
                                // キーコード対象とする.
                                mode = ReadIni.MODE_KEY ;
                                buf = new StringBuffer() ;
                                buf.append( nowWd ) ;
                                
                            }
                            // 以前のキーコードとしての値検知.
                            else if( nowWd == ReadIni.NEXT_VALUE ){
                                
                                nowKey = lastKey ;
                                buf = new StringBuffer() ;
                                mode = ReadIni.MODE_VALUE ;
                                
                            }
                            // 文字列が英字,タブ,スペース以外の場合.
                            else if(
                                nowWd != ReadIni.TAB_CD &&
                                nowWd != ReadIni.SPC_CD &&
                                nowWd != SPC2_CD
                            )
                            {
                                // 読み込み内容が不正.
                                throw new ReadSyntaxException(
                                    "シンタックスエラー(" + tmp + ")が検出されました"
                                ) ;
                            }
                            
                            break ;
                            
                        // セクション名読み込み状態.
                        case ReadIni.MODE_SECTION :
                            
                            if( nowWd == ReadIni.SECTION_ED ){
                                
                                nowSection = UtilCom.trimPlus( buf.toString() ) ;
                                lastKey = ReadIni.DEFAULT_KEY ;
                                buf = null ;
                                mode = ReadIni.MODE_NON ;
                                
                            }
                            else{
                                
                                if(
                                    nowWd == ReadIni.NEXT_LINE_CD ||
                                    nowWd == ReadIni.SNGL_COTE ||
                                    nowWd == ReadIni.DBL_COTE
                                )
                                {
                                    // 読み込み内容が不正.
                                    throw new ReadSyntaxException(
                                        "セクション名内に不正なコードを検出(" + tmp + ")しました"
                                    ) ;
                                }
                                
                                buf.append( nowWd ) ;
                                
                            }
                            
                            break ;
                            
                        // キーコード読み込み状態.
                        case ReadIni.MODE_KEY :
                            
                            if( nowWd == ReadIni.NEXT_VALUE ){
                                
                                nowKey = UtilCom.trimPlus( buf.toString() ) ;
                                lastKey = nowKey ;
                                buf = null ;
                                buf = new StringBuffer() ;
                                mode = ReadIni.MODE_VALUE ;
                                
                            }
                            else{
                                
                                if(
                                    nowWd == ReadIni.SNGL_COTE ||
                                    nowWd == ReadIni.DBL_COTE
                                )
                                {
                                    // 読み込み内容が不正.
                                    throw new ReadSyntaxException(
                                        "キーコード内に不正なコードを検出(" + tmp + ")しました"
                                    ) ;
                                }
                                
                                buf.append( nowWd ) ;
                                
                            }
                            
                            break ;
                            
                        // ワード情報読み込み状態.
                        case ReadIni.MODE_VALUE :
                            
                            // コーテーション区切りの場合.
                            if(
                                nowWd == ReadIni.SNGL_COTE ||
                                nowWd == ReadIni.DBL_COTE
                            )
                            {
                                // シングルコーテーションが検出された場合.
                                if( nowWd == ReadIni.SNGL_COTE ){
                                    
                                    // ダブルコーテーションが存在
                                    if( mode_cote == ReadIni.MODE_COTE_DOUBLE ){
                                        
                                        // 既にシングルコーテーションが検出されたため
                                        // ダブルコーテーションは通常文字列として
                                        // 有効とする.
                                        buf.append( nowWd ) ;
                                        break ;
                                        
                                    }
                                    
                                    mode_cote = ( mode_cote == ReadIni.MODE_COTE_NON ) ?
                                        ReadIni.MODE_COTE_SINGLE : ReadIni.MODE_COTE_NON ;
                                    
                                }
                                // ダブルコーテーションが検出された場合.
                                else{
                                    
                                    // シングルコーテーションが存在
                                    if( mode_cote == ReadIni.MODE_COTE_SINGLE ){
                                        
                                        // 既にダブルコーテーションが検出されたため
                                        // シングルコーテーションは通常文字列として
                                        // 有効とする.
                                        buf.append( nowWd ) ;
                                        break ;
                                        
                                    }
                                    
                                    mode_cote = ( mode_cote == ReadIni.MODE_COTE_NON ) ?
                                        ReadIni.MODE_COTE_DOUBLE : ReadIni.MODE_COTE_NON ;
                                    
                                }
                                
                            }
                            // コーテーションが検出された場合.
                            else if( mode_cote != MODE_COTE_NON ){
                                
                                buf.append( nowWd ) ;
                                
                            }
                            // コーテーションが検出されていない場合.
                            else{
                                
                                // ワード情報終端が検出された場合.
                                if(
                                    ( buf != null && buf.length() != 0 ) &&
                                    (
                                        nowWd == ReadIni.TAB_CD ||
                                        nowWd == ReadIni.SPC_CD
                                    )
                                )
                                {
                                    nextcheck = false ;
                                    
                                    // ￥コードが、この行中に存在する場合.
                                    if( oneLineNextCd > j ){
                                        nextcheck = true ;
                                        for( k = j+1 ; k < oneLineNextCd ; k ++ ){
                                            if(
                                                oneLines[ k ] != ReadIni.TAB_CD &&
                                                oneLines[ k ] != ReadIni.SPC_CD 
                                            )
                                            {
                                                nextcheck = false ;
                                                break ;
                                            }
                                        }
                                    }
                                    
                                    if( nextcheck == false ){
                                        
                                        nowWord = ( buf == null || buf.length() <= 0 ) ?
                                            "" : buf.toString() ;
                                        mode = ReadIni.MODE_NON ;
                                        //exit_line = true ;
                                    }
                                    else{
                                        j = oneLineNextCd - 1 ;
                                        break ;
                                    }
                                    
                                }
                                // ワード情報が検出された場合.
                                else{
                                    
                                    buf.append( nowWd ) ;
                                    
                                }
                                
                            }
                            
                            break ;
                    }
                    
                    // 行終了処理の場合.
                    //if( exit_line == true ){
                    //  break ;
                    //}
                    
                }
                
                // コーテーション括りのまま、行終了の場合.
                if( mode_cote != ReadIni.MODE_COTE_NON ){
                    // 読み込み内容が不正.
                    throw new ReadSyntaxException(
                        "コーテーションの終端(" + tmp + ")が存在しません"
                    ) ;
                }
                // セクション名読み込み最中の場合.
                else if( mode == ReadIni.MODE_SECTION ){
                    // 読み込み内容が不正.
                    throw new ReadSyntaxException(
                        "セクション名の終端(" + tmp + ")が存在しません"
                    ) ;
                }
                // 次行情報に跨らない場合.
                else if( link_line != true ){
                    
                    // 格納対象情報が存在する場合.
                    if( nowKey != null ){
                        
                        nowWord = ( buf == null || buf.length() <= 0 ) ?
                            "" : buf.toString() ;
                        buf = null ;
                        
                        mode = ReadIni.MODE_NON ;
                        ReadIni.settingDirectory( out,nowSection,nowKey,nowWord ) ;
//System.out.println( "nowSection:" + nowSection + " nowKey:" + nowKey + " nowWord:" + nowWord ) ;
                        lastKey = nowKey ;
                        nowKey = null ;
                        nowWord = null ;
                        
                    }
                    
                }
                
                // 次行情報跨り判別情報を、バックアップ.
                lastLinkLine = link_line ;
                
            }
            
            // 情報を読み取り中の場合.
            if( nowKey != null || mode_cote == ReadIni.MODE_COTE_NON ){
                
                nowWord = ( buf == null || buf.length() <= 0 ) ?
                    "" : buf.toString() ;
                buf = null ;
                
                ReadIni.settingDirectory( out,nowSection,nowKey,nowWord ) ;
//System.out.println( "nowSection:" + nowSection + " nowKey:" + nowKey + " nowWord:" + nowWord ) ;

                mode = ReadIni.MODE_NON ;
            }
            else if( mode == ReadIni.MODE_SECTION || mode_cote != ReadIni.MODE_COTE_NON ){
                if( mode == ReadIni.MODE_SECTION ){
                    // 読み込み内容が不正.
                    throw new ReadSyntaxException(
                        "セクション名の終端(" + tmp + ")が存在しません"
                    ) ;
                }
                else{
                    // 読み込み内容が不正.
                    throw new ReadSyntaxException(
                        "コーテーションの終端(" + tmp + ")が存在しません"
                    ) ;
                }
            }
            
        }catch( ReadSyntaxException re ){
            re.addMessage(
                BaseDef.ENTER_SEPARATOR + "(行:" + line + " 列:" + count + " )"
            ) ;
            throw re ;
        }catch( BaseException be ){
            ReadSyntaxException rs = null ;
            rs = new ReadSyntaxException( be ) ;
            rs.addMessage(
                BaseDef.ENTER_SEPARATOR + "(行:" + line + " 列:" + count + " )"
            ) ;
            throw rs ;
        }catch( Exception t ){
            ReadSyntaxException rs = null ;
            rs = new ReadSyntaxException( t ) ;
            rs.addMessage(
                BaseDef.ENTER_SEPARATOR + "(行:" + line + " 列:" + count + " )"
            ) ;
            throw rs ;
        }finally{
            
            oneLines = null ;
            buf = null ;
            nowSection = null ;
            nowKey = null ;
            nowWord = null ;
            lastKey = null ;
            tmp = null ;
            
        }
        
    }
    
    
    /**
     * INI情報の格納.
     */
    private static final void settingDirectory(
        ListDirectory out,String section,String key,String value
    )
        throws BaseException
    {
//System.out.println( "[" + section + "][" + key + "][" + value + "]" ) ;
        
        if( key == null ){
            return ;
        }
        
        value = ( value == null ) ? "" : value ;
        
        try{
            
            if( section != null ){
                
                if( out.isDirectory( section ) == false ){
                    out.createDirectory( section,null ) ;
                }
                
                out.upper( section ) ;
                out.createPause( key,value ) ;
                out.lower() ;
                
            }
            else{
                
                out.createPause( key,value ) ;
                
            }
            
        }catch( BaseException be ){
            throw be ;
        }catch( Exception t ){
            throw new BaseException( t ) ;
        }
        
    }
    
    /**
     * 指定文字列を、対象文字コードが一致するかチェック.
     */
    private static final boolean isCode( char code,String checks )
    {
        int i ;
        int len ;
        
        boolean ret ;
        
        len = checks.length() ;
        for( i = 0,ret = false ; i < len ; i ++ ){
            if( code == checks.charAt( i ) ){
                ret = true ;
                break ;
            }
        }
        
        return ret ;
    }
    
    
}

