/*
 * @(#)ConvertToIndexNo.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.convert;

import java.math.BigDecimal;
import java.util.Date;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.DateTimeFormat;

/**
 * 情報コンバート実装.
 * <BR><BR>
 * 情報をコンバートする場合の実装内容を提供します.
 * 取得方法として、インデックス項番を用いて情報取得します.
 *
 * @version 1.00, 2003/11/04
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public abstract class ConvertToIndexNo
{
    /**
     * DateTime日付変換用.
     */
    private final DateTimeFormat m_format = new DateTimeFormat() ;
    
    /**
     * コンストラクタ.
     */
    protected ConvertToIndexNo()
    {}
    
    /**
     * 対象情報取得処理.
     * <BR><BR>
     * 取得対象の処理を実施します.
     * この処理は、取得対象の処理で渡される処理を
     * 定義します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return String 取得対象の情報が返されます.
     * @exception InputException 入力例外.
     */
    protected abstract Object getStringByNumber( int no )
        throws InputException ;
    
    /**
     * 対象情報設定処理.
     * <BR><BR>
     * 設定対象の処理を実施します.
     * この処理は、設定対象の処理で渡される処理を
     * 定義します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param stream 設定対象の内容を指定します.
     * @exception InputException 入力例外.
     */
    protected abstract void setStringByNumber( int no,String stream )
        throws InputException ;
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を文字情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return String 取得情報に対する内容が返されます.
     */
    public String getString( int no )
    {
        String ret = null ;
        
        try{
            ret = ( String )this.getStringByNumber( no ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を日付・時間情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @param format 読み込み対象の日付フォーマットを設定します.
     * @return Date 取得対象の日付オブジェクトが返されます.
     */
    public Date getDate( int no,String format )
    {
        Date ret = null ;
        
        try{
            
            synchronized( m_format ){
                m_format.create( format ) ;
                m_format.set( ( String )this.getStringByNumber( no ) ) ;
                ret = new Date( m_format.get().getTime() ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報をBigDecimal情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return BigDecimal 取得情報に対する内容が返されます.
     */
    public BigDecimal getBigDecimal( int no )
    {
        BigDecimal ret = null ;
        
        try{
            ret = new BigDecimal( ( String )this.getStringByNumber( no ) ) ;
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を整数情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return Short 取得情報に対する内容が(shortで)返されます.
     */
    public Short getShort( int no )
    {
        Short ret = null ;
        
        try{
            ret = new Short( ( String )this.getStringByNumber( no ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を整数情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return Integer 取得情報に対する内容が(intで)返されます.
     */
    public Integer getInteger( int no )
    {
        Integer ret = null ;
        
        try{
            ret = new Integer( ( String )this.getStringByNumber( no ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を整数情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return Long 取得情報に対する内容が(longで)返されます.
     */
    public Long getLong( int no )
    {
        Long ret = null ;
        
        try{
            ret = new Long( ( String )this.getStringByNumber( no ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を小数点を含む数値情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return Float 取得情報に対する内容が返されます.
     */
    public Float getFloat( int no )
    {
        Float ret = null ;
        
        try{
            ret = new Float( ( String )this.getStringByNumber( no ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を小数点を含む数値情報で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return Double 取得情報に対する内容が返されます.
     */
    public Double getDouble( int no )
    {
        Double ret = null ;
        
        try{
            ret = new Double( ( String )this.getStringByNumber( no ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を判定を行うブール変数で取得します.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return Boolean 取得情報に対する内容が返されます.
     */
    public Boolean getBoolean( int no )
    {
        String code = null ;
        Boolean ret = null ;
        
        try{
            
            code = ( String )this.getStringByNumber( no ) ;
            
            if( code.equalsIgnoreCase( BaseDef.BOOLEAN_TRUE ) == true ){
                ret = new Boolean( true ) ;
            }else if( code.equalsIgnoreCase( BaseDef.BOOLEAN_FALSE ) == true ){
                ret = new Boolean( false ) ;
            }else{
                ret = null ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            code = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を文字情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の文字情報を設定します.
     */
    public void setString( int no,String in )
    {
        try{
            this.setStringByNumber( no,in ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を日付・時間情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param format 対象の日付フォーマットを指定します.
     * @param in 設定対象の日付・時間情報を設定します.
     */
    public void setDate( int no,String format,Date in )
    {
        try{
            
            synchronized( m_format ){
                m_format.create( format ) ;
                m_format.set( in.getTime() ) ;
                this.setStringByNumber( no,m_format.toString() ) ;
            }
            
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報をBigDecimal情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象のBigDecimal情報を設定します.
     */
    public void setBigDecimal( int no,BigDecimal in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setShort( int no,short in )
    {
        try{
            this.setStringByNumber( no,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setShort( int no,Short in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setInteger( int no,int in )
    {
        try{
            this.setStringByNumber( no,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setInteger( int no,Integer in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setLong( int no,long in )
    {
        try{
            this.setStringByNumber( no,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setLong( int no,Long in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setFloat( int no,float in )
    {
        try{
            this.setStringByNumber( no,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setFloat( int no,Float in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setDouble( int no,double in )
    {
        try{
            this.setStringByNumber( no,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setDouble( int no,Double in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報をBoolean情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setBoolean( int no,boolean in )
    {
        try{
            this.setStringByNumber( no,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報をBoolean情報で設定します.
     * <BR>
     * @param no 設定対象の項番を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setBoolean( int no,Boolean in )
    {
        try{
            this.setStringByNumber( no,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
}

