/*
 * @(#)FlagBox.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.box;

import java.util.Arrays;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * フラグ固定配列.
 * <BR><BR>
 * フラグ固定配列を管理します.
 *  
 * @version 1.0.0 2004/10/27
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class FlagBox implements BaseBox
{
	
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            FlagBox.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * フラグ１要素数.
     */
    private static final int FLAG_ELEMENT_LENGTH = BoxDef.ELEMENT_LENGTH ;
    
    /**
     * フラグ１要素内単位.
     */
    private static final int FLAG_ELEMENT = BoxDef.ELEMENT_MASK ;
    
    /**
     * フラグ要素数.
     */
    private static final int FLAG_PAUSE = BoxDef.ELEMENT_PAUSE ;
    
    
    
    /**
     * フラグ管理.
     */
    private long[] m_flags = null ;
    
    /**
     * フラグ個数.
     */
    private int m_size = 0 ;
    
    /**
     * 有効フラグ個数.
     */
    private int m_flgOnLength = 0 ;
    
    /**
     * 次追加位置.
     */
    private int m_nextPoint = 0 ;
    
    /**
     * 追加ラスト位置.
     */
    private int m_lastPoint = 0 ;
    
    
    
    /**
     * コンストラクタ.
     */
    public FlagBox()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 管理対象のフラグ管理長を設定します.
     * <BR>
     * @param size 管理対象のフラグ長を設定します.
     * @exception InputException 入力例外.
     */
    public FlagBox( int size )
        throws InputException
    {
        this.create( size ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 管理対象のフラグ管理長を設定します.
     * <BR>
     * @param size 管理対象のフラグ長を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( int size )
        throws InputException
    {
        int no ;
        
        if( size < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        no = ( ( ( size & FlagBox.FLAG_ELEMENT ) != 0 ) ? 1 : 0 ) +
            ( ( size & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ) ;
        
        m_flags = new long[ no ] ;
        Arrays.fill( m_flags,0L ) ;
        
        m_size = no * FlagBox.FLAG_ELEMENT_LENGTH ;
        m_flgOnLength = 0 ;
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_flags = null ;
        m_size = 0 ;
        m_flgOnLength = 0 ;
        m_nextPoint = 0 ;
        m_lastPoint = 0 ;
    }
    
    /**
     * 領域を追加.
     * <BR><BR>
     * 領域を追加します.
     * <BR>
     * @param size 追加対象のサイズを設定します.
     * @return int 追加された後の全体サイズが返されます.
     */
    public final int addSpace( int size )
    {
        int no ;
        int newSize ;
        int ret ;
        
        long[] tmp = null ;
        
        if( size > 0 && m_flags != null ){
            
            newSize = size + m_size ;
            no = ( ( ( newSize & FlagBox.FLAG_ELEMENT ) != 0 ) ? 1 : 0 ) +
                ( ( newSize & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ) ;
            
            tmp = new long[ no ] ;
            Arrays.fill( tmp,0L ) ;
            
            System.arraycopy( m_flags,0,tmp,0,m_flags.length ) ;
            m_flags = tmp ;
            
            m_size = no * FlagBox.FLAG_ELEMENT_LENGTH ;
            ret = newSize ;
            
        }
        else{
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 領域を削除.
     * <BR><BR>
     * 領域を削除します.
     * <BR>
     * @param size 削除対象のサイズを設定します.
     * @return int 削除された後の全体サイズが返されます.
     */
    public final int removeSpace( int size )
    {
        int no ;
        int no2 ;
        int insSize ;
        int lastSize ;
        int remSize ;
        int newSize ;
        int ret ;
        
        long[] tmp = null ;
        
        if( size > 0 && m_flags != null ){
            
            lastSize = m_lastPoint + 1 ;
            remSize = m_size - size ;
            
            no = ( ( ( remSize & FlagBox.FLAG_ELEMENT ) != 0 ) ? 1 : 0 ) +
                ( ( remSize & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ) ;
            no2 = ( ( ( lastSize & FlagBox.FLAG_ELEMENT ) != 0 ) ? 1 : 0 ) +
                ( ( lastSize & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ) ;
            
            insSize = ( lastSize <= m_size ) ? no2 : no ;
            
            if( m_flags.length > insSize ){
                
                newSize = insSize * FlagBox.FLAG_ELEMENT_LENGTH ;
                ret = m_size = newSize ;
                
                tmp = new long[ insSize ] ;
                Arrays.fill( tmp,0L ) ;
                System.arraycopy( m_flags,0,tmp,0,tmp.length ) ;
                m_flags = tmp ;
                
            }
            else{
                ret = m_size ;
            }
            
        }
        else{
            ret = m_size ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素を有効/無効に設定.
     * <BR><BR>
     * 対象の要素を有効/無効に設定します.
     * <BR>
     * @param flg 対象の条件を設定します.<BR>
     *            [true]を設定した場合有効となります.<BR>
     *            [false]を設定した場合無効となります.
     * @param no 設定対象の項番を設定します.
     */
    public final void add( boolean flg,int no )
    {
        int indNo ;
        int indOff ;
        
        long[] flags = null ;
        
        flags = m_flags ;
        
        if( flags == null || no < 0 || no >= m_size ){
            return ;
        }
        
        indNo = ( no & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ;
        indOff = no & FlagBox.FLAG_ELEMENT ;
        
        if( flg == true ){
            
            if( ( flags[ indNo ] & ( 1L << indOff ) ) == 0 ){
                m_flgOnLength ++ ;
            }
            
            flags[ indNo ] |= ( ( 1L << indOff ) & 0xffffffffffffffffL ) ;
            
            m_nextPoint = indNo ;
            m_lastPoint = ( m_lastPoint < no ) ? no : m_lastPoint ;
            
        }
        else{
            
            if( ( flags[ indNo ] & ( 1L << indOff ) ) != 0 ){
                m_flgOnLength -- ;
            }
            
            flags[ indNo ] &= ( ( ~( 1L << indOff ) ) & 0xffffffffffffffffL ) ;
            
            m_nextPoint = ( m_nextPoint >= indNo ) ? indNo : m_nextPoint ;
            this.getLastCode() ;
            
        }
        
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象の要素を取得します.
     * <BR>
     * @param no 取得対象の要素番号を設定します.
     * @return boolean 取得された内容が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public final boolean get( int no )
    {
        long[] flags = null ;
        
        flags = m_flags ;
        
        if( flags == null || no < 0 || no >= m_size ){
            return false ;
        }
        
        return (
            (
                flags[ ( no & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ] &
                ( 1L << ( no & FlagBox.FLAG_ELEMENT ) ) 
            ) != 0
        ) ? true : false ;
    }
    
    /**
     * 有効領域を検索.
     * <BR><BR>
     * 有効領域を検索します.
     * <BR>
     * @return int 有効な項番が返されます.<BR>
     *             空き領域が存在しない場合[-1]が返されます.
     */
    public final int getSearch()
    {
        int i,j ;
        int len ;
        int max ;
        int size ;
        boolean flg ;
        int ret ;
        
        long[] flags = null ;
        
        ret = -1 ;
        size = 0 ;
        
        size = m_size ;
        flags = m_flags ;
        
        if( flags == null ){
            return -1 ;
        }
        
        len = FlagBox.FLAG_ELEMENT_LENGTH ;
        max = flags.length ;
        
        // 情報を検索.
        for( i = m_nextPoint,ret = -1,flg = false ; ( i < max && flg == false ) ; i ++ ){
            
            // 空き領域が存在する場合.
            if( flags[ i ] != 0xffffffffffffffffL ){
                
                // 空き条件を調べる.
                for( j = 0 ; j < len ; j ++ ){
                    
                    // 空きデータが検知された場合.
                    if( ( flags[ i ] & ( 1L << j ) ) == 0 ){
                        
                        ret = ( i * FlagBox.FLAG_ELEMENT_LENGTH ) + j ;
                        
                        if( ret >= size ){
                            ret = -1 ;
                        }
                        else{
                            flg = true ;
                        }
                        
                        break ;
                        
                    }
                    
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 管理フラグ個数を取得.
     * <BR><BR>
     * 管理されているフラグ個数を取得します.
     * <BR>
     * @return int 管理されているフラグ個数が返されます.
     */
    public final int getMaxSize()
    {
        return m_size ;
    }
    
    /**
     * 有効となっているフラグ個数を取得.
     * <BR><BR>
     * 有効となっているフラグ個数を取得します.
     * <BR>
     * @return int 有効となっているフラグ個数が返されます.
     */
    public final int getUseSize()
    {
        return m_flgOnLength ;
    }
    
    /**
     * 情報存在チェック.
     * <BR><BR>
     * 対象要素位置に情報が存在しているかチェックします.
     * <BR>
     * @param no チェック対象の要素項番を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、要素情報は存在します.
     *                 [false]が返された場合、要素情報は存在しません.
     */
    public final boolean isData( int no )
    {
        long[] flags = null ;
        
        flags = m_flags ;
        
        if( flags == null || no < 0 || no >= m_size ){
            return false ;
        }
        
        return (
            (
                flags[ ( no & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ] &
                ( 1L << ( no & FlagBox.FLAG_ELEMENT ) ) 
            ) != 0
        ) ? true : false ;
    }
    
    /**
     * 利用可能チェック.
     * <BR><BR>
     * オブジェクトが利用可能であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合利用可能です.<BR>
     *                 [false]が返された場合利用不可能です.
     */
    public final boolean isCreate()
    {
        return ( m_flags != null ) ? true : false ;
    }
    
    
    
    /**
     * 最終コードを集計.
     */
    private final void getLastCode()
    {
        int i,j ;
        int len ;
        int lenJ ;
        
        int no ;
        int indNo ;

        int pnt ;
        
        long[] flags = null ;
        
        no = m_lastPoint ;
        if( ( flags = m_flags ) != null ){
            
            indNo = ( no & (~FlagBox.FLAG_ELEMENT) ) >> FlagBox.FLAG_PAUSE ;
            len = indNo ;
            lenJ = FlagBox.FLAG_ELEMENT_LENGTH ;
            
            for( i = len,pnt = -1 ; i >= 0 ; i -- ){
                
                if( flags[ i ] != 0x0000000000000000L ){
                    
                    for( j = 0 ; j < lenJ ; j ++ ){
                        pnt = ( ( flags[ i ] & ( 1L << j ) ) != 0 ) ? j : pnt ;
                    }
                    
                    if( pnt != -1 ){
                        
                        m_lastPoint = ( i * FlagBox.FLAG_ELEMENT_LENGTH ) + pnt ;
                        break ;
                        
                    }
                    
                }
                
                if( pnt == -1 ){
                    m_lastPoint = 0 ;
                }
            }
            
        }
        
    }
    
}

