/*
 * @(#)FloatArray.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.array;

import java.io.Serializable;
import java.util.Arrays;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * FloatArrayオブジェクト.
 * <BR><BR>
 * 数値[float(32bit)]配列を管理するオブジェクトです.<BR>
 * また、このオブジェクトは同期されていません.
 *
 * @version     1.00, 2004/02/02
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class FloatArray implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            FloatArray.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * データ配列 : 空き情報データ.
     */
    private static final float NOT_USE = Float.MAX_VALUE ;
    
    /**
     * 配列管理.
     */
    private float[] m_array = null ;
    
    /**
     * 開始時管理配列数.
     */
    private int m_startLength = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 配列管理数.
     */
    private int m_length = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 現在格納配列数.
     */
    private int m_nowSize = 0 ;
    
    /**
     * ソートフラグ.
     */
    private boolean m_sortFlg = false ;
    
    /**
     * コンストラクタ.
     */
    public FloatArray()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のバッファ係数を設定します.
     * <BR>
     * @param length 対象のバッファ係数を設定します.<BR>
     *               設定可能な最大値は[9999]です.<BR>
     *               設定可能な最小値は[2]です.<BR>
     *               また、これらの設定範囲外を設定した場合、
     *               デフォルトの値[16]となります.
     */
    public FloatArray( int length )
    {
        
        m_startLength = 
            (
                length < ArrayDef.MIN_START_LENGTH ||
                length > ArrayDef.MAX_START_LENGTH
            ) ? ArrayDef.DEFAULT_START_LENGTH :length ;
        
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear()
    {
        
        m_array = null ;
        m_length = ArrayDef.DEFAULT_START_LENGTH ;
        m_nowSize = 0 ;
        m_sortFlg = false ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[Float]情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( Float value )
        throws InputException
    {
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.add( value.floatValue() ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[float(32bit)]情報を追加します.
     */
    public final void add( float value )
    {
        int length ;
        int nowSize ;
        
        float[] info = null ;
        float[] tmp = null ;
        
        info = m_array ;
        length = m_length ;
        nowSize = m_nowSize ;
        
        try{
            
            info[ nowSize ] = value ;
            
        }catch( NullPointerException nul ){
            
            info = new float[ length ] ;
            Arrays.fill( info,FloatArray.NOT_USE ) ;
            info[ nowSize ] = value ;
            
            m_array = info ;
            
        }catch( IndexOutOfBoundsException io ){
            
            length *= 2 ;
            tmp = info ;
            info = new float[ length ] ;
            Arrays.fill( info,FloatArray.NOT_USE ) ;
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            
            info[ nowSize ] = value ;
            
            m_length = length ;
            m_array = info ;
            
        }finally{
            
            tmp = null ;
            info = null ;
            m_sortFlg = false ;
            
        }
        
        m_nowSize ++ ;
        
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象数値[Float]情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,Float value )
        throws InputException
    {
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.set( no,value.floatValue() ) ;
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,float value )
        throws InputException
    {
        int nowLen ;
        
        nowLen = m_nowSize ;
        
        if( no < 0 || no >= nowLen ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_array[ no ] = value ;
        m_sortFlg = false ;
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @return float 削除された数値[float(32bit)]情報が返されます.<BR>
     *              削除情報が存在しない場合等は、[Float.MAX_VALUE]が返されます.
     */
    public final float remove( int no )
    {
        int i ;
        int nowSize ;
        int length ;
        int newLength ;
        
        float[] info = null ;
        float[] tmp = null ;
        
        float ret = Float.MAX_VALUE ;
        
        nowSize = m_nowSize ;
        length = m_length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return Float.MAX_VALUE ;
        }
        
        info = m_array ;
        
        ret = info[ no ] ;
        info[ no ] = FloatArray.NOT_USE ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,1,info,0,( nowSize-1 ) ) ;
            
            info[ nowSize - 1 ] = FloatArray.NOT_USE ;
            
        }else if( ( nowSize - no ) != 1 ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+1,info,no,nowSize-(no+1) ) ;
            
            info[ nowSize - 1 ] = FloatArray.NOT_USE ;
            
        }
        
        nowSize -- ;
        
        if(
            nowSize != 0 &&
            ( length * ArrayDef.REDUCTION_ARRAY ) >= nowSize
        )
        {
            
            newLength = length / 2 ;
            tmp = new float[ newLength ] ;
            Arrays.fill( tmp,FloatArray.NOT_USE ) ;
            System.arraycopy( info,0,tmp,0,newLength ) ;
            
            info = null ;
            info = tmp ;
            
            m_length = newLength ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        m_array = info ;
        m_nowSize = nowSize ;
        m_sortFlg = false ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return int 取得された数値[float(32bit)]情報が返されます.
     *             削除情報が存在しない場合等は、[Float.MAX_VALUE]が返されます.
     */
    public final float get( int no )
    {
        
        if( no < 0 || no >= m_nowSize ){
            return Float.MAX_VALUE ;
        }
        
        return m_array[ no ] ;
    }
    
    /**
     * データを昇順でソート.
     * <BR><BR>
     * データを昇順でソートします.
     */
    public final void sort()
    {
        if( m_nowSize != 0 ){
            Arrays.sort( m_array,0,m_nowSize ) ;
            m_sortFlg = true ;
        }
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( float key )
    {
        return this.indexOf( key,0 ) ;
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( float key,int index )
    {
        int ret ;
        
        if( m_nowSize != 0 ){
            if( m_sortFlg == false ){
                this.sort() ;
            }
            ret = Arrays.binarySearch( m_array,key ) ;
            ret = ( ret < 0 || ret >= m_nowSize ) ? -1 : ret ;
        }
        else{
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の数値[float(32bit)]配列を取得.
     * <BR><BR>
     * 対象の数値[float(32bit)]配列を取得します.<BR>
     * また、この数値[float(32bit)]配列は配列の再構成を行った場合、
     * 情報の保証は行われません.<BR>
     * また、この数値[float(32bit)]群は基本的の読み込み専用で利用する
     * ことを進めます.
     * <BR>
     * @return float[] 対象の数値[float(32bit)]配列が返されます.
     */
    public final float[] getObjects()
    {
        return m_array ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 格納されている情報数が返されます.
     */
    public final int size()
    {
        return m_nowSize ;
    }
    
}

