/*
 * @(#)CharArray.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.array;

import java.io.Serializable;
import java.util.Arrays;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * CharArrayオブジェクト.
 * <BR><BR>
 * 数値[char(16bit)]配列を管理するオブジェクトです.<BR>
 * また、このオブジェクトは同期されていません.
 *
 * @version     1.00, 2004/02/02
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class CharArray implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            CharArray.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 配列管理.
     */
    private char[] m_array = null ;
    
    /**
     * 開始時管理配列数.
     */
    private int m_startLength = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 配列管理数.
     */
    private int m_length = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 現在格納配列数.
     */
    private int m_nowSize = 0 ;
    
    /**
     * コンストラクタ.
     */
    public CharArray()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のバッファ係数を設定します.
     * <BR>
     * @param length 対象のバッファ係数を設定します.<BR>
     *               設定可能な最大値は[9999]です.<BR>
     *               設定可能な最小値は[2]です.<BR>
     *               また、これらの設定範囲外を設定した場合、
     *               デフォルトの値[16]となります.
     */
    public CharArray( int length )
    {
        
        m_startLength = 
            (
                length < ArrayDef.MIN_START_LENGTH ||
                length > ArrayDef.MAX_START_LENGTH
            ) ? ArrayDef.DEFAULT_START_LENGTH :length ;
        
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear()
    {
        
        m_array = null ;
        m_length = ArrayDef.DEFAULT_START_LENGTH ;
        m_nowSize = 0 ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[Character]情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( Character value )
        throws InputException
    {
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.add( value.charValue() ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[char(16bit)]情報を追加します.
     */
    public final void add( char value )
    {
        int length ;
        int nowSize ;
        
        char[] info = null ;
        char[] tmp = null ;
        
        info = m_array ;
        length = m_length ;
        nowSize = m_nowSize ;
        
        try{
            
            info[ nowSize ] = value ;
            
        }catch( NullPointerException nul ){
            
            info = new char[ length ] ;
            info[ nowSize ] = value ;
            
            m_array = info ;
            
        }catch( IndexOutOfBoundsException io ){
            
            length *= 2 ;
            tmp = info ;
            info = new char[ length ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            
            info[ nowSize ] = value ;
            
            m_length = length ;
            m_array = info ;
            
        }finally{
            
            tmp = null ;
            info = null ;
            
        }
        
        m_nowSize ++ ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[char(16bit)]配列情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( char[] value )
        throws InputException
    {
        try{
            
            this.add( 0,-1,value ) ;
            
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param off 追加対象情報のオフセット値を設定します.
     * @param value 設定対象の数値[char(16bit)]配列情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( int off,char[] value )
        throws InputException
    {
        try{
            
            this.add( off,-1,value ) ;
            
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[char(16bit)]配列情報を追加します.
     * @param length 追加対象情報のデータ長を設定します.
     * @exception InputException 入力例外.
     */
    public final void add( char[] value,int length )
        throws InputException
    {
        try{
            
            this.add( 0,length,value ) ;
            
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param off 追加対象情報のオフセット値を設定します.
     * @param length 追加対象情報のデータ長を設定します.
     * @param value 設定対象の数値[char(16bit)]配列情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( int off,int length,char[] value )
        throws InputException
    {
        
        int arrayLen ;
        int createLen ;
        int valueLen ;
        int nowSize ;
        
        char[] info = null ;
        char[] tmp = null ;
        
        if( value == null || ( valueLen = value.length ) <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }else if( off < 0 || off >= valueLen ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        length = ( length <= 0 || ( length + off ) > valueLen ) ?
            valueLen - off : length ;
        
        info = m_array ;
        arrayLen = m_length ;
        nowSize = m_nowSize ;
        
        try{
            
            System.arraycopy( info,nowSize,value,off,length ) ;
            
        }catch( NullPointerException nul ){
            
            for( ; arrayLen <= length ; ){
                arrayLen *= 2 ;
            }
            
            info = new char[ arrayLen ] ;
            System.arraycopy( info,nowSize,value,off,length ) ;
            
            m_length = arrayLen ;
            m_array = info ;
            
        }catch( IndexOutOfBoundsException io ){
            
            createLen = length + nowSize ;
            
            for( ; arrayLen <= createLen ; ){
                arrayLen *= 2 ;
            }
            
            tmp = info ;
            info = new char[ arrayLen ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            System.arraycopy( info,nowSize,value,off,length ) ;
            
            m_length = arrayLen ;
            m_array = info ;
            
        }finally{
            
            tmp = null ;
            info = null ;
            
        }
        
        m_nowSize += length ;
        
    }
    
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象数値[Character]情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,Character value )
        throws InputException
    {
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.set( no,value.charValue() ) ;
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象数値[char(16bit)]情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,char value )
        throws InputException
    {
        int nowLen ;
        
        nowLen = m_nowSize ;
        
        if( no < 0 || no >= nowLen ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_array[ no ] = value ;
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @return char 削除された数値[char(16bit)]情報が返されます.<BR>
     *              削除情報が存在しない場合等は、[Character.MAX_VALUE]が返されます.
     */
    public final char remove( int no )
    {
        int i ;
        int nowSize ;
        int length ;
        int newLength ;
        
        char[] info = null ;
        char[] tmp = null ;
        
        char ret = Character.MAX_VALUE ;
        
        nowSize = m_nowSize ;
        length = m_length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return Character.MAX_VALUE ;
        }
        
        info = m_array ;
        
        ret = info[ no ] ;
        info[ no ] = ( char )ArrayDef.DEFAULT_DATA ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,1,info,0,( nowSize-1 ) ) ;
            
            info[ nowSize - 1 ] = ( char )ArrayDef.DEFAULT_DATA ;
            
        }else if( ( nowSize - no ) != 1 ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+1,info,no,nowSize-(no+1) ) ;
            
            info[ nowSize - 1 ] = ( char )ArrayDef.DEFAULT_DATA ;
            
        }
        
        nowSize -- ;
        
        if(
            nowSize != 0 &&
            ( length * ArrayDef.REDUCTION_ARRAY ) >= nowSize
        )
        {
            
            newLength = length / 2 ;
            tmp = new char[ newLength ] ;
            System.arraycopy( info,0,tmp,0,newLength ) ;
            
            info = null ;
            info = tmp ;
            
            m_length = newLength ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        m_array = info ;
        m_nowSize = nowSize ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @param length 削除対象の長さを設定します.
     * @return char[] 削除された数値[char(16bit)]配列情報が返されます.<BR>
     *                削除情報が存在しない場合等は、[null]が返されます.
     */
    public final char[] remove( int no,int length )
    {
        int i ;
        int nowSize ;
        int arrayLen ;
        int newLength ;
        int createLen ;
        
        char[] info = null ;
        char[] tmp = null ;
        
        char[] ret = null ;
        
        nowSize = m_nowSize ;
        arrayLen = m_length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return null ;
        }
        
        length = ( length <= 0 || ( length + no ) <= nowSize ) ?
            nowSize - no : length ;
        
        info = m_array ;
        
        ret = new char[ length ] ;
        System.arraycopy( info,no,ret,0,length ) ;
        Arrays.fill( info,nowSize - length,nowSize,( char )ArrayDef.DEFAULT_DATA ) ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,no,info,0,( nowSize - length ) ) ;
            
        }else if( ( nowSize - no ) != length ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+length,info,no,( nowSize - (no+length ) ) ) ;
            
        }
        
        nowSize -= length ;
        
        if(
            nowSize != 0 &&
            ( arrayLen * ArrayDef.REDUCTION_ARRAY ) >= nowSize
        )
        {
            createLen = ( int )( arrayLen * ArrayDef.REDUCTION_ARRAY ) ;
            
            for( ; arrayLen <= createLen ; ){
                arrayLen /= 2 ;
            }
            
            arrayLen *= 2 ;
            
            tmp = new char[ arrayLen ] ;
            System.arraycopy( info,0,tmp,0,arrayLen ) ;
            
            info = null ;
            info = tmp ;
            
            m_length = arrayLen ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        m_array = info ;
        m_nowSize = nowSize ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return char 取得された数値[char(16bit)]情報が返されます.
     *             削除情報が存在しない場合等は、[Character.MAX_VALUE]が返されます.
     */
    public final char get( int no )
    {
        
        if( no < 0 || no >= m_nowSize ){
            return Character.MAX_VALUE ;
        }
        
        return m_array[ no ] ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @param length 取得対象の長さを設定します.
     * @return char[] 取得された数値[char(16bit)]配列情報が返されます.
     *                削除情報が存在しない場合等は、[null]が返されます.
     */
    public final char[] get( int no,int length )
    {
        int nowSize ;
        char[] ret = null ;
        
        nowSize = m_nowSize ;
        
        if( no < 0 || no >= nowSize ){
            return null ;
        }
        
        length = ( length <= 0 || ( length + no ) <= nowSize ) ?
            nowSize - no : length ;
        
        ret = new char[ length ] ;
        System.arraycopy( m_array,no,ret,0,length ) ;
        
        return ret ;
        
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param out 取得されたデータが格納されます.
     * @param no 取得対象の項番を設定します.
     * @return int 取得された長さが返されます.
     */
    public final int get( char[] out,int no )
    {
        int ret ;
        int nowSize ;
        
        nowSize = m_nowSize ;
        
        if( out == null || no < 0 || no >= nowSize ){
            return 0 ;
        }
        
        ret = out.length ;
        ret = ( ret <= 0 || ( ret + no ) <= nowSize ) ?
            nowSize - no : ret ;
        
        System.arraycopy( m_array,no,out,0,ret ) ;
        
        return ret ;
        
    }
    
    /**
     * データを昇順でソート.
     * <BR><BR>
     * データを昇順でソートします.
     */
    public final void sort()
    {
        if( m_nowSize != 0 ){
            Arrays.sort( m_array,0,m_nowSize ) ;
        }
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( char key )
    {
        return this.indexOf( key,0 ) ;
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( char key,int index )
    {
        int len ;
        int ret ;
        char[] tmp = null ;
        
        if( m_nowSize != 0 ){
            len = m_nowSize - index ;
            tmp = new char[ len ] ;
            System.arraycopy( m_array,index,tmp,0,len ) ;
            Arrays.sort( tmp ) ;
            ret = Arrays.binarySearch( tmp,key ) ;
            tmp = null ;
            ret = ( ret < 0 || ret >= m_nowSize ) ? -1 : ret ;
        }
        else{
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の数値[char(16bit)]配列を取得.
     * <BR><BR>
     * 対象の数値[char(16bit)]配列を取得します.<BR>
     * また、この数値[char(16bit)]配列は配列の再構成を行った場合、
     * 情報の保証は行われません.<BR>
     * また、この数値[char(16bit)]群は基本的の読み込み専用で利用する
     * ことを進めます.
     * <BR>
     * @return char[] 対象の数値[char(16bit)]配列が返されます.
     */
    public final char[] getObjects()
    {
        return m_array ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 格納されている情報数が返されます.
     */
    public final int size()
    {
        return m_nowSize ;
    }
    
}

