/*
 * @(#)BooleanArray.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util.array;

import java.io.Serializable;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * BooleanArrayオブジェクト.
 * <BR><BR>
 * 数値[boolean(1bit)]配列を管理するオブジェクトです.<BR>
 * また、このオブジェクトは同期されていません.
 *
 * @version     1.00, 2004/02/02
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BooleanArray implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            BooleanArray.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 配列管理.
     */
    private boolean[] m_array = null ;
    
    /**
     * 開始時管理配列数.
     */
    private int m_startLength = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 配列管理数.
     */
    private int m_length = ArrayDef.DEFAULT_START_LENGTH ;
    
    /**
     * 現在格納配列数.
     */
    private int m_nowSize = 0 ;
    
    /**
     * 情報保全タイマー値.
     */
    private long m_timer = 0L ;
    
    /**
     * コンストラクタ.
     */
    public BooleanArray()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のバッファ係数を設定します.
     * <BR>
     * @param length 対象のバッファ係数を設定します.<BR>
     *               設定可能な最大値は[9999]です.<BR>
     *               設定可能な最小値は[2]です.<BR>
     *               また、これらの設定範囲外を設定した場合、
     *               デフォルトの値[16]となります.
     */
    public BooleanArray( int length )
    {
        
        m_startLength = 
            (
                length < ArrayDef.MIN_START_LENGTH ||
                length > ArrayDef.MAX_START_LENGTH
            ) ? ArrayDef.DEFAULT_START_LENGTH :length ;
        
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear()
    {
        
        m_array = null ;
        m_length = ArrayDef.DEFAULT_START_LENGTH ;
        m_nowSize = 0 ;
        
        //this.createTimer() ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[Boolean]情報を追加します.
     * @exception InputException 入力例外.
     */
    public final void add( Boolean value )
        throws InputException
    {
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.add( value.booleanValue() ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[boolean(1bit)]情報を追加します.
     */
    public final void add( boolean value )
    {
        int length ;
        int nowSize ;
        
        boolean[] info = null ;
        boolean[] tmp = null ;
        
        info = m_array ;
        length = m_length ;
        nowSize = m_nowSize ;
        
        try{
            
            info[ nowSize ] = value ;
            
        }catch( NullPointerException nul ){
            
            info = new boolean[ length ] ;
            info[ nowSize ] = value ;
            
            m_array = info ;
            //this.createTimer() ;
            
        }catch( IndexOutOfBoundsException io ){
            
            length *= 2 ;
            tmp = info ;
            info = new boolean[ length ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            
            info[ nowSize ] = value ;
            
            m_length = length ;
            m_array = info ;
            
            //this.createTimer() ;
            
        }finally{
            
            tmp = null ;
            info = null ;
            
        }
        
        m_nowSize ++ ;
        
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象数値[Boolean]情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,Boolean value )
        throws InputException
    {
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.set( no,value.booleanValue() ) ;
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void set( int no,boolean value )
        throws InputException
    {
        int nowLen ;
        
        nowLen = m_nowSize ;
        
        if( no < 0 || no >= nowLen ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_array[ no ] = value ;
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @return boolean 削除された数値[boolean(1bit)]情報が返されます.<BR>
     *              削除情報が存在しない場合等は、[false]が返されます.
     */
    public final boolean remove( int no )
    {
        int i ;
        int nowSize ;
        int length ;
        int newLength ;
        
        boolean[] info = null ;
        boolean[] tmp = null ;
        
        boolean ret = false ;
        
        nowSize = m_nowSize ;
        length = m_length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return false ;
        }
        
        info = m_array ;
        
        ret = info[ no ] ;
        info[ no ] = false ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,1,info,0,( nowSize-1 ) ) ;
            
            info[ nowSize - 1 ] = false ;
            
        }else if( ( nowSize - no ) != 1 ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+1,info,no,nowSize-(no+1) ) ;
            
            info[ nowSize - 1 ] = false ;
            
        }
        
        nowSize -- ;
        
        if(
            nowSize != 0 &&
            ( length * ArrayDef.REDUCTION_ARRAY ) >= nowSize
        )
        {
            
            newLength = length / 2 ;
            tmp = new boolean[ newLength ] ;
            System.arraycopy( info,0,tmp,0,newLength ) ;
            
            info = null ;
            info = tmp ;
            
            m_length = newLength ;
            
            //this.createTimer() ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        m_array = info ;
        m_nowSize = nowSize ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return int 取得された数値[boolean(1bit)]情報が返されます.
     *             削除情報が存在しない場合等は、[false]が返されます.
     */
    public final boolean get( int no )
    {
        
        if( no < 0 || no >= m_nowSize ){
            return false ;
        }
        
        return m_array[ no ] ;
    }
    
    /**
     * 対象の数値[boolean(1bit)]配列を取得.
     * <BR><BR>
     * 対象の数値[boolean(1bit)]配列を取得します.<BR>
     * また、この数値[boolean(1bit)]配列は配列の再構成を行った場合、
     * 情報の保証は行われません.<BR>
     * また、この数値[boolean(1bit)]群は基本的の読み込み専用で利用する
     * ことを進めます.
     * <BR>
     * @return boolean[] 対象の数値[boolean(1bit)]配列が返されます.
     */
    public final boolean[] getObjects()
    {
        return m_array ;
    }
    
    /**
     * 情報保全タイマー値の取得.
     * <BR><BR>
     * 情報保全タイマー値を取得します.<BR>
     * この情報は、メソッド[BooleanArray#getObjects()]のデータ安全有無を
     * 示唆するために利用します.<BR>
     * 方法として、メソッド[BooleanArray#getObjects()]実施後、
     * 当メソッドを呼び出し、タイマー値を保持します.<BR>
     * その後、メソッド[BooleanArray#getObjects()]を呼び出し、
     * 値が変化している場合、そのオブジェクト群は有効でなくなった意味を持ちます.
     * <BR>
     * @return int 情報保全タイマー値が返されます.
     */
    public final long getTimer()
    {
        return m_timer ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 格納されている情報数が返されます.
     */
    public final int size()
    {
        return m_nowSize ;
    }
    
    
    
    
    /**
     * 情報保全タイマー値の更新.
     */
    private final void createTimer()
    {
        m_timer = System.currentTimeMillis() ;
    }
    
}

