/*
 * @(#)KeyToNum.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * 格納情報を、キー名及び項番で管理.
 * <BR><BR>
 * 格納情報を、キー名及び項番で管理します.
 *  
 * @version 1.0.0 2003/11/04
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class KeyToNum
{
    
    /**
     * キー名による情報格納.
     */
    private final CharTableSync m_key = new CharTableSync() ;
    
    /**
     * 項番による、キー名管理.
     */
    private final ObjectArray m_num = new ObjectArray() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object m_sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public KeyToNum()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報をクリアします.
     * <BR><BR>
     * 格納情報をクリアします.
     */
    public final void clear()
    {
        synchronized( m_sync ){
            m_key.clear() ;
            m_num.clear() ;
        }
    }
    
    /**
     * 情報を設定.
     * <BR><BR>
     * 情報を設定します.
     * <BR>
     * @param name 追加対象のキー名を設定します.
     * @param info 追加対象の情報をセットします.
     * @exception InputException 入力例外.
     */
    public final void add( String name,Object info )
        throws InputException
    {
        if( name == null || info == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync ){
                m_key.add( name,info ) ;
                m_num.add( name ) ;
            }
            
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }
    }
    
    /**
     * 情報を取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param name 取得対象のキー名を設定します.
     * @return Object 取得結果の情報が返されます.
     * @exception InputException 入力例外.
     */
    public final Object get( String name ) throws InputException
    {
        Object ret = null ;
        
        if( name == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        synchronized( m_sync ){
            ret = m_key.get( name ) ;
        }
        
        return ret ;
    }
    
    /**
     * 情報を取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return Object 取得結果の情報が返されます.
     * @exception InputException 入力例外.
     */
    public final Object get( int no ) throws InputException
    {
        Object ret = null ;
        String name ;
        
        if( no < 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync ){
                name = ( String )m_num.get( no ) ;
                ret = m_key.get( name ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            name = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納情報名を取得.
     * <BR><BR>
     * 項番を指定して、格納情報数を取得します.
     * <BR>
     * @param num 取得対象の項番を指定します.
     * @return String 項番に対するキー名が返されます.
     * @exception InputException 入力例外.
     */
    public final String getName( int num )
        throws InputException
    {
        String name = null ;
        String ret = null ;
        
        if( num < 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync ){
                ret = ( String )m_num.get( num ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納キー名群の取得.
     * <BR><BR>
     * 格納されたキー名群を取得します.
     * <BR>
     * @return String[] 格納キー名群が返されます.<BR>
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getNames()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync ){
                ret = m_key.getNames() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 項番順で格納キー名群を取得.
     * <BR><BR>
     * 項番順で格納キー名群を取得します.
     * <BR>
     * @return String[] 格納キー名群が返されます.
     */
    public final String[] getNameByNumber()
    {
        int len ;
        String[] ret = null ;
        
        try{
            synchronized( m_sync ){
                
                len = m_key.size() ;
                ret = new String[ len ] ;
                System.arraycopy( m_num.getObjects(),0,ret,0,len ) ;
                
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納情報数を取得.
     * <BR><BR>
     * 格納情報数を取得します.
     * <BR>
     * @return int 格納情報数が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync ){
                ret = m_num.size() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のキー名が存在するかチェック.
     * <BR><BR>
     * 対象のキー名が既に存在するかチェックします.
     * <BR>
     * @param key チェック対象のキー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、存在します.<BR>
     *                 [false]が返された場合、存在しません.
     */
    public final boolean isData( String key )
    {
        boolean ret ;
        
        try{
            synchronized( m_sync ){
                ret = m_key.isData( key ) ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

