/*
 * @(#)IdManagerEx.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import com.JRcServer.commons.serialize.InitSerialize;
import com.JRcServer.commons.serialize.SerializeUtil;
import com.JRcServer.commons.thread.Synchronized;

/**
 * ID発番管理オブジェクト.
 * <BR><BR>
 * ID発番を管理するためのオブジェクトです.<BR>
 * このオブジェクトは、[com.JRcServer.commons.util.IdManager]より、大きな値のIDを管理
 * する場合に利用することを推奨します.<BR>
 * また、IdManagerオブジェクトとIdManagerExオブジェクトの単位の幅は以下の
 * ようになっています.<BR>
 * <PRE>
 *   IdManager      :   0   -   2147483647
 *   IdManagerEx    :   0   -   9223372036854775807
 * </PRE>
 * また、速度を優先する場合は IdManagerオブジェクトを利用します.<BR>
 * また、情報桁幅を多く取りたい場合などはIdManagerExオブジェクトを利用します.
 *
 * @version     1.00, 2004/04/12
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class IdManagerEx implements InitSerialize
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            IdManagerEx.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 取得IDなし.
     */
    public static final long NOT_ID = -1L ;
    
    /**
     * 検索対象数.
     */
    public static final int SEARCH_LENGTH = 1000000 ;
    
    
    /**
     * 利用ID管理項番.
     */
    private final SearchLong m_num = new SearchLong() ;
    
    /**
     * MAX-ID.
     */
    private long m_maxID = 0L ;
    
    /**
     * 開始値.
     */
    private long m_startID = 0L ;
    
    /**
     * 管理ID.
     */
    private long m_id = 0L ;
    
    /**
     * テーブル生成フラグ.
     */
    private volatile boolean m_tblFlg = false ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 任意の最大値を用いて情報を生成します.
     */
    public IdManagerEx()
    {
        this.create( 0L,Long.MAX_VALUE ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     */
    public IdManagerEx( long start,long max )
    {
        this.create( start,max ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行うメソッドです.<BR>
     * 基本的には、このインターフェイスを継承したオブジェクトは、
     * 初期化に必要な処理を実装することで、
     * [com.JRcServer.commons.serialize.SerializeCom.getSerialize()]からの、
     * オブジェクトロードの際に、このメソッドを呼び出してくれるので、
     * ロード後のオブジェクトを、円滑に利用する事が出来ます.
     */
    public void initSerializable()
    {
        m_sync.create() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     */
    public final void create( long start,long max )
    {
        
        if(
            start < 0L || max < 1L ||
            start >= max || max > Long.MAX_VALUE
        )
        {
            start = 0L ;
            max = Long.MAX_VALUE ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync ){
                m_num.create() ;
                m_maxID = max ;
                m_startID = start ;
                m_id = 0L ;
                this.setTblOn() ;
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_sync.create() ;
        
        this.setTblOff() ;
        try{
            synchronized( m_sync.get() ){
                m_num.clear() ;
                m_id = m_startID ;
            }
        }catch( NullPointerException nul ){
            m_num.clear() ;
            m_id = m_startID ;
        }
        
        m_tblFlg = false ;
        
    }
    
    /**
     * 利用中のID情報を追加.
     * <BR><BR>
     * 利用中のID情報を追加します.
     * <BR>
     * @param id 利用中のID情報として追加します.
     */
    public final void addUseID( long id )
    {
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                if(
                    m_startID <= id &&
                    m_maxID >= id &&
                    m_num.isData( id ) == false
                )
                {
                    m_num.add( id ) ;
                }
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * 一意なID項番を取得.
     * <BR><BR>
     * 一意なID項番を取得します.
     * <BR>
     * @return long 一意なID項番が返されます.
     */
    public final long getID()
    {
        int i ;
        int len ;
        long startID ;
        long maxID ;
        
        long ret ;
        boolean okFlg ;
        SearchLong tbl = null ;
        
        this.renewTable() ;
        ret = IdManagerEx.NOT_ID ;
        len = IdManagerEx.SEARCH_LENGTH ;
        okFlg = false ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                tbl = m_num ;
                startID = m_startID ;
                maxID = m_maxID ;
                
                if( maxID - startID <= tbl.size() ){
                    return IdManager.NOT_ID ;
                }
                
            }
            
            for( ;; ){
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    synchronized( m_sync.get() ){
                        
                        ret = m_id ;
                        m_id = ( ret >= maxID ) ? startID : ret + 1L ;
                        
                        if( tbl.isData( ret ) == false ){
                            
                            tbl.add( ret ) ;
                            okFlg = true ;
                            break ;
                            
                        }
                        
                    }
                    
                }
                
                if( okFlg == true ){
                    break ;
                }
                
                UtilCom.cpuCreate() ;
                
            }
        }catch( NullPointerException nul ){
            ret = IdManagerEx.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 利用終了IDの削除.
     * <BR><BR>
     * 利用終了IDを削除します.
     * <BR>
     * @param id 利用終了のIDを設定します.
     */
    public final void removeID( long id )
    {
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                m_num.remove( id ) ;
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 設定ID開始番号の取得.
     * <BR><BR>
     * 設定されているID開始番号を取得します.
     * <BR>
     * @return long 設定されているID開始番号が返されます.
     */
    public final long getStartID()
    {
        long ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_startID ;
            }
        }catch( NullPointerException nul ){
            ret = IdManagerEx.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 設定ID幅(MAX値)を取得します.
     * <BR><BR>
     * 設定されているID幅(MAX値)を取得します.
     * <BR>
     * @return long 設定されているID幅(MAX値)が返されます.
     */
    public final long getMaxID()
    {
        long ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_maxID ;
            }
        }catch( NullPointerException nul ){
            ret = IdManagerEx.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 現在IDを利用している情報数を取得.
     * <BR><BR>
     * 現在IDを利用している情報数を取得します.
     * <BR>
     * @return int 現在IDを利用している情報数が返されます.
     */
    public final int getUseIDSize()
    {
        int ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_num.size() ;
            }
        }catch( NullPointerException nul ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在利用されているID群を取得.
     * <BR><BR>
     * 現在利用されているID群を取得します.
     * <BR>
     * @return long[] 現在利用されているID群が返されます.
     */
    public final long[] getUseIDs()
    {
        long[] ret = null ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_num.getAll() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ID情報が既に利用しているかチェック.
     * <BR><BR>
     * 指定されたID情報が既に利用しているかチェックします.
     * <BR>
     * @param id チェック対象のID情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、指定ID情報は現在利用されています.
     *                 [false]が返された場合、指定ID情報は現在利用されていません.
     */
    public final boolean isUseID( long id )
    {
        boolean ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_num.isData( id ) ;
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 重複管理テーブル未生成時の生成処理.
     */
    private synchronized final void renewTable()
    {
        if( m_tblFlg == false ){
            m_num.create() ;
            m_tblFlg = true ;
        }
    }
    
    /**
     * テーブル生成フラグをONに設定.
     */
    private synchronized final void setTblOn()
    {
        m_tblFlg = true ;
    }
    
    /**
     * テーブル生成フラグをOFFに設定.
     */
    private synchronized final void setTblOff()
    {
        m_tblFlg = false ;
    }
    
}

