/*
 * @(#)DateTimeFormat.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import java.util.Calendar;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.ReadSyntaxException;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.array.IntArray;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * 日付・時間フォーマット解析用定義.
 * <BR><BR>
 * 対象の日付・時間フォーマット解析用のオブジェクトを定義します.
 *
 * @version 1.00, 2003/11/02
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class DateTimeFormat
{
    /**
     * 日付・時間情報 : ２桁西暦情報.
     * <BR><BR>
     * この情報は、２桁の西暦情報フォーマットであることを示します.
     */
    public static final String FORMAT_YEAR_2 = "yy" ;
    
    /**
     * 日付・時間情報 : ４桁西暦情報.
     * <BR><BR>
     * この情報は、４桁の西暦情報フォーマットであることを示します.
     */
    public static final String FORMAT_YEAR_4 = "YYYY" ;
    
    /**
     * 日付・時間情報 : 月情報.
     * <BR><BR>
     * この情報は、月情報フォーマットであることを示します.
     */
    public static final String FORMAT_MONTH = "MM" ;
    
    /**
     * 日付・時間情報 : 日情報.
     * <BR><BR>
     * この情報は、日情報フォーマットであることを示します.
     */
    public static final String FORMAT_DAY = "DD" ;
    
    /**
     * 日付・時間情報 : 時間情報.
     * <BR><BR>
     * この情報は、時間情報フォーマットであることを示します.
     */
    public static final String FORMAT_HOUR = "hh" ;
    
    /**
     * 日付・時間情報 : 分情報.
     * <BR><BR>
     * この情報は、分情報フォーマットであることを示します.
     */
    public static final String FORMAT_MINUTES = "mm" ;
    
    /**
     * 日付・時間情報 : 秒情報.
     * <BR><BR>
     * この情報は、秒情報フォーマットであることを示します.
     */
    public static final String FORMAT_SECOND = "ss" ;
    
    /**
     * 日付・時間情報 : ミリ秒情報.
     * <BR><BR>
     * この情報は、ミリ秒情報フォーマットであることを示します.
     */
    public static final String FORMAT_MILLISECOND = "SSSS" ;
    
    /**
     * 日付・時間情報 : 週情報(英語表記)情報.
     * <BR><BR>
     * この情報は、週情報(英語表記)情報フォーマットであることを示します.
     */
    public static final String FORMAT_WEEK_TO_ENGLISH = "ww" ;
    
    /**
     * 日付・時間情報 : 週情報(日本語表記)情報.
     * <BR><BR>
     * この情報は、週情報(日本語表記)情報フォーマットであることを示します.
     */
    public static final String FORMAT_WEEK_TO_JAPAN = "WW" ;
    
    /**
     * 日付・時間情報 : グレゴリオ暦情報.
     * <BR><BR>
     * この情報は、グレゴリオ暦情報フォーマットであることを示します.
     */
    public static final String FORMAT_GREGORIO = "GG" ;
    
    /**
     * 日付・時間情報 : ２桁西暦情報.
     * <BR><BR>
     * この情報は、２桁の西暦情報であることを示します.
     */
    public static final int TYPE_YEAR_2 = 0x00000001 ;
    
    /**
     * 日付・時間情報 : ４桁西暦情報.
     * <BR><BR>
     * この情報は、４桁の西暦情報であることを示します.
     */
    public static final int TYPE_YEAR_4 = 0x00000002 ;
    
    /**
     * 日付・時間情報 : 月情報.
     * <BR><BR>
     * この情報は、月情報であることを示します.
     */
    public static final int TYPE_MONTH = 0x00000004 ;
    
    /**
     * 日付・時間情報 : 日情報.
     * <BR><BR>
     * この情報は、日情報であることを示します.
     */
    public static final int TYPE_DAY = 0x00000008 ;
    
    /**
     * 日付・時間情報 : 時間情報.
     * <BR><BR>
     * この情報は、時間情報であることを示します.
     */
    public static final int TYPE_HOUR = 0x00000010 ;
    
    /**
     * 日付・時間情報 : 分情報.
     * <BR><BR>
     * この情報は、分情報であることを示します.
     */
    public static final int TYPE_MINUTES = 0x00000020 ;
    
    /**
     * 日付・時間情報 : 秒情報.
     * <BR><BR>
     * この情報は、秒情報であることを示します.
     */
    public static final int TYPE_SECOND = 0x00000040 ;
    
    /**
     * 日付・時間情報 : ミリ秒情報.
     * <BR><BR>
     * この情報は、ミリ秒情報であることを示します.
     */
    public static final int TYPE_MILLISECOND = 0x00000080 ;
    
    /**
     * 日付・時間情報 : 週情報(英語表記).
     * <BR><BR>
     * この情報は、週情報(英語表記)情報であることを示します.
     */
    public static final int TYPE_WEEK_TO_ENGLISH = 0x00010000 ;
    
    /**
     * 日付・時間情報 : 週情報(日本語表記).
     * <BR><BR>
     * この情報は、週情報(日本語表記)情報であることを示します.
     */
    public static final int TYPE_WEEK_TO_JAPAN = 0x00020000 ;
    
    /**
     * 日付・時間情報 : グレゴリオ暦.
     * <BR><BR>
     * この情報は、グレゴリオ暦情報であることを示します.
     */
    public static final int TYPE_GREGORIO = 0x00040000 ;
    
    
    /**
     * フォーマット種別 : 非日付、時間情報.
     */
    private static final int TYPE_NOT_FORMAT = 0x80000000 ;
    
    /**
     * 日付・時間情報 : 情報なし.
     */
    private static final char CHAR_NOT = 0xfffe ;
    
    /**
     * 日付・時間情報 : その他コード.
     */
    private static final char CHAR_ETC = 0xffff ;
    
    
    /**
     * 日付・時間情報 : ２桁西暦フォーマット.
     * <BR><BR>
     * この情報は、２桁の西暦フォーマットであることを示します.
     */
    private static final char CHAR_YEAR_2 = 'y' ;
    private static final int LEN_YEAR_2 = DateTimeFormat.FORMAT_YEAR_2.length() ;
    
    /**
     * 日付・時間情報 : ４桁西暦フォーマット.
     * <BR><BR>
     * この情報は、４桁の西暦フォーマットであることを示します.
     */
    private static final char CHAR_YEAR_4 = 'Y' ;
    private static final int LEN_YEAR_4 = DateTimeFormat.FORMAT_YEAR_4.length() ;
    
    /**
     * 日付・時間情報 : 月フォーマット.
     * <BR><BR>
     * この情報は、月フォーマットであることを示します.
     */
    private static final char CHAR_MONTH = 'M' ;
    private static final int LEN_MONTH = DateTimeFormat.FORMAT_MONTH.length() ;
    
    /**
     * 日付・時間情報 : 日フォーマット.
     * <BR><BR>
     * この情報は、日フォーマットであることを示します.
     */
    private static final char CHAR_DAY = 'D' ;
    private static final int LEN_DAY = DateTimeFormat.FORMAT_DAY.length() ;
    
    /**
     * 日付・時間情報 : 時間フォーマット.
     * <BR><BR>
     * この情報は、時間フォーマットであることを示します.
     */
    private static final char CHAR_HOUR = 'h' ;
    private static final int LEN_HOUR = DateTimeFormat.FORMAT_HOUR.length() ;
    
    /**
     * 日付・時間情報 : 分フォーマット.
     * <BR><BR>
     * この情報は、分フォーマットであることを示します.
     */
    private static final char CHAR_MINUTES = 'm' ;
    private static final int LEN_MINUTES = DateTimeFormat.FORMAT_MINUTES.length() ;
    
    /**
     * 日付・時間情報 : 秒フォーマット.
     * <BR><BR>
     * この情報は、秒フォーマットであることを示します.
     */
    private static final char CHAR_SECOND = 's' ;
    private static final int LEN_SECOND = DateTimeFormat.FORMAT_SECOND.length() ;
    
    /**
     * 日付・時間情報 : ミリ秒フォーマット.
     * <BR><BR>
     * この情報は、ミリ秒フォーマットであることを示します.
     */
    private static final char CHAR_MILLISECOND = 'S' ;
    private static final int LEN_MILLISECOND = DateTimeFormat.FORMAT_MILLISECOND.length() ;
    
    /**
     * 日付・時間情報 : 週フォーマット(英語表記).
     * <BR><BR>
     * この情報は、週フォーマット(英語表記)情報であることを示します.
     */
    private static final char CHAR_WEEK_TO_ENGLISH = 'w' ;
    private static final int LEN_WEEK_TO_ENGLISH = DateTimeFormat.FORMAT_WEEK_TO_ENGLISH.length() ;
    
    /**
     * 日付・時間情報 : 週フォーマット(日本語表記).
     * <BR><BR>
     * この情報は、週フォーマット(日本語表記)情報であることを示します.
     */
    private static final char CHAR_WEEK_TO_JAPAN = 'W' ;
    private static final int LEN_WEEK_TO_JAPAN = DateTimeFormat.FORMAT_WEEK_TO_JAPAN.length() ;
    
    /**
     * 日付・時間情報 : グレゴリオ暦.
     * <BR><BR>
     * この情報は、グレゴリオ暦情報であることを示します.
     */
    private static final char CHAR_GREGORIO = 'G' ;
    private static final int LEN_GREGORIO = DateTimeFormat.FORMAT_GREGORIO.length() ;
    
    /**
     * 日付・時間情報 : その他情報.
     */
    private static final String ETC_TYPE_CODE = "?" ;
    
    /**
     * 桁埋め情報 : ２桁.
     */
    private static final String ADD_REG_2 = "00" ;
    
    /**
     * 桁埋め情報 : ４桁.
     */
    private static final String ADD_REG_4 = "0000" ;
    
    /**
     * チェック情報範囲外.
     */
    private static final int NOT_CHK_LENGTH = Integer.MAX_VALUE ;
    
    
    /**
     * フォーマットヘッダ.
     */
    private static final char[] DATE_TIME_HEADER = {
        DateTimeFormat.CHAR_YEAR_2,
        DateTimeFormat.CHAR_YEAR_4,
        DateTimeFormat.CHAR_MONTH,
        DateTimeFormat.CHAR_DAY,
        DateTimeFormat.CHAR_HOUR,
        DateTimeFormat.CHAR_MINUTES,
        DateTimeFormat.CHAR_SECOND,
        DateTimeFormat.CHAR_MILLISECOND,
        DateTimeFormat.CHAR_WEEK_TO_ENGLISH,
        DateTimeFormat.CHAR_WEEK_TO_JAPAN,
        DateTimeFormat.CHAR_GREGORIO
    } ;
    
    /**
     * フォーマットチェックデータ長.
     */
    private static final int[] DATE_TIME_LENGTH = {
        DateTimeFormat.LEN_YEAR_2,
        DateTimeFormat.LEN_YEAR_4,
        DateTimeFormat.LEN_MONTH,
        DateTimeFormat.LEN_DAY,
        DateTimeFormat.LEN_HOUR,
        DateTimeFormat.LEN_MINUTES,
        DateTimeFormat.LEN_SECOND,
        DateTimeFormat.LEN_MILLISECOND,
        DateTimeFormat.LEN_WEEK_TO_ENGLISH,
        DateTimeFormat.LEN_WEEK_TO_JAPAN,
        DateTimeFormat.LEN_GREGORIO
    } ;
    
    /**
     * フォーマット数値定義.
     */
    private static final int[] DATE_TIME_DEFINES = {
        DateTimeFormat.TYPE_YEAR_2,
        DateTimeFormat.TYPE_YEAR_4,
        DateTimeFormat.TYPE_MONTH,
        DateTimeFormat.TYPE_DAY,
        DateTimeFormat.TYPE_HOUR,
        DateTimeFormat.TYPE_MINUTES,
        DateTimeFormat.TYPE_SECOND,
        DateTimeFormat.TYPE_MILLISECOND,
        DateTimeFormat.TYPE_WEEK_TO_ENGLISH,
        DateTimeFormat.TYPE_WEEK_TO_JAPAN,
        DateTimeFormat.TYPE_GREGORIO
    } ;
    
    
    
    /**
     * フォーマットデータ.
     */
    private String m_format = null ;
    
    /**
     * フォーマットデータ長.
     */
    private int m_length = 0 ;
    
    /**
     * 解析フォーマット種別.
     */
    private int[] m_class = null ;
    
    /**
     * 解析フォーマット規定データ.
     */
    private String[] m_reg = null ;
    
    /**
     * フォーマット種別マスク.
     */
    private int m_clsMask = 0 ;
    
    /**
     * 日付保持情報.
     */
    private final DateTime m_timestamp = new DateTime() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public DateTimeFormat()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * フォーマット情報を構成する設定を行います.
     * <BR>
     * @param format 対象のフォーマット情報を設定します.
     */
    public DateTimeFormat( String format )
    {
        try{
            this.create( format ) ;
        }catch( Exception t ){
            this.clear() ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * フォーマット情報を構成する設定を行います.
     * <BR>
     * @param format 対象のフォーマット情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public final void create( String format )
        throws InputException,ReadSyntaxException
    {
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                this.analisys_Format( format ) ;
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 格納情報をクリアします.
     */
    public final void clear()
    {
        m_sync.clear() ;
        
        m_format = null ;
        m_length = 0 ;
        m_class = null ;
        m_reg = null ;
        m_clsMask = 0 ;
        m_timestamp.clear() ;
        
    }
    
    /**
     * 現在の日付・時間情報をセット.
     * <BR><BR>
     * 現在の日付・時間情報を設定します.
     */
    public final void setNow()
    {
        try{
            synchronized( m_sync.get() ){
                m_timestamp.nowTimestamp() ;
            }
        }catch( NullPointerException nul ){
        }
    }
    
    /**
     * 指定日付・時間情報をセット.
     * <BR><BR>
     * 指定日付・時間情報が格納されているオブジェクトを設定します.
     * <BR>
     * @param time 対象の日付・時間情報が格納されたオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void set( Calendar time )
        throws InputException
    {
        try{
            synchronized( m_sync.get() ){
                m_timestamp.setCalendarObject( time ) ;
            }
        }catch( NullPointerException nul ){
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 指定日付情報をセット.
     * <BR><BR>
     * 指定日付・時間情報が格納されているオブジェクトを設定します.
     * <BR>
     * @param time 対象の日付・時間情報が格納されたオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void set( DateTime time )
        throws InputException
    {
        if( time == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_timestamp.setTime( time.getTime() ) ;
            }
        }catch( NullPointerException nul ){
        }
    }
    
    /**
     * 指定日付情報をセット.
     * <BR><BR>
     * 指定日付・時間情報が格納されているオブジェクトを設定します.
     * <BR>
     * @param time タイマー値が格納された情報を設定します.
     */
    public final void set( long time )
    {
        try{
            synchronized( m_sync.get() ){
                m_timestamp.setTime( time ) ;
            }
        }catch( NullPointerException nul ){
        }
    }
    
    /**
     * フォーマットに則った解析対象情報を設定.
     * <BR><BR>
     * フォーマットに則った解析対象情報を設定します.
     * <BR>
     * @param format 解析対象の文字列を指定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public final void set( String format )
        throws InputException,ReadSyntaxException
    {
        int i,j ;
        int len ;
        int formatLen ;
        int index ;
        int addIndex ;
        int tmp ;
        
        int[] cls = null ;
        String[] reg = null ;
        String nowChk = null ;
        DateTime dt = null ;
        
        if( format == null || ( formatLen = format.length() ) <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        nowChk = format ;
        dt = new DateTime() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                cls = m_class ;
                reg = m_reg ;
                dt.clear() ;
                
                len = reg.length ;
                for( i = 0,index = 0 ; i < len ; i ++ ){
                    
                    nowChk = format.substring( index ) ;
                    
                    switch( cls[ i ] ){
                        case DateTimeFormat.TYPE_YEAR_2 :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の年(2)情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_YEAR_2 ){
                                addIndex = DateTimeFormat.LEN_YEAR_2 ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) +
                                ( ( Calendar.getInstance().get( Calendar.YEAR ) / 100 ) * 100 ) ;
                            dt.setYear( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_YEAR_4 :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の年(4)情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_YEAR_4 ){
                                addIndex = DateTimeFormat.LEN_YEAR_4 ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setYear( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_MONTH :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の月情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_MONTH ){
                                addIndex = DateTimeFormat.LEN_MONTH ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setMonth( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_DAY :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の日情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_DAY ){
                                addIndex = DateTimeFormat.LEN_DAY ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setDay( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_HOUR :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の時情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_HOUR ){
                                addIndex = DateTimeFormat.LEN_HOUR ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setHour( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_MINUTES :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の分情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_MINUTES ){
                                addIndex = DateTimeFormat.LEN_MINUTES ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setMinutes( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_SECOND :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内の秒情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_SECOND ){
                                addIndex = DateTimeFormat.LEN_SECOND ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setSecond( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_MILLISECOND :
                            addIndex = DateTimeFormat.checkIndexNumber( 0,nowChk ) ;
                            if( addIndex == -1 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内のミリ秒情報は" +
                                    "は存在しません(index : " + index +
                                    ")"
                                ) ;
                            }
                            else if( addIndex >= DateTimeFormat.LEN_MILLISECOND ){
                                addIndex = DateTimeFormat.LEN_MILLISECOND ;
                            }
                            
                            tmp = Integer.parseInt( nowChk.substring( 0,addIndex ) ) ;
                            dt.setMilliSecond( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_WEEK_TO_ENGLISH :
                        case DateTimeFormat.TYPE_WEEK_TO_JAPAN :
                        case DateTimeFormat.TYPE_GREGORIO :
                            
                            for( tmp = i,addIndex = 0 ;; tmp ++ ){
                                
                                if( len > tmp ){
                                    
                                    if( cls[ tmp ] == DateTimeFormat.TYPE_GREGORIO ){
                                        if(
                                            nowChk.indexOf( DateTime.EAR_AD.charAt( 0 ) ) >
                                            nowChk.indexOf( DateTime.EAR_BC.charAt( 0 ) )
                                        )
                                        {
                                            if( nowChk.indexOf( DateTime.EAR_AD.charAt( 0 ) ) == -1 ){
                                                throw new ReadSyntaxException(
                                                    "対象のフォーマット(" + format +
                                                    "内の、西暦区分(" + DateTime.EAR_AD +
                                                    "は存在しません(index : " + index +
                                                    ")"
                                                ) ;
                                            }
                                            dt.setGregorioMode( true ) ;
                                        }
                                        else{
                                            if( nowChk.indexOf( DateTime.EAR_BC.charAt( 0 ) ) == -1 ){
                                                throw new ReadSyntaxException(
                                                    "対象のフォーマット(" + format +
                                                    "内の、西暦区分(" + DateTime.EAR_BC +
                                                    "は存在しません(index : " + index +
                                                    ")"
                                                ) ;
                                            }
                                            dt.setGregorioMode( false ) ;
                                        }
                                    }
                                    else if(
                                        cls[ tmp ] != DateTimeFormat.TYPE_WEEK_TO_ENGLISH &&
                                        cls[ tmp ] != DateTimeFormat.TYPE_WEEK_TO_JAPAN
                                    )
                                    {
                                        if( cls[ tmp ] == DateTimeFormat.TYPE_NOT_FORMAT ){
                                            addIndex = DateTimeFormat.checkIndex( addIndex,nowChk,reg[ tmp ] ) ;
                                            break ;
                                        }
                                        else{
                                            addIndex = DateTimeFormat.getIndexNumber( addIndex,nowChk ) ;
                                            break ;
                                        }
                                    }
                                    
                                }
                            }
                            break ;
                            
                        case DateTimeFormat.TYPE_NOT_FORMAT :
                            addIndex = DateTimeFormat.checkIndex( 0,nowChk,reg[ i ] ) ;
                            if( addIndex != 0 ){
                                throw new ReadSyntaxException(
                                    "対象のフォーマット(" + format +
                                    "内に、フォーマット区分(" + reg[ i ] +
                                    "は存在しないか不正な情報が存在します(index : " + index +
                                    ")"
                                ) ;
                            }
                            addIndex = reg[ i ].length() ;
                            break ;
                        default :
                            throw new ReadSyntaxException(
                                "対象のフォーマット(" + format +
                                "内に、不正なフォーマットタイプ(" + reg[ i ] +
                                "が存在します(index : " + index +
                                ")"
                            ) ;
                    }
                    index += addIndex ;
                    
                }
                
            }
            
            m_timestamp.setTime( dt.getTime() ) ;
            
        }catch( NullPointerException nul ){
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }finally{
            try{
                dt.clear() ;
            }catch( Exception t ){
            }
            cls = null ;
            reg = null ;
            nowChk = null ;
            dt = null ;
        }
        
    }
    
    /**
     * 格納日付・時間管理情報を取得.
     * <BR><BR>
     * 格納されている日付・時間管理情報を取得します.
     * <BR>
     * @return long 管理されている日付を示す値が返されます.
     */
    public final long getTime()
    {
        long ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_timestamp.getTime() ;
            }
            
        }catch( NullPointerException nul ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 格納日付・時間管理情報を取得.
     * <BR><BR>
     * 格納されている日付・時間管理情報を取得します.
     * <BR>
     * @return DateTime 日付・時間情報が格納されているオブジェクトが
     *                  返されます.
     */
    public final DateTime get()
    {
        DateTime ret = null ;
        
        try{
            
            ret = new DateTime() ;
            
            synchronized( m_sync.get() ){
                ret.setTime( m_timestamp.getTime() ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * フォーマットに則った格納日付・時間情報を文字列で取得.
     * <BR><BR>
     * フォーマットに則った格納日付・時間情報を文字列で取得します.
     * <BR>
     * @param time 対象のタイム値を設定します.
     * @return String フォーマットに則った情報が返されます.
     */
    public final String getString( long time )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                this.set( time ) ;
                ret = this.getString() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * フォーマットに則った格納日付・時間情報を文字列で取得.
     * <BR><BR>
     * フォーマットに則った格納日付・時間情報を文字列で取得します.
     * <BR>
     * @return String フォーマットに則った情報が返されます.
     */
    public final String getString()
    {
        int i ;
        int len ;
        int tmpLen ;
        
        StringBuffer buf = null ;
        String tmp = null ;
        String ret = null ;
        
        int[] cls = null ;
        String[] reg = null ;
        DateTime dtime = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                buf = new StringBuffer( m_length * 2 ) ;
                
                dtime = m_timestamp ;
                cls = m_class ;
                reg = m_reg ;
                len = reg.length ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    switch( cls[ i ] ){
                        case DateTimeFormat.TYPE_YEAR_2 :
                            tmp = String.valueOf( dtime.getYear() ) ;
                            tmpLen = tmp.length() - DateTimeFormat.LEN_YEAR_2 ;
                            tmp = tmp.substring( ( tmpLen < 0 ) ? 0 : tmpLen ) ;
                            buf.append( DateTimeFormat.ADD_REG_2.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_YEAR_4 :
                            tmp = String.valueOf( dtime.getYear() ) ;
                            tmpLen = tmp.length() - DateTimeFormat.LEN_YEAR_4 ;
                            tmp = tmp.substring( ( tmpLen < 0 ) ? 0 : tmpLen ) ;
                            buf.append( DateTimeFormat.ADD_REG_4.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_MONTH :
                            tmp = String.valueOf( dtime.getMonth() ) ;
                            buf.append( DateTimeFormat.ADD_REG_2.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_DAY :
                            tmp = String.valueOf( dtime.getDay() ) ;
                            buf.append( DateTimeFormat.ADD_REG_2.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_HOUR :
                            tmp = String.valueOf( dtime.getHour() ) ;
                            buf.append( DateTimeFormat.ADD_REG_2.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_MINUTES :
                            tmp = String.valueOf( dtime.getMinutes() ) ;
                            buf.append( DateTimeFormat.ADD_REG_2.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_SECOND :
                            tmp = String.valueOf( dtime.getSecond() ) ;
                            buf.append( DateTimeFormat.ADD_REG_2.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_MILLISECOND :
                            tmp = String.valueOf( dtime.getMilliSecond() ) ;
                            buf.append( DateTimeFormat.ADD_REG_4.substring( tmp.length() ) ) ;
                            buf.append( tmp ) ;
                            break ;
                        case DateTimeFormat.TYPE_WEEK_TO_ENGLISH :
                            buf.append( dtime.getWeek( false ) ) ;
                            break ;
                        case DateTimeFormat.TYPE_WEEK_TO_JAPAN :
                            buf.append( dtime.getWeek( true ) ) ;
                            break ;
                        case DateTimeFormat.TYPE_GREGORIO :
                            buf.append( dtime.getGregorio() ) ;
                            break ;
                        case DateTimeFormat.TYPE_NOT_FORMAT :
                            buf.append( reg[ i ] ) ;
                            break ;
                        default :
                            buf.append( DateTimeFormat.ETC_TYPE_CODE ) ;
                            break ;
                    }
                    
                }
                
                ret = buf.toString() ;
                
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }finally{
            buf = null ;
            tmp = null ;
            cls = null ;
            reg = null ;
            dtime = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納フォーマット情報の取得.
     * <BR><BR>
     * 格納フォーマット情報を取得します.
     * <BR>
     * @return String フォーマット情報が返されます.
     */
    public final String getFormat()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_format ;
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定タイプ情報がフォーマットに含まれているかチェック.
     * <BR><BR>
     * 指定タイプ情報がフォーマットに含まれているかチェックします.
     * <BR>
     * @param type タイプ情報を設定します.<BR>
     *             [DateTimeFormat#TYPE_YEAR_2]のタイプは、西暦２桁情報です.<BR>
     *             [DateTimeFormat#TYPE_YEAR_4]のタイプは、西暦４桁情報です.<BR>
     *             [DateTimeFormat#TYPE_MONTH]のタイプは、月情報です.<BR>
     *             [DateTimeFormat#TYPE_DAY]のタイプは、日付情報です.<BR>
     *             [DateTimeFormat#TYPE_HOUR]のタイプは、時間情報です.<BR>
     *             [DateTimeFormat#TYPE_MINUTES]のタイプは、分情報です.<BR>
     *             [DateTimeFormat#TYPE_SECOND]のタイプは、秒情報です.<BR>
     *             [DateTimeFormat#TYPE_MILLISECOND]のタイプは、ミリ秒情報です.<BR>
     *             [DateTimeFormat#TYPE_WEEK_TO_ENGLISH]のタイプは週情報(英)です.<BR>
     *             [DateTimeFormat#TYPE_WEEK_TO_JAPAN]のタイプは週情報(日)です.<BR>
     *             [DateTimeFormat#TYPE_GREGORIO]のタイプはグレゴリオ暦です.<BR>
     *             これらチェック対象に対するタイプ情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象の情報は存在します.<BR>
     *                 [false]が返された場合、対象の情報は存在しません.
     * @exception InputException 入力例外.
     */
    public final boolean isType( int type ) throws InputException
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( ( m_clsMask & type ) == type ) ? true : false ;
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * フォーマットに則った格納日付・時間管理情報を取得.
     * <BR><BR>
     * フォーマットに則った格納日付・時間管理情報を取得します.
     * <BR>
     * @return String フォーマットに則った情報が返されます.
     */
    public final String toString()
    {
        return this.getString() ;
    }
    
    
    
    
    /**
     * フォーマット解析処理.
     */
    private final void analisys_Format( String format )
        throws InputException,ReadSyntaxException
    {
        int i ;
        int len ;
        int pnt ;
        int sx,befEd ;
        int type,beforeType ;
        int nowLen ;
        int chkLen ;
        int mask ;
        char nowChk ;
        
        ObjectArray objAry = null ;
        IntArray typeAry = null ;
        Object[] tmp = null ;
        int[] intTmp = null ;
        
        char[] chrFormat = null ;
        
        if( format == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        len = format.length() ;
        chrFormat = new char[ len ] ;
        format.getChars( 0,len,chrFormat,0 ) ;
        
        typeAry = new IntArray() ;
        objAry = new ObjectArray() ;
        pnt = 0 ;
        
        try{
            
            for(
                i = 0,
                sx = 0,
                mask = 0,
                befEd = 0,
                nowLen = 0,
                nowChk = DateTimeFormat.CHAR_NOT,
                chkLen = DateTimeFormat.NOT_CHK_LENGTH,
                type = DateTimeFormat.TYPE_NOT_FORMAT,
                beforeType = DateTimeFormat.TYPE_NOT_FORMAT
                ; i < len ; i ++,pnt++
            )
            {
                
                switch( chrFormat[ i ] ){
                    case DateTimeFormat.CHAR_YEAR_2 :
                        if( nowChk != DateTimeFormat.CHAR_YEAR_2 ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_YEAR_2 ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_YEAR_2 ;
                            chkLen = DateTimeFormat.LEN_YEAR_2 ;
                            type = DateTimeFormat.TYPE_YEAR_2 ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_YEAR_4 :
                        if( nowChk != DateTimeFormat.CHAR_YEAR_4 ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_YEAR_4 ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_YEAR_4 ;
                            chkLen = DateTimeFormat.LEN_YEAR_4 ;
                            type = DateTimeFormat.TYPE_YEAR_4 ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_MONTH :
                        if( nowChk != DateTimeFormat.CHAR_MONTH ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_MONTH ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_MONTH ;
                            chkLen = DateTimeFormat.LEN_MONTH ;
                            type = DateTimeFormat.TYPE_MONTH ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_DAY :
                        if( nowChk != DateTimeFormat.CHAR_DAY ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_DAY ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_DAY ;
                            chkLen = DateTimeFormat.LEN_DAY ;
                            type = DateTimeFormat.TYPE_DAY ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_HOUR :
                        if( nowChk != DateTimeFormat.CHAR_HOUR ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_HOUR ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_HOUR ;
                            chkLen = DateTimeFormat.LEN_HOUR ;
                            type = DateTimeFormat.TYPE_HOUR ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_MINUTES :
                        if( nowChk != DateTimeFormat.CHAR_MINUTES ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_MINUTES ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_MINUTES ;
                            chkLen = DateTimeFormat.LEN_MINUTES ;
                            type = DateTimeFormat.TYPE_MINUTES ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_SECOND :
                        if( nowChk != DateTimeFormat.CHAR_SECOND ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_SECOND ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_SECOND ;
                            chkLen = DateTimeFormat.LEN_SECOND ;
                            type = DateTimeFormat.TYPE_SECOND ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_MILLISECOND :
                        if( nowChk != DateTimeFormat.CHAR_MILLISECOND ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_MILLISECOND ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_MILLISECOND ;
                            chkLen = DateTimeFormat.LEN_MILLISECOND ;
                            type = DateTimeFormat.TYPE_MILLISECOND ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_WEEK_TO_ENGLISH :
                        if( nowChk != DateTimeFormat.CHAR_WEEK_TO_ENGLISH ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_WEEK_TO_ENGLISH ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_WEEK_TO_ENGLISH ;
                            chkLen = DateTimeFormat.LEN_WEEK_TO_ENGLISH ;
                            type = DateTimeFormat.TYPE_WEEK_TO_ENGLISH ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_WEEK_TO_JAPAN :
                        if( nowChk != DateTimeFormat.CHAR_WEEK_TO_JAPAN ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_WEEK_TO_JAPAN ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_WEEK_TO_JAPAN ;
                            chkLen = DateTimeFormat.LEN_WEEK_TO_JAPAN ;
                            type = DateTimeFormat.TYPE_WEEK_TO_JAPAN ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    case DateTimeFormat.CHAR_GREGORIO :
                        if( nowChk != DateTimeFormat.CHAR_GREGORIO ){
                            beforeType = ( nowChk != DateTimeFormat.CHAR_NOT ) ?
                                DateTimeFormat.TYPE_NOT_FORMAT : DateTimeFormat.TYPE_GREGORIO ;
                            sx = i ;
                            nowLen = 1 ;
                            nowChk = DateTimeFormat.CHAR_GREGORIO ;
                            chkLen = DateTimeFormat.LEN_GREGORIO ;
                            type = DateTimeFormat.TYPE_GREGORIO ;
                        }
                        else{
                            nowLen ++ ;
                        }
                        break ;
                    default :
                        
                        sx = 0 ;
                        nowLen = 0 ;
                        nowChk = DateTimeFormat.CHAR_ETC ;
                        chkLen = DateTimeFormat.NOT_CHK_LENGTH ;
                        type = DateTimeFormat.TYPE_NOT_FORMAT ;
                        beforeType = DateTimeFormat.TYPE_NOT_FORMAT ;
                        break ;
                        
                }
                
                if( nowLen >= chkLen ){
                    
                    if( beforeType == DateTimeFormat.TYPE_NOT_FORMAT ){
                        
                        typeAry.add( DateTimeFormat.TYPE_NOT_FORMAT ) ;
                        objAry.add( format.substring( befEd,sx ) ) ;
                        
                    }
                    
                    mask |= type ;
                    typeAry.add( type ) ;
                    objAry.add( "" ) ;
                    
                    befEd = sx + chkLen ;
                    sx = 0 ;
                    nowLen = 0 ;
                    nowChk = DateTimeFormat.CHAR_NOT ;
                    chkLen = DateTimeFormat.NOT_CHK_LENGTH ;
                    beforeType = type ;
                    type = DateTimeFormat.TYPE_NOT_FORMAT ;
                    
                }
            }
            
            if( beforeType == DateTimeFormat.TYPE_NOT_FORMAT ){
                
                typeAry.add( DateTimeFormat.TYPE_NOT_FORMAT ) ;
                objAry.add( format.substring( befEd,len ) ) ;
                
            }
            
            m_format = format ;
            m_length = m_format.length() ;
            m_clsMask = mask ;
            m_class = typeAry.getObjects() ;
            tmp = objAry.getObjects() ;
            intTmp = typeAry.getObjects() ;
            
            len = objAry.size() ;
            m_reg = new String[ len ] ;
            m_class = new int[ len ] ;
            
            for( i = 0 ; i < len ; i ++ ){
                m_reg[ i ] = ( String )tmp[ i ] ;
                m_class[ i ] = intTmp[ i ] ;
            }
            
        }catch( InputException in ){
            
            this.clear() ;
            
            throw new ReadSyntaxException(
                "フォーマット情報(" + format +
                ")の位置[" + pnt +
                "]で文法不正です"
            ) ;
            
        }finally{
            
            objAry = null ;
            typeAry = null ;
            tmp = null ;
            intTmp = null ;
            chrFormat = null ;
            
        }
        
    }
    
    /**
     * 対象文字列が、指定文字列に格納されている場合、
     * その先頭インデックス値を取得.
     * 対象文字列が該当しない場合、[-1]が返されます.
     */
    private static final int checkIndex( int index,String format,String chk )
    {
        int i,j ;
        int len ;
        int chkLen ;
        int ret ;
        
        char[] formatChr = null ;
        char[] chkChr = null ;
        
        len = format.length() ;
        chkLen = chk.length() ;
        
        if( chkLen <= 0 || len <= 0 || index >= len ){
            if( chkLen <= 0 ){
                ret = -1 ;
            }
            ret = 0 ;
        }
        else{
            
            formatChr = new char[ len ] ;
            chkChr = new char[ chkLen ] ;
            
            format.getChars( 0,len,formatChr,0 ) ;
            chk.getChars( 0,chkLen,chkChr,0 ) ;
            
            for( i = index,j = 0,ret = -1 ; i < len ; i ++ ){
                
                if( formatChr[ i ] == chkChr[ j ] ){
                    
                    ret = ( ret == -1 ) ? i : ret ;
                    
                    j ++ ;
                    
                    if( chkLen >= j ){
                        break ;
                    }
                }
                else{
                    ret = -1 ;
                }
            }
            
            formatChr = null ;
            chkChr = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 対象の文字列の指定位置からどこまで数値情報であるかチェック.
     * 数値情報が存在する文字の長さが返されます.
     * 情報が存在しない場合[-1]が返されます.
     */
    private static final int checkIndexNumber( int index,String format )
    {
        int i ;
        int len ;
        int ret ;
        
        char[] formatChr = null ;
        
        len = format.length() ;
        
        if( len <= 0 || index >= len ){
            ret = -1 ;
        }
        else{
            
            formatChr = new char[ len ] ;
            format.getChars( 0,len,formatChr,0 ) ;
            
            for( i = index,ret = 0 ; i < len ; i ++ ){
                if( formatChr[ i ] < '0' || formatChr[ i ] > '9' ){
                    ret = ( ret == 0 ) ? -1 : ret ;
                    break ;
                }
                ret ++ ;
            }
            
        }
        
        formatChr = null ;
        
        return ret ;
    }
    
    /**
     * 文字列内のインデックス位置以降に数値文字コードが存在するかチェック.
     * 取得された情報は、文字列内の数値文字コードの直前の位置を示します.
     * 情報が存在しない場合[-1]が返されます.
     */
    private static final int getIndexNumber( int index,String format )
    {
        int i ;
        int len ;
        int ret ;
        
        char[] formatChr = null ;
        
        len = format.length() ;
        
        if( len == 0 || index >= len ){
            ret = -1 ;
        }
        else{
            
            formatChr = new char[ len ] ;
            format.getChars( 0,len,formatChr,0 ) ;
            
            for( i = index,ret = -1 ; i < len ; i ++ ){
                if( formatChr[ i ] >= '0' && formatChr[ i ] <= '9' ){
                    ret = i ;
                    break ;
                }
            }
            
        }
        
        formatChr = null ;
        
        return ret ;
    }
    
}


