/*
 * @(#)DateTime.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import java.io.Serializable;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * 日付、時間情報管理.
 * <BR><BR>
 * 日付、時間情報をサポートします.
 *  
 * @version 1.0.0 2003/11/01
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class DateTime implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            DateTime.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 週・日本語表示(日曜日).
     * <BR><BR>
     * 週情報(日)を日本語表示します.
     */
    public static final String WEEK_JP_SUNDAY = "日" ;
    
    /**
     * 週・英語表示(日曜日).
     * <BR><BR>
     * 週情報(日)を英語表示します.
     */
    public static final String WEEK_ENG_SUNDAY = "SUN" ;
    
    /**
     * 週・日本語表示(月曜日).
     * <BR><BR>
     * 週情報(月)を日本語表示します.
     */
    public static final String WEEK_JP_MONDAY = "月" ;
    
    /**
     * 週・英語表示(月曜日).
     * <BR><BR>
     * 週情報(月)を英語表示します.
     */
    public static final String WEEK_ENG_MONDAY = "MON" ;
    
    /**
     * 週・日本語表示(火曜日).
     * <BR><BR>
     * 週情報(火)を日本語表示します.
     */
    public static final String WEEK_JP_TUESDAY = "火" ;
    
    /**
     * 週・英語表示(火曜日).
     * <BR><BR>
     * 週情報(火)を英語表示します.
     */
    public static final String WEEK_ENG_TUESDAY = "TUE" ;
    
    /**
     * 週・日本語表示(水曜日).
     * <BR><BR>
     * 週情報(水)を日本語表示します.
     */
    public static final String WEEK_JP_WEDNESDAY = "水" ;
    
    /**
     * 週・英語表示(水曜日).
     * <BR><BR>
     * 週情報(水)を英語表示します.
     */
    public static final String WEEK_ENG_WEDNESDAY = "WED" ;
    
    
    /**
     * 週・日本語表示(木曜日).
     * <BR><BR>
     * 週情報(木)を日本語表示します.
     */
    public static final String WEEK_JP_THURSDAY = "木" ;
    
    /**
     * 週・英語表示(木曜日).
     * <BR><BR>
     * 週情報(木)を英語表示します.
     */
    public static final String WEEK_ENG_THURSDAY = "THU" ;
    
    /**
     * 週・日本語表示(金曜日).
     * <BR><BR>
     * 週情報(金)を日本語表示します.
     */
    public static final String WEEK_JP_FRIDAY = "金" ;
    
    /**
     * 週・英語表示(金曜日).
     * <BR><BR>
     * 週情報(金)を英語表示します.
     */
    public static final String WEEK_ENG_FRIDAY = "FRI" ;
    
    /**
     * 週・日本語表示(土曜日).
     * <BR><BR>
     * 週情報(土)を日本語表示します.
     */
    public static final String WEEK_JP_SATURDAY = "土" ;
    
    /**
     * 週・英語表示(土曜日).
     * <BR><BR>
     * 週情報(土)を英語表示します.
     */
    public static final String WEEK_ENG_SATURDAY = "SAT" ;
    
    /**
     * グレゴリオ暦 : 西暦.
     */
    public static final String EAR_AD = "A.D" ;
    
    /**
     * グレゴリオ暦 : 紀元前.
     */
    public static final String EAR_BC = "B.C" ;
    
    /**
     * 日時・時間変換 : その他.
     */
    public static final String ETC_CODE = "?" ;
    
    
    
    /**
     * カレンダオブジェクト.
     */
    private Calendar m_cal = null ;
    
    /**
     * ロケール.
     */
    private Locale m_locale = null ;
    
    /**
     * タイムゾーン.
     */
    private TimeZone m_timezone = null ;
    
    /**
     * グレゴリオ暦.
     */
    private String m_ear = DateTime.ETC_CODE ;
    
    /**
     * 週情報 : 日本語記述.
     */
    private String m_weekJp = DateTime.ETC_CODE ;
    
    /**
     * 週情報 : 英語記述.
     */
    private String m_weekEng = DateTime.ETC_CODE ;
    
    /**
     * コンストラクタ.
     */
    public DateTime()
    {
        m_locale = null ;
        m_timezone = null ;
        this.clear() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * Localeを設定してオブジェクトを生成します.
     * <BR>
     * @param locale 対象のLocaleオブジェクトを設定します.
     */
    public DateTime( Locale locale )
    {
        m_locale = locale ;
        m_timezone = null ;
        this.clear() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * TimeZoneを設定してオブジェクトを生成します.
     * <BR>
     * @param timezone 対象のTimeZoneオブジェクトを設定します.
     */
    public DateTime( TimeZone timezone )
    {
        m_locale = null ;
        m_timezone = timezone ;
        this.clear() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * LocaleとTimeZoneを設定してオブジェクトを生成します.
     * <BR>
     * @param locale 対象のLocaleオブジェクトを設定します.
     * @param timezone 対象のTimeZoneオブジェクトを設定します.
     */
    public DateTime( Locale locale,TimeZone timezone )
    {
        m_locale = locale ;
        m_timezone = timezone ;
        this.clear() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_cal = null ;
        m_ear = DateTime.ETC_CODE ;
        m_weekJp = DateTime.ETC_CODE ;
        m_weekEng = DateTime.ETC_CODE ;
        
        if( m_locale != null ){
            if( m_timezone != null ){
                m_cal = new GregorianCalendar( m_timezone,m_locale ) ;
            }
            else{
                m_cal = new GregorianCalendar( m_locale ) ;
            }
        }
        else if( m_timezone != null ){
            m_cal = new GregorianCalendar( m_timezone ) ;
        }
        else{
            m_cal = new GregorianCalendar() ;
        }
        
        m_cal.clear() ;
        
    }
    
    /**
     * グレゴリオ暦モードの設定.
     * <BR><BR>
     * グレゴリオ暦モードを設定します.
     * <BR>
     * @param mode グレゴリオ暦モードを設定します.<BR>
     *             [true]を設定した場合、西暦(AD)を設定します.<BR>
     *             [false]を設定した場合、紀元前(BC)を設定します.
     */
    public final void setGregorioMode( boolean mode )
    {
        if( mode == true ){
            m_cal.set(Calendar.ERA,GregorianCalendar.AD ) ;
            m_ear = DateTime.EAR_AD ;
        }
        else{
            m_cal.set(Calendar.ERA,GregorianCalendar.BC ) ;
            m_ear = DateTime.EAR_BC ;
        }
    }
    
    /**
     * グレゴリオ暦情報の取得.
     * <BR><BR>
     * グレゴリオ暦情報を取得します.
     * <BR>
     * @return boolean グレゴリオ暦モードが返されます.<BR>
     *                 [true]が返された場合、西暦(AD)です.<BR>
     *                 [false]が返された場合、紀元前(BC)です.
     */
    public final boolean getGregorioMode()
    {
        return ( m_cal.get(Calendar.ERA) == GregorianCalendar.AD ) ? true : false ;
    }
    
    /**
     * 年情報設定.
     * <BR><BR>
     * 年情報を設定します.
     * <BR>
     * @param year 設定対象の年情報をセットします.
     */
    public final void setYear(int year)
    {
        m_cal.set(Calendar.YEAR,year) ;
    }
    
    /**
     * 年情報取得.
     * <BR><BR>
     * 年情報を取得します.
     * <BR>
     * @return int 格納年情報が取得されます.
     */
    public final int getYear()
    {
        return m_cal.get(Calendar.YEAR) ;
    }
    
    /**
     * 月情報設定.
     * <BR><BR>
     * 月情報を設定します.
     * <BR>
     * @param month 設定対象の月情報をセットします.
     */
    public final void setMonth(int month)
    {
        m_cal.set(Calendar.MONTH,month-1) ;
    }
    
    /**
     * 月情報取得.
     * <BR><BR>
     * 月情報を取得します.
     * <BR>
     * @return int 格納月情報が取得されます.
     */
    public final int getMonth()
    {
        return m_cal.get(Calendar.MONTH) + 1 ;
    }
    
    /**
     * 日情報設定.
     * <BR><BR>
     * 日情報を設定します.
     * <BR>
     * @param day 設定対象の日情報をセットします.
     */
    public final void setDay(int day)
    {
        m_cal.set(Calendar.DAY_OF_MONTH,day) ;
    }
    
    /**
     * 日情報取得.
     * <BR><BR>
     * 日情報を取得します.
     * <BR>
     * @return int 格納日情報が取得されます.
     */
    public final int getDay()
    {
        return m_cal.get(Calendar.DAY_OF_MONTH) ;
    }
    
    /**
     * 時情報設定.
     * <BR><BR>
     * 時情報を設定します.
     * <BR>
     * @param hour 設定対象の時情報をセットします.
     */
    public final void setHour(int hour)
    {
        m_cal.set(Calendar.HOUR_OF_DAY,hour) ;
    }
    
    /**
     * 時情報取得.
     * <BR><BR>
     * 時情報を取得します.
     * <BR>
     * @return int 格納時情報が取得されます.
     */
    public final int getHour()
    {
        return m_cal.get(Calendar.HOUR_OF_DAY) ;
    }
    
    /**
     * 分情報設定.
     * <BR><BR>
     * 分情報を設定します.
     * <BR>
     * @param minutes 設定対象の分情報をセットします.
     */
    public final void setMinutes(int minutes)
    {
        m_cal.set(Calendar.MINUTE,minutes) ;
    }
    
    /**
     * 分情報取得.
     * <BR><BR>
     * 分情報を取得します.
     * <BR>
     * @return int 格納分情報が取得されます.
     */
    public final int getMinutes()
    {
        return m_cal.get(Calendar.MINUTE) ;
    }
    
    /**
     * 秒情報設定.
     * <BR><BR>
     * 秒情報を設定します.
     * <BR>
     * @param second 設定対象の秒情報をセットします.
     */
    public final void setSecond(int second)
    {
        m_cal.set(Calendar.SECOND,second) ;
    }
    
    /**
     * 秒情報取得.
     * <BR><BR>
     * 秒情報を取得します.
     * <BR>
     * @return int 格納秒情報が取得されます.
     */
    public final int getSecond()
    {
        return m_cal.get(Calendar.SECOND) ;
    }
    
    /**
     * ミリ秒情報設定.
     * <BR><BR>
     * ミリ秒情報を設定します.
     * <BR>
     * @param milliSecond 設定対象のミリ秒情報をセットします.
     */
    public final void setMilliSecond(int milliSecond)
    {
        m_cal.set(Calendar.MILLISECOND,milliSecond) ;
    }
    
    /**
     * ミリ秒情報取得.
     * <BR><BR>
     * ミリ秒情報を取得します.
     * <BR>
     * @return int 格納ミリ秒情報が取得されます.
     */
    public final int getMilliSecond()
    {
        return m_cal.get(Calendar.MILLISECOND) ;
    }
    
    /**
     * 週情報を取得.
     * <BR><BR>
     * 週情報を取得します.
     * <BR>
     * @param mode 取得モードを設定します.<BR>
     *             [true]を設定した場合、日本語内容で取得されます.<BR>
     *             [false]を設定した場合、英語内容で取得されます.
     * @return Stirng 週情報が返されます.
     */
    public final String getWeek( boolean mode )
    {
        this.getWeekCode( m_cal ) ;
        return ( mode == true ) ? m_weekJp : m_weekEng ;
    }
    
    /**
     * グレゴリオ暦情報の取得.
     * <BR><BR>
     * グレゴリオ暦情報を取得します.
     * <BR>
     * @return String グレゴリオ暦が返されます.
     */
    public final String getGregorio()
    {
        this.setEarCode( m_cal ) ;
        return m_ear ;
    }
    
    /**
     * [long]を時刻として設定.
     * <BR><BR>
     * [long]を時刻として設定します.
     * <BR>
     * @param time 時刻(1970年1月1日からの差)を設定します.
     */
    public final void setTime( long time )
    {
        m_cal.clear() ;
        m_cal.setTime( new java.sql.Date( time ) ) ;
    }
    
    /**
     * 設定時刻を[long]で取得.
     * <BR><BR>
     * 設定時刻を[long]で取得します.
     * <BR>
     * @return long 設定時刻(1970年1月1日からの差)を取得します.
     */
    public final long getTime()
    {
        return m_cal.getTime().getTime() ;
    }
    
    /**
     * カレンダー情報を設定.
     * <BR><BR>
     * カレンダー情報を設定します.
     * <BR>
     * @param date 設定対象のカレンダーオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void setCalendarObject( Calendar date )
        throws InputException
    {
        if( date == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        DateTime.setCalendarInfo( date,m_cal ) ;
    }
    
    /**
     * カレンダー情報を取得.
     * <BR><BR>
     * カレンダー情報を取得します.
     * <BR>
     * @return Calendar カレンダーオブジェクトが返されます.
     */
    public final Calendar getCalendarObject()
    {
        return this.getCalendarInfo() ;
    }
    
    /**
     * 日付情報を設定.
     * <BR><BR>
     * 日付情報を設定します.
     * <BR>
     * @param date 設定対象の日付オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void setDateObject( java.sql.Date date )
        throws InputException
    {
        if( date == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.setTime( date.getTime() ) ;
    }
    
    /**
     * 日付情報を取得.
     * <BR><BR>
     * 日付情報を取得します.
     * <BR>
     * @return java.sql.Date 日付オブジェクトが返されます.
     */
    public final java.sql.Date getDateObject()
    {
        
        return new java.sql.Date( this.getTime() ) ;
    }
    
    /**
     * 時間情報を設定.
     * <BR><BR>
     * 時間情報を設定します.
     * <BR>
     * @param time 設定対象の時間オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void setTimeObject( java.sql.Time time )
        throws InputException
    {
        if( time == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.setTime( time.getTime() ) ;
    }
    
    /**
     * 時間情報を取得.
     * <BR><BR>
     * 時間情報を取得します.
     * <BR>
     * @return java.sql.Time 時間オブジェクトが返されます.
     */
    public final java.sql.Time getTimeObject()
    {
        
        return new java.sql.Time( this.getTime() ) ;
    }
    
    /**
     * タイムスタンプ情報を設定.
     * <BR><BR>
     * タイムスタンプ情報を設定します.
     * <BR>
     * @param time 設定対象のタイムスタンプオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void setTimestampObject( java.sql.Timestamp time )
        throws InputException
    {
        if( time == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.setTime( time.getTime() ) ;
    }
    
    /**
     * タイムスタンプ情報を取得.
     * <BR><BR>
     * タイムスタンプ情報を取得します.
     * <BR>
     * @return java.sql.Timestamp タイムスタンプオブジェクトが返されます.
     */
    public final java.sql.Timestamp getTimestampObject()
    {
        
        return new java.sql.Timestamp( this.getTime() ) ;
    }
    
    /**
     * 現在タイムスタンプセット.
     * <BR><BR>
     * 現在タイムスタンプをセットします.
     */
    public final void nowTimestamp()
    {
        this.setTime( System.currentTimeMillis() ) ;
    }
    
    /**
     * 文字表示.
     * <BR><BR>
     * オブジェクト内容を文字情報として取得します.
     * <BR>
     * @return String 文字情報が返されます.
     */
    public final String toString()
    {
        String ret = null ;
        StringBuffer buf = null ;
        Calendar cal = null ;
        
        buf = new StringBuffer() ;
        cal = m_cal ;
        buf.append( "[" ) ;
        buf.append( this.getGregorio() ) ;
        buf.append( "]" ) ;
        buf.append( cal.get(Calendar.YEAR) ) ;
        buf.append( "/" ) ;
        buf.append( cal.get(Calendar.MONTH)+1 ) ;
        buf.append( "/" ) ;
        buf.append( cal.get(Calendar.DAY_OF_MONTH) ) ;
        buf.append( " " ) ;
        buf.append( cal.get(Calendar.HOUR_OF_DAY) ) ;
        buf.append( ":" ) ;
        buf.append( cal.get(Calendar.MINUTE) ) ;
        buf.append( ":" ) ;
        buf.append( cal.get(Calendar.SECOND) ) ;
        buf.append( "." ) ;
        buf.append( cal.get(Calendar.MILLISECOND) ) ;
        buf.append( "(" ) ;
        buf.append( this.getWeek( true ) ) ;
        buf.append( ")" ) ;
        
        ret = buf.toString().trim() ;
        buf = null ;
        cal = null ;
        
        return ret ;
    }
    
    /**
     * カレンダー情報を変数セット.
     */
    private static final void setCalendarInfo( Calendar src,Calendar dest )
    {
        dest.clear() ;
        dest.set(Calendar.YEAR,src.get(Calendar.YEAR) ) ;
        dest.set(Calendar.MONTH,src.get(Calendar.MONTH) ) ;
        dest.set(Calendar.DAY_OF_MONTH,src.get(Calendar.DAY_OF_MONTH) ) ;
        dest.set(Calendar.HOUR_OF_DAY,src.get(Calendar.HOUR_OF_DAY) ) ;
        dest.set(Calendar.MINUTE,src.get(Calendar.MINUTE) ) ;
        dest.set(Calendar.SECOND,src.get(Calendar.SECOND) ) ;
        dest.set(Calendar.MILLISECOND,src.get(Calendar.MILLISECOND) ) ;
        dest.set(Calendar.ERA,src.get(Calendar.ERA) ) ;
    }
    
    /**
     * カレンダー情報を取得.
     */
    private final Calendar getCalendarInfo()
    {
        Calendar ret = null ;
        
        ret = new GregorianCalendar() ;
        DateTime.setCalendarInfo( m_cal,ret ) ;
        
        return ret ;
    }
    
    /**
     * 週情報取得.
     */
    private final void getWeekCode( Calendar date )
    {
        switch(date.get(Calendar.DAY_OF_WEEK)){
            case Calendar.SUNDAY :
                m_weekJp = WEEK_JP_SUNDAY ;
                m_weekEng = WEEK_ENG_SUNDAY ;
                break ;
            case Calendar.MONDAY :
                m_weekJp = WEEK_JP_MONDAY ;
                m_weekEng = WEEK_ENG_MONDAY ;
                break ;
            case Calendar.TUESDAY :
                m_weekJp = WEEK_JP_TUESDAY ;
                m_weekEng = WEEK_ENG_TUESDAY ;
                break ;
            case Calendar.WEDNESDAY :
                m_weekJp = WEEK_JP_WEDNESDAY ;
                m_weekEng = WEEK_ENG_WEDNESDAY ;
                break ;
            case Calendar.THURSDAY :
                m_weekJp = WEEK_JP_THURSDAY ;
                m_weekEng = WEEK_ENG_THURSDAY ;
                break ;
            case Calendar.FRIDAY :
                m_weekJp = WEEK_JP_FRIDAY ;
                m_weekEng = WEEK_ENG_FRIDAY ;
                break ;
            case Calendar.SATURDAY :
                m_weekJp = WEEK_JP_SATURDAY ;
                m_weekEng = WEEK_ENG_SATURDAY ;
                break ;
            default :
                m_weekJp = DateTime.ETC_CODE ;
                m_weekEng = DateTime.ETC_CODE ;
                break ;
        }
    }
    
    /**
     * グレゴリオ暦情報の取得.
     */
    private final void setEarCode( Calendar date )
    {
        switch( date.get(Calendar.ERA) ){
            case GregorianCalendar.AD :
                m_ear = DateTime.EAR_AD ;
                break ;
            case GregorianCalendar.BC :
                m_ear = DateTime.EAR_BC ;
                break ;
            default :
                m_ear = DateTime.ETC_CODE ;
                break ;
        }
    }
    
}

