/*
 * @(#)DataMonitorBase.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.SettingException;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;

/**
 * DataMoniter管理オブジェクト.
 * <BR><BR>
 * データ監視モニター管理オブジェクトを定義します.
 *
 * @version     1.00, 2003/12/04
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public abstract class DataMonitorBase extends ExecutionThread implements DataMonitor
{
    
    /**
     * 内部クラス.
     */
    class DataMonitorPause{
        
        /**
         * タイムアウト監視用.
         */
        public long SET_TIME = 0L ;
        
        /**
         * 監視用データ.
         */
        public Object MONITER_DATA = null ;
        
    }
    
    /**
     * タイムアウト無効値.
     */
    public static final long NOT_SET_TIMER = 0L ;
    
    /**
     * スリープタイミング.
     */
    private static final int SLEEP_TIMING = 150 ;
    
    
    /**
     * 監視スレッド.
     */
    private LoopThread m_thread = null ;
    
    /**
     * データ監視タイマー.
     */
    private int m_timer = 0 ;
    
    /**
     * 情報生成チェック.
     */
    private boolean m_isCreate = false ;
    
    
    /**
     * データ格納オブジェクト.
     */
    protected final ArrayTable m_table = new ArrayTable() ;
    
    /**
     * 同期オブジェクト.
     */
    protected final Object m_sync = new Object() ;
    
    
    /**
     * コンストラクタ.
     */
    public DataMonitorBase()
    {
        
    }
    
    
    /**
     * 情報生成.
     * <BR><BR>
     * 監視対象の情報を生成します.
     * <BR>
     * @param timer 監視タイムアウト値を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void create( int timer )
        throws InputException,AccessException
    {
        LoopThread lt = null ;
        
        if( timer <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.clear() ;
        
        synchronized( m_sync ){
            
            try{
                
                lt = new LoopThread( this ) ;
                m_timer = timer ;
                lt.startThread() ;
                m_thread = lt ;
                m_isCreate = true ;
                
            }catch( InputException in ){
                throw in ;
            }catch( SettingException st ){
                throw new AccessException( st ) ;
            }finally{
                lt = null ;
            }
            
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 監視対象の情報をクリアします.
     */
    public final void clear()
    {
        
        try{
            m_thread.clear() ;
        }catch( Exception t1 ){
        }
        
        try{
            m_table.clear() ;
        }catch( Exception t2 ){
        }
        
        m_thread = null ;
        
        synchronized( m_sync ){
            m_isCreate = false ;
        }
        
    }
    
    /**
     * 監視対象情報を追加.
     * <BR><BR>
     * 監視対象の情報を追加します.
     * <BR>
     * @param key 監視対象キー情報を設定します.
     * @param obj 監視対象の情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public final void add( String key,Object obj )
        throws InputException,ExecutionException
    {
        DataMonitorPause pause = null ;
        
        synchronized( m_sync ){
            if( m_isCreate == false ){
                throw new ExecutionException( "情報追加処理を行うことは不可状態です" ) ;
            }
        }
        
        if( key == null || key.length() == 0 || obj == null ){
            throw new InputException( "入力情報が不正です" ) ;
        }
        
        pause = new DataMonitorPause() ;
        pause.MONITER_DATA = obj ;
        pause.SET_TIME = System.currentTimeMillis() ;
        
        synchronized( m_sync ){
            
            try{
                m_table.add( key,pause ) ;
            }catch( InputException in ){
                throw in ;
            }finally{
                pause = null ;
            }
            
        }
    }
    
    /**
     * 監視対象情報を追加.
     * <BR><BR>
     * 監視対象の情報を追加します.
     * <BR>
     * @param key 監視対象キー情報を設定します.
     * @param obj 監視対象の情報を設定します.
     * @param timer 監視対象のタイマー情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public final void add( String key,Object obj,long timer )
        throws InputException,ExecutionException
    {
        DataMonitorPause pause = null ;
        
        synchronized( m_sync ){
            if( m_isCreate == false ){
                throw new ExecutionException( "情報追加処理を行うことは不可状態です" ) ;
            }
        }
        
        if( key == null || key.length() == 0 || obj == null ){
            throw new InputException( "入力情報が不正です" ) ;
        }
        
        pause = new DataMonitorPause() ;
        pause.MONITER_DATA = obj ;
        pause.SET_TIME = timer ;
        
        synchronized( m_sync ){
            
            try{
                m_table.add( key,pause ) ;
            }catch( InputException in ){
                throw in ;
            }finally{
                pause = null ;
            }
            
        }
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 格納されている監視対象情報をクリアします.
     * <BR>
     * @param key 対象のキーに対する情報全てを削除します.
     * @exception InputException 入力例外.
     */
    public final void remove( String key ) throws InputException
    {
        
        int i ;
        int len ;
        ArrayTable at = null ;
        
        synchronized( m_sync ){
            
            try{
                
                at = m_table ;
                len = at.getElements( key ) ;
                
                if( len != 0 ){
                    for( i = 0 ; i < len ; i ++ ){
                        ( ( DataMonitorPause )at.get( key,i ) ).SET_TIME =  
                        DataMonitorBase.NOT_SET_TIMER ;
                    }
                    
                    at.remove( key ) ;
                }
                
            }catch( InputException in ){
                throw in ;
            }finally{
                at = null ;
            }
        }
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 格納されている監視対象情報のうち、
     * 指定項番情報の内容を削除します.
     * <BR>
     * @param key 削除対象のキー情報を設定します.
     * @param no 削除対象の項番を設定します.
     * @exception InputException 入力例外.
     */
    public final void remove( String key,int no ) throws InputException
    {
        
        synchronized( m_sync ){
            try{
                ( ( DataMonitorPause )m_table.get( key,no ) ).SET_TIME = DataMonitorBase.NOT_SET_TIMER ;
                m_table.remove( key,no ) ;
            }catch( InputException in ){
                throw in ;
            }
        }
        
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 格納されている監視対象情報を取得します.
     * <BR>
     * @param key 取得対象のキー情報を設定します.
     * @return Object[] 取得された内容が返されます.
     * @exception InputException 入力例外.
     */
    public final Object[] get( String key ) throws InputException
    {
        int i ;
        int len ;
        
        ArrayTable at = null ;
        Object[] ret = null ;
        
        synchronized( m_sync ){
            try{
                
                at = m_table ;
                len = at.getElements( key ) ;
                
                if( len == 0 ){
                    return null ;
                }
                
                ret = new Object[ len ] ;
                
                for( i = 0 ; i < len ; i ++ ){
                    ret[ i ] = ( ( DataMonitorPause )at.get( key,i ) ).MONITER_DATA ;
                }
                
                return ret ;
                
            }catch( InputException in ){
                throw in ;
            }finally{
                at = null ;
            }
        }
        
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 格納されている監視対象情報を取得します.
     * <BR>
     * @param key 取得対象のキー情報を設定します.
     * @param no 取得対象の項番を設定します.
     * @return Object 取得された内容が返されます.
     * @exception InputException 入力例外.
     */
    public final Object get( String key,int no ) throws InputException
    {
        Object ret = null ;
        
        synchronized( m_sync ){
            try{
                
                ret = ( ( DataMonitorPause )m_table.get( key,no ) ).MONITER_DATA ;
                return ret ;
                
            }catch( InputException in ){
                throw in ;
            }
        }
        
    }
    
    /**
     * 監視対象タイマー値を設定.
     * <BR><BR>
     * 対象の監視タイマー値を設定します.
     * <BR>
     * @param key 設定対象のキー名を設定します.
     * @param no 設定対象の項番を設定します.
     * @param timer 設定対象のタイマー情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void setPauseToNowTimer( String key,int no,long timer )
        throws InputException
    {
        
        if( key == null || no < 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        synchronized( m_sync ){
            try{
                
                ( ( DataMonitorPause )m_table.get( key,no ) ).SET_TIME = timer ;
                
            }catch( InputException in ){
                throw in ;
            }
        }
    }
    
    /**
     * 監視対象情報タイマー値の取得.
     * <BR><BR>
     * 監視対象情報タイマー値を取得します.
     * <BR>
     * @param key 取得対象のキー名を設定します.
     * @param no 取得対象の項番を設定します.
     * @return long 格納タイマー値が返されます.
     * @exception InputException 入力例外.
     */
    public final long getPauseToNowTimer( String key,int no ) throws InputException
    {
        long ret ;
        
        synchronized( m_sync ){
            try{
                
                ret = ( ( DataMonitorPause )m_table.get( key,no ) ).SET_TIME ;
                return ret ;
                
            }catch( InputException in ){
                throw in ;
            }
        }
        
    }
    
    /**
     * 指定キー名に対する情報長を取得.
     * <BR><BR>
     * 指定キー名に対する情報長を取得します.
     * <BR>
     * @param key 取得対象のキー情報を設定します.
     * @return int 指定キー情報に対する情報長が返されます.
     *             情報が存在しない場合[-1]が返されます.
     */
    public final int getElements( String key )
    {
        int ret ;
        
        synchronized( m_sync ){
            try{
                
                ret = m_table.getElements( key ) ;
                
            }catch( Exception t ){
                
                ret = -1 ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 格納キー情報長を取得.
     * <BR><BR>
     * 格納されているキー情報長を取得します.
     * <BR>
     * @return int 格納されているキー情報長が返されます.
     *             情報が存在しない場合[-1]が返されます.
     */
    public final int getLength()
    {
        int ret ;
        
        synchronized( m_sync ){
            try{
                
                ret = m_table.getKeySize() ;
                
            }catch( Exception t ){
                
                ret = -1 ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 格納データ長を取得.
     * <BR><BR>
     * 格納されているデータ長を取得します.
     * <BR>
     * @return int 格納されているデータ長を全て取得します.
     *             情報が存在しない場合[-1]が返されます.
     */
    public final int getAllSize()
    {
        
        int ret ;
        
        synchronized( m_sync ){
            try{
                
                ret = m_table.size() ;
                
            }catch( Exception t ){
                
                ret = -1 ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 格納キー情報群を取得.
     * <BR><BR>
     * 格納されているキー名群を取得します.
     * <BR>
     * @return String[] 格納されているキー名群を取得します.
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getKeys()
    {
        String[] ret = null ;
        
        synchronized( m_sync ){
            try{
                
                ret = m_table.getNames() ;
                
            }catch( Exception t ){
                
                ret = null ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 監視タイマー値の取得.
     * <BR><BR>
     * 設定対象の監視タイマー値が返されます.
     * <BR>
     * @return int 監視タイマー値が返されます.
     */
    public final int getTimer()
    {
        int ret ;
        
        synchronized( m_sync ){
            
            ret = m_timer ;
            
        }
        
        return ret ;
    }
    
    /**
     * 情報生成チェック.
     * <BR><BR>
     * 情報が生成されているかチェックします.
     * <BR>
     * @return boolean 情報生成チェックが返されます.<BR>
     *                 [true]が返された場合、情報は生成されています.<BR>
     *                 [false]が返された場合、情報は生成されていません.
     */
    public final boolean isCreate()
    {
        boolean ret ;
        
        synchronized( m_sync ){
            
            ret = m_isCreate ;
            
        }
        
        return ret ;
    }
    
    /**
     * モニター監視.
     * <BR><BR>
     * モニター監視処理を実施します.
     * <BR>
     * @exception ExecutionException 実行例外.
     */
    public abstract void moniter() throws ExecutionException ;
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init(Object obj) throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit(Object obj) throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop(Object obj) throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution(Object obj) throws ExecutionException
    {
        try{
            
            UtilCom.idleTime() ;
            UtilCom.sleep( DataMonitorBase.SLEEP_TIMING ) ;
            
            this.moniter() ;
            
        }catch( ExecutionException ex ){
            throw ex ;
        }
    }
    
}
