/*
 * @(#)DataMonitor.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;

/**
 * DataMonitorインターフェイス.
 * <BR><BR>
 * データ監視モニターのインターフェイスを定義します.
 *
 * @version     1.00, 2003/12/04
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public interface DataMonitor
{
    
    /**
     * 情報生成.
     * <BR><BR>
     * 監視対象の情報を生成します.
     * <BR>
     * @param timer 監視タイムアウト値を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void create( int timer )
        throws InputException,AccessException ;
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 監視対象の情報をクリアします.
     */
    public void clear() ;
    
    /**
     * 監視対象情報を追加.
     * <BR><BR>
     * 監視対象の情報を追加します.
     * <BR>
     * @param key 監視対象キー情報を設定します.
     * @param obj 監視対象の情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public void add( String key,Object obj )
        throws InputException,ExecutionException ;
    
    /**
     * 監視対象情報を追加.
     * <BR><BR>
     * 監視対象の情報を追加します.
     * <BR>
     * @param key 監視対象キー情報を設定します.
     * @param obj 監視対象の情報を設定します.
     * @param timer 監視対象のタイマー情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public void add( String key,Object obj,long timer )
        throws InputException,ExecutionException ;
    
    /**
     * 情報削除.
     * <BR><BR>
     * 格納されている監視対象情報をクリアします.
     * <BR>
     * @param key 対象のキーに対する情報全てを削除します.
     * @exception InputException 入力例外.
     */
    public void remove( String key ) throws InputException ;
    
    /**
     * 情報削除.
     * <BR><BR>
     * 格納されている監視対象情報のうち、
     * 指定項番情報の内容を削除します.
     * <BR>
     * @param key 削除対象のキー情報を設定します.
     * @param no 削除対象の項番を設定します.
     * @exception InputException 入力例外.
     */
    public void remove( String key,int no ) throws InputException ;
    
    /**
     * 情報取得.
     * <BR><BR>
     * 格納されている監視対象情報を取得します.
     * <BR>
     * @param key 取得対象のキー情報を設定します.
     * @return Object[] 取得された内容が返されます.
     * @exception InputException 入力例外.
     */
    public Object[] get( String key ) throws InputException ;
    
    /**
     * 情報取得.
     * <BR><BR>
     * 格納されている監視対象情報を取得します.
     * <BR>
     * @param key 取得対象のキー情報を設定します.
     * @param no 取得対象の項番を設定します.
     * @return Object 取得された内容が返されます.
     * @exception InputException 入力例外.
     */
    public Object get( String key,int no ) throws InputException ;
    /**
     * 指定キー名に対する情報長を取得.
     * <BR><BR>
     * 指定キー名に対する情報長を取得します.
     * <BR>
     * @param key 取得対象のキー情報を設定します.
     * @return int 指定キー情報に対する情報長が返されます.
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getElements( String key ) ;
    
    /**
     * 格納キー情報長を取得.
     * <BR><BR>
     * 格納されているキー情報長を取得します.
     * <BR>
     * @return int 格納されているキー情報長が返されます.
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getLength() ;
    
    /**
     * 格納データ長を取得.
     * <BR><BR>
     * 格納されているデータ長を取得します.
     * <BR>
     * @return int 格納されているデータ長を全て取得します.
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getAllSize() ;
    
    /**
     * 格納キー情報群を取得.
     * <BR><BR>
     * 格納されているキー名群を取得します.
     * <BR>
     * @return String[] 格納されているキー名群を取得します.
     *                  情報が存在しない場合[null]が返されます.
     */
    public String[] getKeys() ;
    
    /**
     * 監視タイマー値の取得.
     * <BR><BR>
     * 設定対象の監視タイマー値が返されます.
     * <BR>
     * @return int 監視タイマー値が返されます.
     */
    public int getTimer() ;
    
    /**
     * 情報生成チェック.
     * <BR><BR>
     * 情報が生成されているかチェックします.
     * <BR>
     * @return boolean 情報生成チェックが返されます.<BR>
     *                 [true]が返された場合、情報は生成されています.<BR>
     *                 [false]が返された場合、情報は生成されていません.
     */
    public boolean isCreate() ;
    
}
