/*
 * @(#)CharUtil.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import java.io.Serializable;
import java.util.ArrayList;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * 文字コード情報結合管理クラス。
 * <BR><BR>
 * 文字コード情報の結合処理を管理します。<BR>
 * また、取得の場合は、追加管理領域を結合して、１つの文字列情報として渡されます.<BR>
 * また、このオブジェクトは、同期処理に対応していません.
 *
 * <BR>
 *
 * @version 1.00, 2003/10/27
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class CharUtil implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            CharTable.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 情報管理.
     */
    private final ArrayList m_info = new ArrayList() ;
    
    /**
     * 情報サイズ管理.
     */
    private int m_length = 0 ;
    
    /**
     * コンストラクタ.
     */
    public CharUtil()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報のクリアー.
     * <BR><BR>
     * 生成されている情報をクリアーします。
     */
    public final void clear()
    {
        m_info.clear() ;
        m_length = 0 ;
    }
    
    /**
     * 文字列管理情報にバイト情報をセットします。
     * <BR><BR>
     * 文字列管理情報にバイト情報をセットします。
     * <BR>
     * @param pause 設定する文字コード情報
     */
    public final void add(char pause)
    {
        char[] info = null ;
        info = new char[1] ;
        info[0] = pause ;
        m_info.add(info) ;
        m_length += 1 ;
        info = null ;
    }
    
    /**
     * 文字列管理情報に文字列情報をセットします。
     * <BR><BR>
     * 文字列管理情報に文字列情報をセットします。
     * <BR>
     * @param info 設定する文字列情報
     * @exception InputException 入力例外
     */
    public final void add(char[] info)
        throws InputException
    {
        try{
            this.add(info,0,-1) ;
        }
        catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * 文字列管理情報に文字列情報をセットします。
     * <BR><BR>
     * 文字列管理情報に文字列情報をセットします。
     * <BR>
     * @param info 設定する文字列情報
     * @param size 設定サイズ
     * @exception InputException 入力例外
     */
    public final void add(char[] info,int size)
        throws InputException
    {
        try{
            this.add(info,0,size) ;
        }
        catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * 文字列管理情報に文字列情報をセットします。
     * <BR><BR>
     * 文字列管理情報に文字列情報をセットします。
     * <BR>
     * @param info 設定する文字列情報
     * @param off オフセット値
     * @param size 設定する文字列サイズ
     * @exception InputException 入力例外
     */
    public final void add(char[] info,int off,int size)
        throws InputException
    {
        char[] tmp = null ;
        
        try{
            if(size == -1){
                size = info.length ;
            }
            tmp = new char[size] ;
            System.arraycopy(info,off,tmp,0,size) ;
            m_info.add(tmp) ;
            m_length += size ;
        }catch(IndexOutOfBoundsException io){
            throw new InputException(io) ;
        }catch(ArrayStoreException as){
            throw new InputException(as) ;
        }catch(NullPointerException np){
            throw new InputException(np) ;
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * 文字列情報の取得を行います。
     * <BR><BR>
     * 文字列情報を全て取得します。
     * <BR>
     * @return char[] 管理された全ての文字列情報を取得します。
     */
    public final char[] get()
    {
        try{
            return this.get(0,-1) ;
        }catch(InputException in){
            return null ;
        }
    }
    
    /**
     * 文字列情報の取得を行います。
     * <BR><BR>
     * 文字列情報を全て取得します。
     * <BR>
     * @param size 取得サイズ
     * @return char[] 管理された全ての文字列情報を取得します。
     * @exception InputException 入力例外
     */
    public final char[] get(int size)
        throws InputException
    {
        try{
            return this.get(0,size) ;
        }catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * 文字列情報の取得を行います。
     * <BR><BR>
     * 文字列情報を全て取得します。
     * <BR>
     * @param off オフセット値
     * @param size 取得サイズ
     * @return char[] 管理された全ての文字列情報を取得します。<BR>
     *                情報が設定されていない場合、[null]が返されます.
     */
    public final char[] get(int off,int size)
        throws InputException
    {
        int i ;
        int len,len2 ;
        int cur = 0 ;
        int allSize = 0 ;
        char[] info = null ;
        char[] tmp = null ;
        
        ArrayList array = null ;
        
        if(size != -1 && (size < 0 || size > allSize || off < 0 || off + size > allSize)){
            throw new InputException( "指定情報は範囲外です" ) ;
        }else if((allSize = this.size()) == 0){
            return null ;
        }
        
        array = m_info ;
        len = array.size() ;
        info = new char[allSize] ;
        
        for(i = 0 ; i < len ; i ++){
            tmp = (char[])array.get(i) ;
            len2 = tmp.length ;
            System.arraycopy(tmp,0,info,cur,len2) ;
            cur += len2 ;
        }
        
        tmp = null ;
        
        if(size == -1){
            return info ;
        }
        
        tmp = new char[size] ;
        System.arraycopy(info,off,tmp,0,size) ;
        info = null ;
        return tmp ;
        
    }
    
    /**
     * 管理された文字列総情報数を取得します。
     * <BR><BR>
     * 管理された文字列総情報数を取得します。
     * <BR>
     * @return int 管理された文字列総情報数
     */
    public final int size()
    {
        return m_length ;
    }
}
