/*
 * @(#)ByteUtil.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.util;

import java.io.Serializable;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * バイナリ情報結合管理クラス。
 * <BR><BR>
 * バイナリ情報の結合処理を管理します。<BR>
 * また、取得の場合は、追加管理領域を結合して、１つのバイナリとして渡されます.<BR>
 * また、このオブジェクトは、同期処理に対応していません.
 *
 * @version     1.00, 2000/11/29
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ByteUtil implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ByteUtil.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 情報管理.
     */
    private final ObjectArray m_info = new ObjectArray() ;
    
    /**
     * 情報サイズ管理.
     */
    private int m_length = 0 ;
    
    /**
     * コンストラクタ.
     */
    public ByteUtil()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch(  Exception t  ){
        }
        
    }
    
    /**
     * 情報のクリアー.
     * <BR><BR>
     * 生成されている情報をクリアーします。
     */
    public final void clear()
    {
        m_info.clear() ;
        m_length = 0 ;
    }
    
    /**
     * 指定バイナリ情報を直接追加.
     * <BR><BR>
     * 指定バイナリ情報を直接追加します.<BR>
     * [ByteUtil.add()]では、追加対象バイナリを新たに生成して
     * 追加します。しかし、このメソッドでは指定バイナリを新しい
     * 領域を生成することなく、対象バイナリをそのまま追加します.<BR>
     * そのため、このメソッドで追加したバイナリ内を変更する状態で
     * 利用する事はできません(格納内容が変わります).
     * <BR>
     * @param pause 設定するバイナリ情報.
     */
    public final void put( byte[] pause )
        throws InputException
    {
        if( pause == null || pause.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_info.add( pause ) ;
        m_length += pause.length ;
    }
    
    /**
     * バイナリ情報を追加します.
     * <BR><BR>
     * バイナリ管理情報にバイト情報をセットします.
     * <BR>
     * @param pause 設定するバイト情報
     */
    public final void add( byte pause )
    {
        byte[] tmp = null ;
        
        tmp = new byte[ 1 ] ;
        tmp[ 0 ] = pause ;
        
        try{
            this.add( tmp,0,-1 ) ;
        }catch( InputException in ){
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * バイナリ情報を追加します.
     * <BR><BR>
     * バイナリ管理情報にバイナリ情報をセットします.
     * <BR>
     * @param info 設定するバイナリ情報
     * @exception InputException 入力例外
     */
    public final void add( byte[] info )
        throws InputException
    {
        try{
            this.add( info,0,-1 ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * バイナリ情報を追加します.
     * <BR><BR>
     * バイナリ管理情報にバイナリ情報をセットします.
     * <BR>
     * @param info 設定するバイナリ情報
     * @param size 設定サイズ
     * @exception InputException 入力例外
     */
    public final void add( byte[] info,int size )
        throws InputException
    {
        try{
            this.add( info,0,size ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * バイナリ情報を追加します.
     * <BR><BR>
     * バイナリ管理情報にバイナリ情報をセットします.
     * <BR>
     * @param info 設定するバイナリ情報
     * @param off オフセット値
     * @param size 設定するバイナリサイズ
     * @exception InputException 入力例外
     */
    public final void add( byte[] info,int off,int size )
        throws InputException
    {
        byte[] tmp = null ;
        
        try{
            
            if( size == -1 ){
                
                m_length += info.length ;
                m_info.add( info ) ;
                
            }else{
                
                tmp = new byte[ size ] ;
                System.arraycopy( info,off,tmp,0,size ) ;
                m_info.add( tmp ) ;
                m_length += size ;
                
            }
            
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }catch( ArrayStoreException as ){
            throw new InputException( as ) ;
        }catch( NullPointerException np ){
            throw new InputException( np ) ;
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * 指定条件の間に追加.
     * <BR><BR>
     * 指定条件の間にバイナリ情報を追加します.
     * <BR>
     * @param no 追加する間の位置を設定します.
     * @param pause 設定するバイト情報.
     */
    public final void between( int no,byte pause )
    {
        byte[] tmp = null ;
        
        tmp = new byte[ 1 ] ;
        tmp[ 0 ] = pause ;
        
        try{
            this.between( no,tmp,0,-1 ) ;
        }catch( InputException in ){
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * 指定条件の間に追加.
     * <BR><BR>
     * 指定条件の間にバイナリ情報を追加します.
     * <BR>
     * @param no 追加する間の位置を設定します.
     * @param info 設定するバイナリ情報
     * @exception InputException 入力例外
     */
    public final void between( int no,byte[] info )
        throws InputException
    {
        try{
            this.between( no,info,0,-1 ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 指定条件の間に追加.
     * <BR><BR>
     * 指定条件の間にバイナリ情報を追加します.
     * <BR>
     * @param no 追加する間の位置を設定します.
     * @param info 設定するバイナリ情報
     * @param size 設定サイズ
     * @exception InputException 入力例外
     */
    public final void between( int no,byte[] info,int size )
        throws InputException
    {
        try{
            this.between( no,info,0,size ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 指定条件の間に追加.
     * <BR><BR>
     * 指定条件の間にバイナリ情報を追加します.
     * <BR>
     * @param no 追加する間の位置を設定します.
     * @param info 設定するバイナリ情報
     * @param off オフセット値
     * @param size 設定するバイナリサイズ
     * @exception InputException 入力例外
     */
    public final void between( int no,byte[] info,int off,int size )
        throws InputException
    {
        int len ;
        byte[] tmp = null ;
        Object[] objs = null ;
        
        try{
            
            len = m_info.size() - 1 ;
            no = ( no < 0 || no > len ) ? ( ( no < 0 ) ? 0 : len+1 ) : no ;
            
            len ++ ;
            if( no == len ){
                this.add( info,off,size ) ;
            }
            else{
                
                if( size == -1 ){
                    
                    tmp = info ;
                    m_length += info.length ;
                    
                }else{
                    
                    tmp = new byte[ size ] ;
                    System.arraycopy( info,off,tmp,0,size ) ;
                    m_length += size ;
                    
                }
                
                m_info.add( new Object() ) ;
                objs = m_info.getObjects() ;
                
                if( no == 0 ){
                    System.arraycopy( objs,0,objs,1,len ) ;
                    objs[ 0 ] = tmp ;
                }
                else{
                    System.arraycopy( objs,no,objs,no+1,len - no ) ;
                    objs[ no ] = tmp ;
                }
                
            }
            
        }catch( IndexOutOfBoundsException io ){
            throw new InputException( io ) ;
        }catch( ArrayStoreException as ){
            throw new InputException( as ) ;
        }catch( NullPointerException np ){
            throw new InputException( np ) ;
        }finally{
            tmp = null ;
            objs = null ;
        }
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * バイナリ情報を全て取得します.
     * <BR>
     * @return byte[] 管理された全てのバイナリ情報を取得します。
     */
    public final byte[] get()
    {
        try{
            return this.get( 0,-1 ) ;
        }catch( InputException in ){
            return null ;
        }
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * バイナリ情報を全て取得します.
     * <BR>
     * @param size 取得サイズ
     * @return byte[] 管理された全てのバイナリ情報を取得します。
     * @exception InputException 入力例外
     */
    public final byte[] get( int size )
        throws InputException
    {
        try{
            return this.get( 0,size ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * バイナリ情報を全て取得します.
     * <BR>
     * @param off オフセット値
     * @param size 取得サイズ
     * @return byte[] 管理された全てのバイナリ情報を取得します。<BR>
     *                情報が設定されていない場合、[ null ]が返されます.
     */
    public final byte[] get( int off,int size )
        throws InputException
    {
        int i ;
        int len,len2 ;
        int cur = 0 ;
        int allSize = 0 ;
        
        byte[] ret = null ;
        byte[] tmp = null ;
        
        ObjectArray array = null ;
        allSize = m_length ;
        
        if(
            size != -1 &&
            (
                size < 0 || size > allSize ||
                off < 0 || off + size > allSize
            )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }else if( allSize == 0 ){
            return null ;
        }
        
        array = m_info ;
        len = array.size() ;
        ret = new byte[ allSize ] ;
        
        for( i = 0 ; i < len ; i ++ ){
            
            tmp = ( byte[] )array.get( i ) ;
            len2 = tmp.length ;
            System.arraycopy( tmp,0,ret,cur,len2 ) ;
            cur += len2 ;
            
            tmp = null ;
            
        }
        
        tmp = null ;
        
        if( size != -1 ){
            
            tmp = new byte[ size ] ;
            System.arraycopy( ret,off,tmp,0,size ) ;
            ret = tmp ;
            tmp = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * バイナリ情報を全て取得します.<BR>
     * また、この処理を実施した場合、格納されているバイナリ情報は破棄されます.
     * <BR>
     * @return byte[] 管理された全てのバイナリ情報を取得します。
     */
    public final byte[] getToClear()
    {
        try{
            return this.getToClear( 0,-1 ) ;
        }catch( InputException in ){
            return null ;
        }
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * バイナリ情報を全て取得します.<BR>
     * また、この処理を実施した場合、格納されているバイナリ情報は破棄されます.
     * <BR>
     * @param size 取得サイズ
     * @return byte[] 管理された全てのバイナリ情報を取得します。
     * @exception InputException 入力例外
     */
    public final byte[] getToClear( int size )
        throws InputException
    {
        try{
            return this.getToClear( 0,size ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * バイナリ情報の取得.
     * <BR><BR>
     * バイナリ情報を全て取得します.<BR>
     * また、この処理を実施した場合、格納されているバイナリ情報は破棄されます.
     * <BR>
     * @param off オフセット値
     * @param size 取得サイズ
     * @return byte[] 管理された全てのバイナリ情報を取得します。<BR>
     *                情報が設定されていない場合、[ null ]が返されます.
     */
    public final byte[] getToClear( int off,int size )
        throws InputException
    {
        int i ;
        int len,len2 ;
        int cur = 0 ;
        int allSize = 0 ;
        
        byte[] tmp = null ;
        Object[] objs = null ;
        byte[] ret = null ;
        
        ObjectArray array = null ;
        allSize = m_length ;
        
        if(
            size != -1 &&
            (
                size < 0 || size > allSize ||
                off < 0 || off + size > allSize
            )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }else if( allSize == 0 ){
            return null ;
        }
        
        array = m_info ;
        len = array.size() ;
        objs = array.getObjects() ;
        ret = new byte[ allSize ] ;
        
        for( i = 0 ; i < len ; i ++ ){
            
            tmp = ( byte[] )objs[ i ] ;
            objs[ i ] = null ;
            
            len2 = tmp.length ;
            System.arraycopy( tmp,0,ret,cur,len2 ) ;
            tmp = null ;
            cur += len2 ;
            
        }
        
        objs = null ;
        tmp = null ;
        
        this.clear() ;
        
        if( size != -1 ){
            
            tmp = new byte[ size ] ;
            System.arraycopy( ret,off,tmp,0,size ) ;
            ret = tmp ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 管理されたバイナリ総情報数を取得.
     * <BR><BR>
     * 管理されたバイナリ総情報数を取得します。
     * <BR>
     * @return int 管理されたバイナリ総情報数
     */
    public final int size()
    {
        return m_length ;
    }
    
}
