/*
 * @(#)ThreadManager.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.thread;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.IdManager;
import com.JRcServer.commons.util.UtilCom;

/**
 * スレッド管理オブジェクト.
 * <BR><BR>
 * スレッドを管理するためのオブジェクトです.
 *
 * @version     1.00, 2004/01/31
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ThreadManager
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ThreadManager.class ) ;
    
    /**
     * スレッドID : 無効値.
     */
    protected static final int NOT_ID = -1 ;
    
    /**
     * スレッドID : 設定スレッド値未定.
     */
    protected static final int NOSET_ID = Integer.MIN_VALUE ;
    
    /**
     * MAX-ID.
     */
    private static final int MAX_ID = 999999999 ;
    
    
    /**
     * シングルトン.
     */
    private static final ThreadManager SNGL = new ThreadManager() ;
    
    /**
     * ローカル管理オブジェクト.
     */
    private LocalManager m_local = null ;
    
    /**
     * スレッド管理テーブル.
     */
    private final CharTable m_table = new CharTable() ;
    
    /**
     * スレッドマネージャ監視モニター.
     */
    private ThreadMonitor m_monitor = null ;
    
    /**
     * ID発番管理オブジェクト.
     */
    private final IdManager m_id = new IdManager( 0,ThreadManager.MAX_ID ) ;
    
    /**
     * スレッド生成待ち状態数.
     */
    private volatile int m_waitcount = 0 ;
    
    /**
     * 生成フラグ.
     */
    private volatile boolean m_createFlag = false ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 生成処理用同期.
     */
    private final Object m_createSync = new Object() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private ThreadManager()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroyManager() ;
        }catch( Exception t ){
            throw t ;
        }
        
    }
    
    /**
     * シングルトンオブジェクトの取得.
     * <BR><BR>
     * シングルトンのオブジェクトを取得します.
     * <BR>
     * @return ThreadManager 対象のオブジェクトが返されます.
     */
    public static final ThreadManager getInstance()
    {
        ThreadManager ret = null ;
        
        synchronized( SNGL ){
            ret = SNGL ;
        }
        
        return ret ;
    }
    
    /**
     * ThreadManager初期処理.
     * <BR><BR>
     * ThreadManagerの初期化処理です.
     */
    public synchronized final void initManager()
    {
        this.destroyManager() ;
        
        if( m_createFlag == false ){
            try{
                
                if( m_sync.isUse() == false ){
                    m_sync.create() ;
                }
                
                m_monitor = new ThreadMonitor( m_table,m_sync ) ;
                
                // ローカルマネージャ生成.
                m_local = new LocalManager( m_table,m_sync ) ;
                
                // IDマネージャ生成.
                //m_id.create( 0,ThreadManager.MAX_ID ) ;
                m_createFlag = true ;
                
                LOG.info( "## スレッドマネージャが生成されました:" + this ) ;
                
            }catch( Exception t ){
                LOG.error( "## スレッドマネージャ生成時にエラーが発生しました",t ) ;
                this.destroyManager() ;
            }
        }
        
    }
    
    /**
     * ThreadManagerを破棄.
     * <BR><BR>
     * ThreadManagerを破棄します.
     */
    public synchronized final void destroyManager()
    {
        boolean bef = false ;
        
        try{
            
            try{
                m_local.closeManager() ;
            }catch( Exception t ){
            }
            
            try{
                m_monitor.clear() ;
            }catch( Exception t ){
            }
            
            bef = m_createFlag ;
            m_sync.clear() ;
            m_table.clear() ;
            //m_id.clear() ;
            m_monitor = null ;
            m_local = null ;
            m_createFlag = false ;
            
        }catch( Exception tt ){
        }
        
        if( bef == true ){
            LOG.info( "## スレッドマネージャが破棄されました" ) ;
        }
    }
    
    /**
     * スレッド破棄時呼び出し.
     * <BR><BR>
     * スレッドを破棄する際に呼び出します.
     * <BR>
     * @param thread 対象のスレッドオブジェクトを設定します.
     */
    public final void clear( Thread thread )
    {
        
        if( thread == null ){
            return ;
        }
        
        this.clear( thread.getName() ) ;
        
    }
    
    /**
     * スレッド破棄時呼び出し.
     * <BR><BR>
     * スレッドを破棄する際に呼び出します.
     * <BR>
     * @param name 対象のスレッド名を設定します.
     */
    public final void clear( String name )
    {
        
        if( name == null ){
            return ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                ( ( ThreadState )m_table.get( name ) ).clear() ;
            }
            
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 対象のスレッド終了状態の設定.
     * <BR><BR>
     * 対象のスレッド終了状態を設定します.
     * <BR>
     * @param thread 対象のスレッドオブジェクトを設定します.
     */
    public final void exitThread( Thread thread )
    {
        
        if( thread == null ){
            return ;
        }
        
        this.exitThread( thread.getName() ) ;
        
    }
    
    /**
     * 対象のスレッド終了状態の設定.
     * <BR><BR>
     * 対象のスレッド終了状態を設定します.
     * <BR>
     * @param name 対象のスレッド名を設定します.
     */
    public final void exitThread( String name )
    {
        
        if( name == null ){
            return ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                ( ( ThreadState )m_table.get( name ) ).endThread() ;
            }
            
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 対象のスレッド終了状態の取得.
     * <BR><BR>
     * 対象のスレッド終了状態を取得します.
     * <BR>
     * @param thread 取得対象のスレッドオブジェクトを設定します.
     * @return boolean 終了状態を表すフラグ情報が返されます.<BR>
     *                 [true]が返された場合、終了状態です.
     *                 [false]が返された場合、実施状態です.
     */
    public final boolean getExitThread( Thread thread )
    {
        
        if( thread == null ){
            return true ;
        }
        
        return this.getExitThread( thread.getName() ) ;
    }
    
    /**
     * 対象のスレッド終了状態の取得.
     * <BR><BR>
     * 対象のスレッド終了状態を取得します.
     * <BR>
     * @param name 取得対象のスレッド名を設定します.
     * @return boolean 終了状態を表すフラグ情報が返されます.<BR>
     *                 [true]が返された場合、終了状態です.
     *                 [false]が返された場合、実施状態です.
     */
    public final boolean getExitThread( String name )
    {
        
        boolean ret ;
        
        if( name == null ){
            return true ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( ( ThreadState )m_table.get( name ) ).isEndThread() ;
            }
            
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象スレッド要素オブジェクトの取得.
     * <BR><BR>
     * 対象スレッド要素オブジェクトを取得します.
     * <BR>
     * @param thread 取得対象のスレッドオブジェクトを設定します.
     * @return ThreadState 対象スレッド要素オブジェクトが返されます.
     */
    public final ThreadState getState( Thread thread )
    {
        
        if( thread == null ){
            return null ;
        }
        
        return this.getState( thread.getName() ) ;
    }
    
    /**
     * 対象スレッド要素オブジェクトの取得.
     * <BR><BR>
     * 対象スレッド要素オブジェクトを取得します.
     * <BR>
     * @param name 対象のスレッド名を設定します.
     * @return ThreadState 対象スレッド要素オブジェクトが返されます.
     */
    public final ThreadState getState( String name )
    {
        
        ThreadState ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( ThreadState )m_table.get( name ) ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ローカルスレッド要素オブジェクトの取得.
     * <BR><BR>
     * ローカルスレッド要素オブジェクトを取得します.
     * <BR>
     * @return ThreadState ローカルスレッド要素オブジェクトが返されます.
     */
    public final ThreadState getLocalState()
    {
        
        ThreadState ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_local.get() ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在動作しているスレッド数を取得.
     * <BR><BR>
     * 現在動作しているスレッド数を取得します.
     * <BR>
     * @return int 現在動作しているスレッド数が返されます.
     */
    public final int getCount()
    {
        return this.getThreadCount() ;
    }
    
    /**
     * 現在登録されているスレッド名群を取得.
     * <BR><BR>
     * 現在登録されているスレッド名群を取得します.
     * <BR>
     * @return String[] 現在登録されているスレッド名群が返されます.
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getNames()
    {
        String[] ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_table.getNames() ;
            }
            
        }catch( Exception t ){
        }
        
        return ret ;
        
    }
    
    /**
     * 現在登録されている数を取得.
     * <BR><BR>
     * 現在登録されている数を取得します.
     * <BR>
     * @return int 現在登録されている数を取得します.
     */
    public final int size()
    {
        int ret ;
        
        ret = 0 ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_table.size() ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在実施途中のスレッド数を取得.
     * <BR><BR>
     * 現在実施されているスレッドの個数を取得します.
     * <BR>
     * @return int 現在実施途中のスレッド数が返されます.
     */
    public final int getWaitThread()
    {
        int ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_waitcount ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド同期オブジェクトの取得.
     * <BR><BR>
     * スレッド同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized スレッド同期オブジェクトが返されます.
     */
    protected final Synchronized getSync()
    {
        return m_sync ;
    }
    
    /**
     * 対象スレッド名存在チェック.
     * <BR><BR>
     * 対象スレッド名が存在するかチェックします.
     * <BR>
     * @param name チェック対象の名前を設定します.
     * @return boolean チェック結果が返されます.
     */
    protected final boolean isName( String name )
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_table.isData( name ) ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド終了検知.
     * <BR><BR>
     * 対象のスレッドが終了されたか否かを検知します.
     * <BR>
     * @exception ExecutionException スレッド実施例外.
     */
    public static final void isExit() throws ExecutionException
    {
        try{
            ThreadManager.getInstance().isExitExecutionThread() ;
        }catch( ExecutionException ee ){
            throw ee ;
        }
        
    }
    
    /**
     * 利用可能ID情報を取得.
     * <BR><BR>
     * 利用可能ID情報を取得します.
     * <BR>
     * @return int 利用可能なID情報が返されます.<BR>
     *             利用可能IDが見つからない場合
     *             [ThreadManager#NOT_ID]が返されます.
     */
    protected static final int getID()
    {
        return ThreadManager.getInstance().m_id.getID() ;
    }
    
    /**
     * 利用終了IDの削除.
     * <BR><BR>
     * 利用終了IDを削除します.
     * <BR>
     * @param id 利用終了のIDを設定します.
     */
    protected static final void removeID( int id )
    {
        ThreadManager.getInstance().m_id.removeID( id ) ;
    }
    
    /**
     * 最終アクセス時間の更新.
     * <BR><BR>
     * 最終アクセス時間を更新します.
     * <BR>
     * @param name 最終アクセス時間を設定するスレッド名を設定します.
     */
    protected final void setLastAccess( String name )
    {
        Object pause = null ;
        
        if( name == null ){
            return  ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                if( ( pause = m_table.get( name ) ) != null ){
                    ( ( ThreadState )pause ).setLastAccessTime() ;
                }
            }
            
        }catch( Exception t ){
        }
        
        pause = null ;
        
    }
    
    /**
     * メソッド初期処理.
     * <BR><BR>
     * メソッドの初期処理を行います.
     */
    protected final void initMethod()
    {
        try{
            
            synchronized( m_sync.get() ){
                m_local.chkThread() ;
            }
            
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッドステータス登録.
     * <BR><BR>
     * スレッドステータスの登録処理を行います.
     * <BR>
     * @param thread 登録対象のスレッドオブジェクトを指定します.
     */
    protected final void initMethod( Thread thread )
    {
        ThreadState state = null ;
        
        if( thread == null ){
            return ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                state = new ThreadState( thread ) ;
                m_table.add( thread.getName(),state ) ;
                m_waitcount ++ ;
                
            }
            
        }catch( Exception t ){
            m_waitcount = 0 ;
        }finally{
            state = null ;
        }
        
    }
    
    /**
     * スレッドステータス登録失敗時の処理.
     * <BR><BR>
     * スレッドステータス登録失敗時の処理を行います.
     */
    protected final void initMethodByError()
    {
        try{
            synchronized( m_sync.get() ){
                m_waitcount -- ;
            }
        }catch( Exception t ){
            m_waitcount = 0 ;
        }
    }
    
    /**
     * スレッド処理実施時に実行.
     * <BR><BR>
     * スレッド処理実施時に実行します.
     * この処理は[ThreadManager#initMethod( Thread )]を実施した
     * メソッドが、スレッド生成時において、実行する必要があります.
     * <BR>
     * @param thread 対象のスレッドオブジェクトを設定します.
     */
    protected final void startMethod( Thread thread )
    {
        String name = null ;
        
        if( thread == null ){
            return ;
        }
        
        try{
            
            name = thread.getName() ;
            
            for( ;; ){
                
                synchronized( m_sync.get() ){
                    
                    if( m_table.get( name ) != null ){
                        
                        m_waitcount -- ;
                        break ;
                        
                    }
                    
                }
                
                UtilCom.idleTime() ;
                
            }
            
        }catch( Exception t ){
            m_waitcount = 0 ;
        }
    }
    
    
    
    /**
     * 終了チェック処理.
     * <BR><BR>
     * 終了処理に対するチェック処理を行います.
     * <BR>
     * @exception ExecutionException 実行例外.
     */
    private final void isExitExecutionThread()
        throws ExecutionException
    {
        ThreadState pause = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if(
                    ( pause = m_local.get() ) == null ||
                    pause.isEndThread() == true
                )
                {
                    
                    throw new ExecutionException(
                        "スレッド処理( name :" +
                        Thread.currentThread().getName() +
                        " )は動作していないか、終了しました",
                        ExecutionException.LEVEL_STOP
                        
                    ) ;
                    
                }
            }
            
        }catch( ExecutionException ex ){
            throw ex ;
        }catch( Exception t ){
            throw new ExecutionException( t,ExecutionException.LEVEL_STOP ) ;
        }finally{
            pause = null ;
        }
        
    }
    
    
    
    /**
     * スレッドカウント.
     */
    private volatile int m_threadCount = 0 ;
    
    /**
     * スレッドカウントインクリメント.
     */
    protected synchronized final void addThreadCount()
    {
        m_threadCount ++ ;
    }
    
    /**
     * スレッドカウントデクリメント.
     */
    protected synchronized final void removeThreadCount()
    {
        m_threadCount -- ;
    }
    
    /**
     * スレッドカウント取得.
     */
    protected synchronized final int getThreadCount()
    {
        return m_threadCount ;
    }
}

