/*
 * @(#)TObjectManager.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.thread;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.ExistException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.UtilCom;

/**
 * Object情報をスレッド単位で管理.
 * <BR><BR>
 * 格納されているObject情報をスレッド単位で管理します.<BR>
 * このオブジェクト内で管理されている内容は、同期処理を意識する必要は
 * ありません.
 *
 * @version 1.00, 2003/10/31
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class TObjectManager extends ExecutionThread
{
    
    /**
     * オブジェクト管理要素.
     */
    class OMPause{
        
        /**
         * タイマー格納.
         */
        public long timer = 0L ;
        
        /**
         * オブジェクト情報.
         */
        public Object info = null ;
        
        /**
         * ガページコレクタ処理.
         */
        protected final void finalize() throws Exception
        {
            info = null ;
            timer = 0L ;
        }
        
    }
    
    
    /**
     * デフォルトタイマー値.
     * 300秒 : 5分.
     */
    private static final long DEFAULT_TIMER = 300000L ;
    
    /**
     * 最小監視タイマー値.
     * 60秒.
     */
    private static final long MIN_TIMER = 60000L ;
    
    /**
     * 最大監視タイマー値.
     * 86400秒 : １日.
     */
    private static final long MAX_TIMER = 86400000L ;
    
    
    /**
     * デフォルトバッファ長.
     */
    private static final int DEFAULT_BUFFER = 5 ;
    
    /**
     * スリープタイミング.
     */
    private static final int SLEEP_TIMING = 250 ;
    
    
    /**
     * 情報管理オブジェクト.
     */
    private final CharTable m_info = new CharTable() ;
    
    /**
     * 監視タイマー値.
     */
    private volatile long m_timer = TObjectManager.DEFAULT_TIMER ;
    
    
    /**
     * 監視スレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public TObjectManager()
    {
        try{
            this.create( TObjectManager.DEFAULT_TIMER ) ;
        }catch( Exception t ){
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象監視タイマー値を設定します.<BR>
     * 設定対象の値はミリ秒単位です.
     * <BR>
     * @param timer 設定対象のタイマー値を設定します.<BR>
     *              設定可能の最小値は[60秒]です.<BR>
     *              設定可能の最大値は[1日]です.
     * @exception InputException 入力例外.
     */
    public TObjectManager( long timer ) throws InputException
    {
        try{
            this.create( timer ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象監視タイマー値を設定します.<BR>
     * 設定対象の値はミリ秒単位です.
     * <BR>
     * @param timer 設定対象のタイマー値を設定します.<BR>
     *              設定可能の最小値は[60秒]です.<BR>
     *              設定可能の最大値は[1日]です.
     * @exception InputException 入力例外.
     */
    public final void create( long timer ) throws InputException
    {
        
        if(
            timer <= TObjectManager.MIN_TIMER ||
            timer >= TObjectManager.MAX_TIMER
        )
        {
            throw new InputException( "指定タイマー値は設定範囲外です" ) ;
        }
        
        try{
            
            this.clear() ;
            
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                m_thread.create( this ) ;
                m_thread.startThread() ;
                m_info.clear() ;
                m_timer = timer ;
            }
            
        }catch( Exception t ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.<BR>
     * また、この処理では、管理スレッド情報は削除されません.
     */
    public final void clear()
    {
        m_sync.clear() ;
        
        try{
            m_info.clear() ;
        }catch( Exception t ){
        }
        try{
            m_thread.clear() ;
        }catch( Exception t ){
        }
        
        m_timer = TObjectManager.DEFAULT_TIMER ;
        
    }
    
    /**
     * 対象のオブジェクトを設定.
     * <BR><BR>
     * 管理情報に対して、対象のオブジェクト情報を設定します.
     * <BR>
     * @param key 設定対象のキー名を設定します.
     * @param info 対象の情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExistException 情報存在例外.
     */
    public final void put( String key,Object info )
        throws InputException,ExistException
    {
        String keyName = null ;
        OMPause pause = null ;
        
        if( key == null || info == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        keyName = TObjectManager.getKeyNameTo( key ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_info.get( keyName ) != null ){
                    throw new ExistException(
                        "(" + key + ")指定キー名の情報は存在します"
                    ) ;
                }
                
                pause = new OMPause() ;
                pause.timer = System.currentTimeMillis() + m_timer ;
                pause.info = info ;
                
                m_info.add( keyName,pause ) ;
                
            }
            
        }catch( NullPointerException nul ){
        }catch( ExistException ee ){
            throw ee ;
        }catch( InputException in ){
            throw in ;
        }finally{
            keyName = null ;
            pause = null ;
        }
    }
    
    /**
     * 対象情報をクリア.
     * <BR><BR>
     * 管理情報から、対象情報を削除します.
     * <BR>
     * @param key 削除対象のキー名を設定します.
     * @return Object 削除されたオブジェクト情報が返されます.<BR>
     *                キー名に対して情報が存在しない場合、[null]が返されます.
     */
    public final Object remove( String key )
    {
        Object ret = null ;
        OMPause pause = null ;
        
        if( key == null ){
            return null ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                pause = ( OMPause )m_info.remove(
                    TObjectManager.getKeyNameTo( key )
                ) ;
                
                ret = pause.info ;
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InputException in ){
            ret = null ;
        }finally{
            
            try{
                pause.timer = 0L ;
                pause.info = null ;
            }catch( Exception t ){
            }
            
            pause = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 対象のオブジェクト情報を取得.
     * <BR><BR>
     * 対象のオブジェクト情報を取得します.
     * <BR>
     * @param key 取得対象のキー名を設定します.
     * @return Object 対象のオブジェクト情報が返されます.
     * @exception InputException 入力例外.
     */
    public final Object get( String key ) throws InputException
    {
        
        OMPause pause = null ;
        Object ret = null ;
        
        if( key == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                pause = ( OMPause )m_info.get( TObjectManager.getKeyNameTo( key ) ) ;
                pause.timer = System.currentTimeMillis() + m_timer ;
                ret = pause.info ;
            }
            
        }catch( NullPointerException nul ){
        }finally{
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * 監視タイマー値の取得.
     * <BR><BR>
     * 監視タイマー値を取得します.
     * また、取得される情報値はミリ秒単位です.
     * <BR>
     * @return long 現在設定されている監視タイマー値が返されます.
     */
    public final long getTimer()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( long )m_timer ;
            }
        }catch( NullPointerException nul ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納情報数を取得します.
     * <BR>
     * @return int 格納情報数が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_info.size() ;
            }
        }catch( NullPointerException nul ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init(Object obj) throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit(Object obj) throws ExecutionException
    {
        try{
            synchronized( m_sync.get() ){
                m_info.clear() ;
            }
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop(Object obj) throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution(Object obj) throws ExecutionException
    {
        
        UtilCom.idleTime() ;
        UtilCom.sleep( TObjectManager.SLEEP_TIMING ) ;
        this.loopExecution() ;
        
    }
    
    /**
     * オブジェクト監視処理.
     */
    private final void loopExecution() throws ExecutionException
    {
        int i ;
        int len ;
        long timer ;
        
        OMPause pause = null ;
        CharTable table = null ;
        String[] keys = null ;
        
        try{
            
            table = m_info ;
            keys = table.getNames() ;
            
            if( keys != null ){
                
                len = keys.length ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    timer = System.currentTimeMillis() ;
                    
                    synchronized( m_sync.get() ){
                        
                        try{
                            
                            pause = ( OMPause )table.get( keys[ i ] ) ;
                            
                            if( timer <= pause.timer ){
                                
                                table.remove( keys[ i ] ) ;
                                
                            }
                            
                        }catch( Exception t ){
                            pause = null ;
                        }finally{
                            keys[ i ] = null ;
                        }
                        
                    }
                    
                    UtilCom.idleTime() ;
                }
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ex ){
            throw ex ;
        }
        
        keys = null ;
        pause = null ;
        table = null ;
    }
    
    
    /**
     * キー名に付加する情報を生成.
     */
    private static final String getKeyNameTo( String key )
    {
        StringBuffer buf = null ;
        String ret ;
        
        buf = new StringBuffer() ;
        buf.append( "(@" ) ;
        buf.append( Thread.currentThread().getName() ) ;
        buf.append( "@):" ) ;
        buf.append( key ) ;
        ret = buf.toString() ;
        
        buf = null ;
        return ret ;
    }
    
    
    
}
