/*
 * @(#)ObjectThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.thread;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.SettingException;
import com.JRcServer.commons.util.UtilCom;

/**
 * オブジェクトスレッド.
 * <BR><BR>
 * オブジェクトを設定し、その処理をスレッドで実行します。<BR>
 * このオブジェクトは、主にスレッドインスタンス(ObjectThread)
 * を再利用するために存在します.
 * また、このスレッド処理ではスレッド状況がスレッドログに
 * 登録されません.
 * <BR>
 *
 * @version 1.0.0 2003/11/02
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
class ObjectThread extends ProcessThread
{
    /**
     * クリアスレッドフラグ名.
     */
    private static final String CLEAR_NAME = "clear-name" ;
    
    /**
     * スレッド終了フラグ名.
     */
    private static final String EXIT_NAME = "exit-name" ;
    
    /**
     * スレッド処理開始フラグ.
     */
    private static final String GOTO_NAME = "goto-name" ;
    
    
    
    /**
     * 実行処理.
     */
    private volatile ExecutionThread m_exec = null ;
    
    /**
     * 実行引数値.
     */
    private Object m_val = null ;
    
    /**
     * スレッド実施処理.
     */
    private final BaseThread m_thread = new BaseThread() ;
    
    /**
     * スレッドステータス.
     */
    private ThreadState m_state = null ;
    
    /**
     * スレッド実施時間.
     */
    private volatile long m_nowTime = 0L ;
    
    /**
     * スレッド同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public ObjectThread()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * スレッドを利用可能な状態にします.
     * <BR>
     * @para priority 対象のプライオリティを設定します.
     * @exception InputException 入力例外.
     */
    public ObjectThread( int priority )
        throws InputException
    {
        this.create( priority ) ;
    }
    
    /**
     * スレッド生成.
     * <BR><BR>
     * スレッドを利用可能な状態にします.
     */
    public final void create()
    {
        this.clear() ;
        
        try{
            
            m_sync.create() ;
            
            m_thread.create( this,null,true ) ;
            m_thread.setObjectName( null ) ;
            
            m_state = this.initState() ;
            
            this.upThread() ;
            
        }catch( ExecutionException ee ){
            this.clear() ;
        }catch( BaseException be ){
            this.clear() ;
        }catch( Exception t ){
            this.clear() ;
        }
    }
    
    /**
     * スレッド生成.
     * <BR><BR>
     * スレッドを利用可能な状態にします.
     * <BR>
     * @para priority 対象のプライオリティを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( int priority )
        throws InputException
    {
        this.clear() ;
        
        try{
            
            m_sync.create() ;
            
            m_thread.create( this,null,true ) ;
            m_thread.setObjectName( null ) ;
            m_thread.setPriority( priority ) ;
            
            m_state = this.initState() ;
            
            this.upThread() ;
            
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( ExecutionException ee ){
            this.clear() ;
        }catch( BaseException be ){
            this.clear() ;
        }catch( Exception t ){
            this.clear() ;
        }
    }
    
    /**
     * スレッドクリア.
     * <BR><BR>
     * スレッド情報をクリア(利用不可)にします.
     */
    public final void clear()
    {
        
        m_sync.clear() ;
        
        try{
            this.setFlg( ObjectThread.EXIT_NAME,true ) ;
        }catch( Exception t ){
        }
        
        try{
            m_thread.clear() ;
        }catch( Exception t ){
        }
        
        try{
            m_state.clear() ;
        }catch( Exception t ){
        }
        
        m_exec = null ;
        m_val = null ;
        m_state = null ;
        
    }
    
    /**
     * オブジェクトスレッド処理設定／実施.
     * <BR><BR>
     * スレッド処理時に実行するオブジェクトを設定します。
     * <BR>
     * @param exec 設定対象のオブジェクト
     * @exception InputException 入力例外.
     * @exception SettingException 設定例外.
     */
    public final void execThread( ExecutionThread exec )
        throws InputException,SettingException
    {
        this.execThread( exec,null ) ;
    }
    
    /**
     * オブジェクトスレッド処理設定／実施.
     * <BR><BR>
     * スレッド処理時に実行するオブジェクトを設定します。
     * <BR>
     * @param exec 設定対象のオブジェクト
     * @param pattern スレッド実行引数を設定します.
     * @exception InputException 入力例外.
     * @exception SettingException 設定例外.
     */
    public final void execThread( ExecutionThread exec,Object pattern )
        throws InputException,SettingException
    {
        if( exec == null ){
            
            throw new InputException( "引数が不正です" ) ;
            
        }else if( this.isCreate() == false ){
            
            throw new SettingException( "スレッド実行の準備ができていません" ) ;
            
        }
        
        try{
            synchronized( m_sync.get() ){
                m_thread.setObjectName( null ) ;
                m_exec = exec ;
                m_val = pattern ;
                
                this.setFlg( ObjectThread.GOTO_NAME,true ) ;
            }
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッド破棄を実施.
     * <BR><BR>
     * スレッド破棄を実施します.
     */
    public final void destroyThread()
    {
        
        try{
            
            synchronized( m_sync.get() ){
                this.setFlg( ObjectThread.EXIT_NAME,true ) ;
            }
            
            UtilCom.idleSleep( BaseThread.DESTROY_WAIT ) ;
            
        }catch( Exception t ){
        }finally{
            
            try{
                m_thread.destroyThread() ;
            }catch( Exception t1 ){
            }
            
        }
        
    }
    
    /**
     * スレッドステータス情報を取得.
     * <BR><BR>
     * スレッドステータス情報を取得します.
     * <BR>
     * @return int スレッドステータス情報が返されます.
     */
    public final int getState()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getThreadState() ;
            }
        }catch( Exception t ){
            ret = ExecutionThread.STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実施時間の取得.
     * <BR><BR>
     * スレッドが実施された時間を取得します.
     * <BR>
     * @return long スレッドが実施された時間が返されます.
     */
    public final long getThreadTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_nowTime ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドプライオリティを取得.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを取得します.
     * <BR>
     * @return int 設定されているスレッドプライオリティが返されます.<BR>
     *             スレッドが存在しない場合[-1]が返されます.
     */
    public final int getPriority()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getPriority() ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行用オブジェクトを取得.
     * <BR><BR>
     * スレッド実行用オブジェクトを取得します.
     * <BR>
     * @return ExecutionThread スレッド実行用オブジェクトが返されます.
     */
    public final ExecutionThread getExecutionThread()
    {
        ExecutionThread ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_exec ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行チェック.
     * <BR><BR>
     * スレッド実行チェックを行います.
     * <BR>
     * @return boolean スレッドが実行中であるかチェックが返されます.<BR>
     *                 [true]が返された場合、スレッド情報は実施中です.<BR>
     *                 [false]が返された場合、スレッド情報は実施されていません.
     */
    public final boolean isExec()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getFlg( ObjectThread.GOTO_NAME ) ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    /**
     * スレッド生成チェック.
     * <BR><BR>
     * スレッド生成チェックを行います.
     * <BR>
     * @return boolean スレッド生成チェックが返されます.<BR>
     *                 [true]が返された場合、スレッド情報は生成されています.<BR>
     *                 [false]が返された場合、スレッド情報は生成されていません.
     */
    public final boolean isCreate()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( this.getFlg( ObjectThread.CLEAR_NAME ) == true ) ?
                    false : true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行.
     * <BR><BR>
     * スレッド実行を行う処理部を実装します.
     * @param in スレッド処理での引数情報を指定します.
     * @exception ExecutionException 実行例外.
     * @exception ThreadDeath スレッド破棄通知.
     */
    protected void processThread( Object in )
        throws ExecutionException,ThreadDeath
    {
        
        boolean endFlg ;
        endFlg = false ;
        
        try{
            
            this.setFlg( ObjectThread.CLEAR_NAME,false ) ;
            this.setFlg( ObjectThread.EXIT_NAME,false ) ;
            this.setFlg( ObjectThread.GOTO_NAME,false ) ;
                
            synchronized( m_sync.get() ){
                m_nowTime = System.currentTimeMillis() ;
            }
            
            // スレッド待機.
            for(;;){
                
                UtilCom.idleTime() ;
                
                // スレッド終了を検知.
                if( this.getFlg( ObjectThread.EXIT_NAME ) == true ){
                    break ;
                }
                // スレッド実施を検知.
                else if( this.getFlg( ObjectThread.GOTO_NAME ) == true ){
                    
                    // スレッド処理が実施された場合.
                    try{
                        
                        endFlg = true ;
                        
                        // 初期処理.
                        m_exec.init( m_val ) ;
                        
                        endFlg = false ;
                        
                        // 実施処理.
                        m_exec.execution( m_val ) ;
                        
                        endFlg = true ;
                        
                        // 終了処理.
                        m_exec.exit( m_val ) ;
                        
                    }catch( ExecutionException eex ){
                        
                        // 終了処理が行われておらず,ストップ処理が検知された場合.
                        if(
                            endFlg == false &&
                            eex.getLevel() == ExecutionException.LEVEL_STOP
                        )
                        {
                            
                            try{
                                // 終了処理.
                                m_exec.exit( m_val ) ;
                            }catch( Exception tt ){
                            }
                            
                        }
                        
                    }finally{
                        
                        this.setFlg( ObjectThread.GOTO_NAME,false ) ;
                        
                        synchronized( m_sync.get() ){
                            
                            m_exec = null ;
                            m_val = null ;
                            
                            m_nowTime = System.currentTimeMillis() ;
                            
                        }
                        
                    }
                }
                else{
                    UtilCom.idleTime() ;
                }
                
            }
            
        }catch( ThreadDeath death ){
            throw death ;
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ex ){
            throw ex ;
        }catch( Exception t ){
        }finally{
            
            try{
                this.setFlg( ObjectThread.CLEAR_NAME,true ) ;
                this.setFlg( ObjectThread.EXIT_NAME,false ) ;
                this.setFlg( ObjectThread.GOTO_NAME,false ) ;
            }catch( Exception tt ){
            }
            
            m_nowTime = 0L ;
            m_exec = null ;
            m_val = null ;
            
        }
    }
    
    /**
     * スレッド破棄時の処理.
     * <BR><BR>
     * スレッド破棄時の処理を生成します.
     * <BR>
     * @param in スレッド処理で渡される引数情報です.
     * @exception Exception 処理エラーを設定します.
     */
    protected void destroyThread( Object in )
        throws Exception
    {
        m_exec.destroy( in ) ;
    }
    
    /**
     * スレッド例外発生処理実施.
     * <BR><BR>
     * スレッド処理での例外が発生した場合、実施されます.
     * <BR>
     * @param err 実行例外が返されます.
     */
    protected final void toException( ExecutionException err )
    {
        // 特になし..
    }
    
    
    
    
    
    
    /**
     * スレッド起動処理.
     */
    private final void upThread()
    {
        try{
            if( this.getFlg( ObjectThread.CLEAR_NAME ) == true ){
                // スレッド実行.
                m_thread.startThread() ;
            }
        }catch( Exception t ){
        }
    }
    
    
    /**
     * ステータス初期化処理.
     */
    private synchronized final ThreadState initState()
        throws AccessException
    {
        boolean[] value = null ;
        ThreadState ret = null ;
        
        try{
            
            // スレッド対象のパラメータ値を取得.
            ret = ThreadManager.
                getInstance().
                    getState( m_thread.getName() ) ;
            
            // 各パラメータを初期処理.
            value = new boolean[ 1 ] ;
            value[ 0 ] = true ;
            ret.setParameter( ObjectThread.CLEAR_NAME,value ) ;
            
            value = new boolean[ 1 ] ;
            value[ 0 ] = true ;
            ret.setParameter( ObjectThread.EXIT_NAME,value ) ;
            
            value = new boolean[ 1 ] ;
            value[ 0 ] = false ;
            ret.setParameter( ObjectThread.GOTO_NAME,value ) ;
            
        }catch( Exception t ){
            ret = null ;
            throw new AccessException( t ) ;
        }finally{
            value = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * フラグの設定.
     */
    private synchronized final void setFlg( String name,boolean flg )
        throws NullPointerException
    {
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            state = m_state ;
            obj = ( boolean[] )state.getParameter( name ) ;
            obj[ 0 ] = flg ;
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
    }
    
    /**
     * フラグの取得.
     */
    private synchronized final boolean getFlg( String name )
        throws NullPointerException
    {
        boolean ret ;
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            state = m_state ;
            obj = ( boolean[] )state.getParameter( name ) ;
            ret = obj[ 0 ] ;
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
        
        return ret ;
    }
    
}

