/*
 * @(#)LoopThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.thread;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.SettingException;
import com.JRcServer.commons.util.UtilCom;

/**
 * ループスレッド.
 * <BR><BR>
 * ループスレッド処理をサポートします。
 * <BR>
 *
 * @version 1.0.0 2003/11/01
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class LoopThread extends ProcessThread
{
    
    /**
     * スレッド動作フラグ名.
     */
    private static final String RUN_NAME = "run" ;
    
    /**
     * スレッド終了フラグ名.
     */
    private static final String END_NAME = "end" ;
    
    /**
     * スレッドストップフラグ.
     */
    private static final String STOP_NAME = "stop" ;
    
    
    
    /**
     * 実行オブジェクト.
     */
    private ExecutionThread m_exec = null ;
    
    /**
     * スレッド実施処理.
     */
    private final BaseThread m_thread = new BaseThread() ;
    
    /**
     * スレッドステータス.
     */
    private ThreadState m_state = null ;
    
    /**
     * スレッド実施時間.
     */
    private volatile long m_nowTime = 0L ;
    
    /**
     * スレッド同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public LoopThread()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec ループ処理を行う[ExecutionThread]インターフェイスを設定します.
     * @exception InputException 入力例外.
     */
    public LoopThread( ExecutionThread exec )
        throws InputException
    {
        try{
            this.create( exec ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec ループ処理を行う[ExecutionThread]インターフェイスを設定します.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @exception InputException 入力例外.
     */
    public LoopThread( ExecutionThread exec,Object param )
        throws InputException
    {
        try{
            this.create( exec,param ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param exec ループ処理を行う[ExecutionThread]インターフェイスを設定します.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @exception InputException 入力例外.
     */
    public LoopThread( boolean mode,ExecutionThread exec,Object param )
        throws InputException
    {
        try{
            this.create( mode,exec,param ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec ループ処理を行う[ExecutionThread]インターフェイスを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( ExecutionThread exec )
        throws InputException
    {
        try{
            this.create( exec,null ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param exec ループ処理を行う[ExecutionThread]インターフェイスを設定します.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( ExecutionThread exec,Object param )
        throws InputException
    {
        this.create( true,exec,param ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param exec ループ処理を行う[ExecutionThread]インターフェイスを設定します.
     * @param param スレッド処理時に渡されるパラメータを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( boolean mode,ExecutionThread exec,Object param )
        throws InputException
    {
        if( exec == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.clear() ;
        
        try{
            
            m_thread.create( this,param,true ) ;
            
            if( mode == true ){
                m_thread.setObjectName( exec.getClass().getName() ) ;
            }
            else{
                m_thread.setObjectName( null ) ;
            }
            
            m_exec = exec ;
            m_state = this.initState() ;
            m_sync.create() ;
            
        }catch( Exception t ){
            
            this.clear() ;
            
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        try{
            m_thread.clear() ;
        }catch( Exception t ){
        }
        
        try{
            m_state.clear() ;
        }catch( Exception tt ){
        }
        
        m_exec = null ;
        m_state = null ;
        m_nowTime = 0L ;
        m_sync.clear() ;
        
    }
    
    /**
     * スレッド開始処理を実施.
     * <BR><BR>
     * スレッド開始処理を実施します.
     * <BR>
     * @exception SettingException 設定例外.
     */
    public final void startThread()
        throws SettingException
    {
        int state ;
        
        if( this.isExecution() == false ){
            throw new SettingException( "設定情報が不足しています" ) ;
        }
        
        // スレッド実行.
        m_thread.startThread() ;
        
        // スレッド実行待ち.
        /*for( ;; ){
            
            state = m_thread.getState() ;
            
            if(
                state == ExecutionThread.STATE_NOT ||
                state == ExecutionThread.STATE_EXECUTION
            )
            {
                if( state == ExecutionThread.STATE_NOT ){
                    this.clear() ;
                    throw new SettingException(
                        "スレッド実行に失敗しました"
                    ) ;
                }
                break ;
            }
            
            UtilCom.cpuCreate() ;
            
        }*/
        
    }
    
    /**
     * スレッド終了処理を実施.
     * <BR><BR>
     * スレッド終了処理を実施します.
     */
    public final void exitThread()
    {
        
        try{
            
            synchronized( m_sync.get() ){
                
                this.setFlg( LoopThread.END_NAME,true ) ;
                
            }
            
        }catch( Exception t ){
        }finally{
            
            try{
                m_thread.exitThread() ;
            }catch( Exception t1 ){
            }
            
        }
        
    }
    
    /**
     * ループストップ処理.
     * <BR><BR>
     * ループストップ処理のON／OFFを行います.
     */
    public final void stopLoop()
    {
        boolean flg ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                flg = ( this.getFlg( LoopThread.STOP_NAME ) == true ) ?
                    false : true ;
                
                this.setFlg( LoopThread.END_NAME,flg ) ;
                
            }
            
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * スレッド破棄を実施.
     * <BR><BR>
     * スレッド破棄を実施します.
     */
    public final void destroyThread()
    {
        
        try{
            
            synchronized( m_sync.get() ){
                this.setFlg( LoopThread.END_NAME,true ) ;
            }
            
            UtilCom.idleSleep( BaseThread.DESTROY_WAIT ) ;
            
        }catch( Exception t ){
        }finally{
            
            try{
                m_thread.destroyThread() ;
            }catch( Exception t1 ){
            }
            
        }
        
    }
    
    /**
     * スレッドプライオリティを設定.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを設定します.
     * <BR>
     * @param newPriority 新しいスレッドプライオリティを設定します.
     * @exception InputException 入力例外.
     */
    public final void setPriority( int newPriority )
        throws InputException
    {
        try{
            synchronized( m_sync.get() ){
                m_thread.setPriority( newPriority ) ;
            }
        }catch( InputException in ){
            throw in ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッドステータス情報を取得.
     * <BR><BR>
     * スレッドステータス情報を取得します.
     * <BR>
     * @return int スレッドステータス情報が返されます.
     */
    public final int getState()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getThreadState() ;
            }
        }catch( Exception t ){
            ret = ExecutionThread.STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実施時間の取得.
     * <BR><BR>
     * スレッドが実施された時間を取得します.
     * <BR>
     * @return long スレッドが実施された時間が返されます.
     */
    public final long getThreadTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_nowTime ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドプライオリティを取得.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを取得します.
     * <BR>
     * @return int 設定されているスレッドプライオリティが返されます.<BR>
     *             スレッドが存在しない場合[-1]が返されます.
     */
    public final int getPriority()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getPriority() ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行用オブジェクトを取得.
     * <BR><BR>
     * スレッド実行用オブジェクトを取得します.
     * <BR>
     * @return ExecutionThread スレッド実行用オブジェクトが返されます.
     */
    public final ExecutionThread getExecutionThread()
    {
        ExecutionThread ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_exec ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド開始チェック.
     * <BR><BR>
     * スレッドが開始されているかチェックします.
     * <BR>
     * @return boolean スレッドチェック.<BR>
     * スレッドが開始している場合は、戻り値は[true]で返されます。<BR>
     * スレッドが開始されていない場合、戻り値は[false]で返されます。
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = this.getFlg( LoopThread.RUN_NAME ) ;
                
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ループストップチェック.
     * <BR><BR>
     * ループストップチェックを取得します.
     * <BR>
     * @return boolean [true]が返された場合、ループ処理はストップしています.
     *                 [false]が返された場合、ループ処理はストップしていません.
     */
    public final boolean isStop()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = this.getFlg( LoopThread.STOP_NAME ) ;
                
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ループエンドチェック.
     * <BR><BR>
     * ループエンドチェックを取得します.
     * <BR>
     * @return boolean [true]が返された場合、ループ処理は終了待ち状態です.
     *                 [false]が返された場合、ループ処理は終了待ち状態ではありません.
     */
    public final boolean isEnd()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = this.getFlg( LoopThread.END_NAME ) ;
                
            }
            
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行.
     * <BR><BR>
     * スレッド実行を行う処理部を実装します.
     * @param in スレッド処理での引数情報を指定します.
     * @exception ExecutionException 実行例外.
     * @exception ThreadDeath スレッド破棄通知.
     */
    protected void processThread( Object in )
        throws ExecutionException,ThreadDeath
    {
        
        try{
            
            synchronized( m_sync.get() ){
                
                this.setFlg( LoopThread.RUN_NAME,true ) ;
                this.setFlg( LoopThread.END_NAME,false ) ;
                this.setFlg( LoopThread.STOP_NAME,false ) ;
                
                m_nowTime = System.currentTimeMillis() ;
                
            }
            
            m_exec.init( in ) ;
            
            for( ;; ){
                
                //Thread.yield() ;
                ThreadManager.isExit() ;
                
                if( this.isEnd() == true ){
                    break ;
                }
                
                if( this.isStop() == true ){
                    
                    m_exec.stop( in ) ;
                    
                    synchronized( m_sync.get() ){
                        m_nowTime = System.currentTimeMillis() ;
                    }
                    
                    UtilCom.idleTime() ;
                    
                    continue ;
                }
                
                m_exec.execution( in ) ;
                
                ThreadManager.isExit() ;
                //Thread.yield() ;
                
                synchronized( m_sync.get() ){
                    m_nowTime = System.currentTimeMillis() ;
                }
                
            }
            
        }catch( ThreadDeath death ){
            throw death ;
        }catch( NullPointerException nul ){
        }catch( ExecutionException ex1 ){
            
            throw ex1 ;
            
        }catch( Exception t ){
        }finally{
            
            try{
                
                m_exec.exit( in ) ;
                
            }catch( Exception tt ){
            }
            
            try{
                
                synchronized( m_sync.get() ){
                    
                    this.setFlg( LoopThread.RUN_NAME,false ) ;
                    this.setFlg( LoopThread.END_NAME,true ) ;
                    this.setFlg( LoopThread.STOP_NAME,false ) ;
                    m_nowTime = 0L ;
                    
                }
                
            }catch( Exception tt ){
                m_nowTime = 0L ;
            }
            
        }
    }
    
    /**
     * スレッド破棄時の処理.
     * <BR><BR>
     * スレッド破棄時の処理を生成します.
     * <BR>
     * @param in スレッド処理で渡される引数情報です.
     * @exception Exception 処理エラーを設定します.
     */
    protected void destroyThread( Object in )
        throws Exception
    {
        m_exec.destroy( in ) ;
    }
    
    /**
     * スレッド例外発生処理実施.
     * <BR><BR>
     * スレッド処理での例外が発生した場合、実施されます.
     * <BR>
     * @param err 実行例外が返されます.
     */
    protected void toException( ExecutionException err )
    {
        // 特になし..
    }
    
    
    
    
    /**
     * 情報設定準備チェック.
     */
    private final boolean isExecution()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( m_exec == null ) ? false : true ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ステータス初期化処理.
     */
    private final ThreadState initState()
        throws AccessException
    {
        boolean[] value = null ;
        ThreadState ret = null ;
        
        try{
            
            // スレッド対象のパラメータ値を取得.
            ret = ThreadManager.
                getInstance().
                    getState( m_thread.getName() ) ;
            
            // 各パラメータを初期処理.
            value = new boolean[ 1 ] ;
            value[ 0 ] = false ;
            ret.setParameter( LoopThread.RUN_NAME,value ) ;
            
            value = new boolean[ 1 ] ;
            value[ 0 ] = true ;
            ret.setParameter( LoopThread.END_NAME,value ) ;
            
            value = new boolean[ 1 ] ;
            value[ 0 ] = false ;
            ret.setParameter( LoopThread.STOP_NAME,value ) ;
            
        }catch( Exception t ){
            
            ret = null ;
            throw new AccessException( t ) ;
            
        }finally{
            
            value = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * フラグの設定.
     */
    private final void setFlg( String name,boolean flg )
        throws NullPointerException
    {
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                state = m_state ;
                obj = ( boolean[] )state.getParameter( name ) ;
                obj[ 0 ] = flg ;
                
            }
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
    }
    
    /**
     * フラグの取得.
     */
    private final boolean getFlg( String name )
        throws NullPointerException
    {
        boolean ret ;
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                state = m_state ;
                obj = ( boolean[] )state.getParameter( name ) ;
                ret = obj[ 0 ] ;
                
            }
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
        
        return ret ;
    }
    
}
