/*
 * @(#)UserManagerEngine.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.sys.user;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.conv.CodeBase32;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ByteUtil;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.ConvertParam;
import com.JRcServer.commons.util.IdManager;
import com.JRcServer.commons.util.NumberTable;
import com.JRcServer.commons.util.UtilCom;

/**
 * ユーザ管理ユーザ管理.
 * <BR><BR>
 * ユーザ情報を管理するユーザ管理情報です.
 *  
 * @version 2005/09/18
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
class UserManagerEngine extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( UserManagerEngine.class ) ;
    
    /**
     * 基本キャラクターセット.
     */
    public static final String CHARSET = BaseDef.UTF8 ;
    
    /**
     * ヘッダキャラクターセット.
     */
    public static final String HEADER_CHARSET = BaseDef.UTF8 ;
    
    /**
     * ROOTユーザ名.
     */
    public static final String ROOT_USER = "root" ;
    
    /**
     * GUESTユーザ名.
     */
    public static final String GUEST_USER = "guest" ;
    
    /**
     * パスワード無し情報.
     */
    public static final String NOSET_PASSWD = "" ;
    
    /**
     * ROOTパスワード.
     */
    public static final String ROOT_PASSWD = NOSET_PASSWD ;
    
    /**
     * ログインID開始値.
     */
    public static final int START_LOGIN_ID = 1 ;
    
    /**
     * ログインID最大値.
     */
    public static final int MAX_LOGIN_ID = 9999999 ;
    
    /**
     * 保存間隔 : デフォルト値.
     * 30sec.
     */
    public static final long DEF_SAVE_TIMING = 30000 ;
    
    /**
     * 保存間隔 : 最小値.
     * 5sec.
     */
    public static final long MIN_SAVE_TIMING = 5000 ;
    
    /**
     * 保存間隔 : 最大値.
     * 120000sec.
     */
    public static final long MAX_SAVE_TIMING = 120000 ;
    
    /**
     * 非アップデート時間.
     */
    public static final long NOT_UPDATE_TIME = Long.MAX_VALUE - MAX_SAVE_TIMING ;
    
    /**
     * ログイン管理数 - 無限アクセス.
     */
    public static final int FREE_MAX_LOGIN = 0 ;
    
    /**
     * ログイン管理数 - 停止状態.
     */
    public static final int STOP_MAX_LOGIN = -1 ;
    
    
    
    /**
     * デフォルトユーザログイン最大数.
     */
    private static final int DEF_LOGIN = 5 ;
    
    /**
     * 同時ユーザログイン最大数.
     */
    private static final int MAX_LOGIN = 512 ;
    
    /**
     * ユーザ管理ファイルヘッダ.
     */
    private static final String USER_HEADER = "hd@URg" ;
    
    
    
    /**
     * ユーザ名管理テーブル.
     */
    private final CharTable m_table = new CharTable() ;
    
    /**
     * ユーザID管理.
     */
    private final NumberTable m_idTable = new NumberTable() ;
    
    /**
     * ID発行オブジェクト.
     */
    private final IdManager m_idMan = new IdManager(
        START_LOGIN_ID,MAX_LOGIN_ID
    ) ;
    
    /**
     * ユーザ情報格納ファイル名.
     */
    private String m_userFileName = null ;
    
    /**
     * データ書き込み最終時間.
     */
    private long m_lastTime = NOT_UPDATE_TIME ;
    
    /**
     * 更新間隔設定.
     */
    private long m_updateTimer = 0L ;
    
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 書き込み用.
     */
    private final Synchronized m_writeSync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    protected UserManagerEngine()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * ユーザユーザ管理エンジンを生成します.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void create( String name )
        throws InputException,AccessException
    {
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        this.initUserManager( name,DEF_SAVE_TIMING ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * ユーザユーザ管理エンジンを生成します.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @param timing データがアップデートされてからの保存間隔を設定します.<BR>
     *               また、指定単位はミリ秒です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void create( String name,long timing )
        throws InputException,AccessException
    {
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        this.initUserManager( name,timing ) ;
    }
    
    /**
     * ユーザ管理をクリア.
     * <BR><BR>
     * ユーザ管理をクリアします.
     */
    protected final void clear()
    {
        
        // 同期終了.
        m_sync.clear() ;
        m_writeSync.clear() ;
        
        // スレッドストップ.
        m_thread.clear() ;
        
        // ユーザ管理名が存在する場合.
        if( m_userFileName != null && m_userFileName.length() > 0 ){
            
            try{
                
                // ユーザ管理保存.
                this.save( m_userFileName ) ;
                
            }catch( Exception e ){
                
                LOG.error( "error",e ) ;
                
            }
            
        }
        
        m_table.clear() ;
        m_idTable.clear() ;
        m_idMan.clear() ;
        m_userFileName = null ;
        m_lastTime = NOT_UPDATE_TIME ;
        m_updateTimer = 0L ;
        
    }
    
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * ユーザ情報を追加します.<BR>
     * また、最大接続数はデフォルト値(5)で設定されます.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void addUser( String user,String passwd )
        throws InputException
    {
        this.addUser( user,passwd,false,false,DEF_LOGIN,null ) ;
    }
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * ユーザ情報を追加します.<BR>
     * また、最大接続数はデフォルト値(5)で設定されます.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @param rootOwner 対象ユーザに対してルート権限を与えるか設定します.
     * @param rootGroup 対象ユーザに対してルートグループ権限を与えるか設定します.
     * @exception InputException 入力例外.
     */
    public final void addUser( String user,String passwd,boolean rootOwner,boolean rootGroup )
        throws InputException
    {
        this.addUser( user,passwd,rootOwner,rootGroup,DEF_LOGIN,null ) ;
    }
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * ユーザ情報を追加します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @param rootOwner 対象ユーザに対してルート権限を与えるか設定します.
     * @param rootGroup 対象ユーザに対してルートグループ権限を与えるか設定します.
     * @param max ユーザ同時接続数を設定します.<BR>
     *            設定可能な最大値は[512]です.<BR>
     *            また[0]とするか、最大値以上を設定した場合、同時ユーザ接続数の上限が無くなります.<BR>
     *            また[-1]を設定することで、このユーザを利用出来なくします.
     * @param extension 拡張情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void addUser( String user,String passwd,boolean rootOwner,boolean rootGroup,int max,String[] extension )
        throws InputException
    {
        int i ;
        int len ;
        int id ;
        
        CharTable tbl = null ;
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        passwd = ( ( passwd == null || passwd.length() <= 0 ) ?
            UserManagerEngine.NOSET_PASSWD :
            UtilCom.trimPlus( passwd )
        ) ;
        
        max = ( max < 0 ) ? STOP_MAX_LOGIN :
            (
                ( max > MAX_LOGIN ) ?
                    FREE_MAX_LOGIN : max
            ) ;
        
        if( extension != null ){
            if( ( len = extension.length ) != 0 ){
                for( i = 0 ; i < len ; i ++ ){
                    extension[ i ] = UtilCom.trimPlus( extension[ i ] ) ;
                }
            }
            else{
                extension = null ;
            }
        }
        else{
            extension = null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                tbl = m_table ;
                
                if( tbl.isData( user ) == true ){
                    throw new InputException(
                        "指定ユーザ名(" + user +
                        ")は既に存在します"
                    ) ;
                }
                
                if( ( id = m_idMan.getID() ) <= 0 ){
                    throw new InputException(
                        "ユーザIDの発行に失敗しました"
                    ) ;
                }
                
                child = new UserChild() ;
                child.m_userID = id ;
                child.m_passwd = passwd ;
                child.m_rootOwner = rootOwner ;
                child.m_rootGroup = rootGroup ;
                child.m_max = max ;
                child.m_extension = extension ;
                
                synchronized( m_writeSync.get() ){
                    
                    tbl.add( user,child ) ;
                    m_idTable.add( id,user ) ;
                    
                    m_lastTime = System.currentTimeMillis() ;
                    
                }
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }finally{
            tbl = null ;
            child = null ;
        }
    }
    
    /**
     * ユーザ削除.
     * <BR><BR>
     * 対象のユーザ名を削除します.
     * <BR>
     * @param user 削除対象のユーザ名を設定します.
     * @exception InputException 入力例外.
     */
    public final void removeUser( String user )
        throws InputException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        if( UserManagerEngine.checkUserName( true,user ) == true ){
            throw new InputException(
                "対象ユーザ名(" + user +
                ")は削除する事は出来ません"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    synchronized( m_writeSync.get() ){
                        
                        child = ( UserChild )m_table.remove( user ) ;
                        
                        if( child != null ){
                            
                            // ユーザID条件を削除.
                            m_idMan.removeID( child.m_userID ) ;
                            m_idTable.remove( child.m_userID ) ;
                            
                            m_lastTime = System.currentTimeMillis() ;
                            
                        }
                    }
                    
                }
                
            }
        }catch( Exception e ){
            LOG.error( "error",e ) ;
        }finally{
            child = null ;
        }
        
    }
    
    /**
     * 対象ユーザに対するパスワード変更.
     * <BR><BR>
     * 対象のユーザ情報のパスワードを変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param newPasswd 変更対象のパスワード名を設定します.
     * @exception InputException 入力例外.
     */
    public final void renewPasswd( String user,String newPasswd )
        throws InputException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        newPasswd = ( ( newPasswd == null || newPasswd.length() <= 0 ) ?
            UserManagerEngine.NOSET_PASSWD :
            UtilCom.trimPlus( newPasswd )
        ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    synchronized( m_writeSync.get() ){
                        child = ( UserChild )m_table.get( user ) ;
                        child.m_passwd = newPasswd ;
                        m_lastTime = System.currentTimeMillis() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するROOT権限を変更.
     * <BR><BR>
     * 対象のユーザ情報のROOT権限を変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param owner 対象ユーザに対してルート権限を与えるか設定します.
     * @exception InputException 入力例外.
     */
    public final void renewRootOwner( String user,boolean owner )
        throws InputException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        if( UserManagerEngine.checkUserName( true,user ) == true ){
            throw new InputException(
                "対象ユーザ名(" + user +
                ")はROOT権限を変更する事が出来ません"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    synchronized( m_writeSync.get() ){
                        child = ( UserChild )m_table.get( user ) ;
                        child.m_rootOwner = owner ;
                        m_lastTime = System.currentTimeMillis() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するROOTグループを変更.
     * <BR><BR>
     * 対象のユーザ情報のROOTグループを変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param group 対象ユーザに対してルートグループを与えるか設定します.
     * @exception InputException 入力例外.
     */
    public final void renewRootGroup( String user,boolean group )
        throws InputException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        if( UserManagerEngine.checkUserName( true,user ) == true ){
            throw new InputException(
                "対象ユーザ名(" + user +
                ")はROOTグループ権限を変更する事が出来ません"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    synchronized( m_writeSync.get() ){
                        child = ( UserChild )m_table.get( user ) ;
                        child.m_rootGroup = group ;
                        m_lastTime = System.currentTimeMillis() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対する最大接続数を変更.
     * <BR><BR>
     * 対象のユーザ情報の最大接続数を変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param newMax 変更対象の最大接続数を設定します.
     * @exception InputException 入力例外.
     */
    public final void renewMaxUserCount( String user,int newMax )
        throws InputException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        newMax = ( newMax < 0 ) ? STOP_MAX_LOGIN :
            (
                ( newMax > MAX_LOGIN ) ?
                    FREE_MAX_LOGIN : newMax
            ) ;
        
        if( UserManagerEngine.checkUserName( false,user ) == true ){
            throw new InputException(
                "対象ユーザ名(" + user +
                ")は最大接続数を変更する事が出来ません"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    synchronized( m_writeSync.get() ){
                        child = ( UserChild )m_table.get( user ) ;
                        child.m_max = newMax ;
                        m_lastTime = System.currentTimeMillis() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対する拡張情報を変更.
     * <BR><BR>
     * 対象のユーザ情報の拡張情報を変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param newExtension 変更対象の拡張情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void renewExtension( String user,String[] newExtension )
        throws InputException
    {
        int i ;
        int len ;
        
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        if( newExtension != null ){
            if( ( len = newExtension.length ) != 0 ){
                for( i = 0 ; i < len ; i ++ ){
                    newExtension[ i ] = UtilCom.trimPlus( newExtension[ i ] ) ;
                }
            }
            else{
                newExtension = null ;
            }
        }
        else{
            newExtension = null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    synchronized( m_writeSync.get() ){
                        child = ( UserChild )m_table.get( user ) ;
                        child.m_extension = newExtension ;
                        m_lastTime = System.currentTimeMillis() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザのログインカウントを１インクリメント.
     * <BR><BR>
     * 対象ユーザのログインカウントを１インクリメントします.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @exception InputException 入力例外.
     * @exception LoginException ログイン例外.
     */
    public final void addUserCount( String user )
        throws InputException,LoginException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    
                    if( child.m_max <= STOP_MAX_LOGIN ){
                        throw new LoginException(
                            "対象のユーザ(" + user +
                            ")はログイン禁止状態です"
                        ) ;
                    }
                    else if( child.m_max != FREE_MAX_LOGIN && child.m_max <= child.m_nowUserCount ){
                        throw new LoginException(
                            "対象のユーザ(" + user +
                            ")は最大ログイン数を越しています"
                        ) ;
                    }
                    
                    child.m_nowUserCount ++ ;
                    
                }
                
            }
        }catch( LoginException le ){
            throw le ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザのログインカウントを１デクリメント.
     * <BR><BR>
     * 対象ユーザのログインカウントを１デクリメントします.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @exception InputException 入力例外.
     */
    public final void removeUserCount( String user )
        throws InputException
    {
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    
                    if( child.m_max != STOP_MAX_LOGIN ){
                        
                        child = ( UserChild )m_table.get( user ) ;
                        child.m_nowUserCount = ( child.m_nowUserCount <= 0 ) ?
                            0 : child.m_nowUserCount - 1 ;
                            
                    }
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するパスワード取得.
     * <BR><BR>
     * 対象のユーザ名に対するパスワードを取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return String 対象のパスワードが返されます.
     * @exception InputException 入力例外.
     */
    public final String getPasswd( String user )
        throws InputException
    {
        UserChild child = null ;
        String ret = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_passwd ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対するルート権限を取得.
     * <BR><BR>
     * 対象のユーザ名に対するルート権限を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return boolean 対象のルート権限が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     * @exception InputException 入力例外.
     */
    public final boolean getRootOwner( String user )
        throws InputException
    {
        boolean ret = false ;
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_rootOwner ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対するルートグループ権限を取得.
     * <BR><BR>
     * 対象のユーザ名に対するルートグループ権限を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return boolean 対象のルートグループ権限が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     * @exception InputException 入力例外.
     */
    public final boolean getRootGroupOwner( String user )
        throws InputException
    {
        boolean ret = false ;
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_rootGroup ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対する現在の接続数を取得.
     * <BR><BR>
     * 対象のユーザ名に対する現在の接続数を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return int 現在の接続数が返されます.
     * @exception InputException 入力例外.
     */
    public final int getNowUserCount( String user )
        throws InputException
    {
        int ret = 0 ;
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_nowUserCount ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対する最大接続数を取得.
     * <BR><BR>
     * 対象のユーザ名に対する最大接続数を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return int 対象の最大接続数が返されます.<BR>
     *             [-1]が返された場合、ログイン出来ません.
     *             [0]が返された場合、無制限にログイン出来ます.
     * @exception InputException 入力例外.
     */
    public final int getMaxUserCount( String user )
        throws InputException
    {
        int ret = STOP_MAX_LOGIN ;
        UserChild child = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_max ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = STOP_MAX_LOGIN ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対する拡張情報を取得.
     * <BR><BR>
     * 対象のユーザ名に対する拡張情報を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return String[] 対象の拡張情報が返されます.
     * @exception InputException 入力例外.
     */
    public final String[] getExtension( String user )
        throws InputException
    {
        UserChild child = null ;
        String[] ret = null ;
        
        user = UtilCom.trimPlus( user ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_extension ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザ名に対するユーザIDを取得.
     * <BR><BR>
     * 対象のユーザに対するユーザIDを取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return int 対象のユーザIDが返されます.<BR>
     *             [IdManager.NOT_ID]が返された場合、対象のユーザ名は
     *             存在しません
     */
    public final int getUserNameByUserID( String user )
    {
        int ret ;
        UserChild child = null ;
        
        ret = IdManager.NOT_ID ;
        
        try{
            
            user = UtilCom.trimPlus( user ) ;
            
            synchronized( m_sync.get() ){
                if( m_table.isData( user ) == true ){
                    
                    child = ( UserChild )m_table.get( user ) ;
                    ret = child.m_userID ;
                    
                }
                
            }
        }catch( Exception e ){
            ret = IdManager.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザIDに対するユーザ名を取得.
     * <BR><BR>
     * 対象のユーザIDに対するユーザ名を取得します.
     * <BR>
     * @param id 対象のユーザIDを設定します.
     * @return String 対象のユーザ名が返されます.
     */
    public final String getUserIDByUserName( int id )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( String )m_idTable.get( id ) ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * ユーザ名一覧を取得.
     * <BR><BR>
     * 管理ユーザ名一覧を取得します.
     * <BR>
     * @return String[] 管理されているユーザ名一覧が返されます.
     */
    public final String[] getUsers()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_table.getNames() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザ名数を取得.
     * <BR><BR>
     * 管理ユーザ名数を取得します.
     * <BR>
     * @return int 管理されているユーザ名数を取得します.
     */
    public final int getUserLength()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_table.size() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザ存在チェック.
     * <BR><BR>
     * 対象ユーザ名が存在するかチェックします.
     * <BR>
     * @param user チェック対象のユーザ名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合対象ユーザ名は存在します.<BR>
     *                 [false]が返された場合対象ユーザ名は存在しません.
     */
    public final boolean isUser( String user )
    {
        boolean ret ;
        
        try{
            user = UtilCom.trimPlus( user ) ;
            synchronized( m_sync.get() ){
                ret = m_table.isData( user ) ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 利用可能ユーザチェック.
     * <BR><BR>
     * 利用可能なユーザチェックを行います.
     * <BR>
     * @param user チェック対象のユーザ名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合対象ユーザ名は存在します.<BR>
     *                 [false]が返された場合対象ユーザ名は存在しません.
     */
    public final boolean isUseUser( String user )
    {
        boolean ret = false ;
        UserChild child = null ;
        
        try{
            
            user = UtilCom.trimPlus( user ) ;
            
            synchronized( m_sync.get() ){
                
                if( m_table.isData( user ) == true ){
                    child = ( UserChild )m_table.get( user ) ;
                    
                    if(
                        child.m_max <= STOP_MAX_LOGIN ||
                        (
                            child.m_max != FREE_MAX_LOGIN &&
                            child.m_max <= child.m_nowUserCount
                        )
                    ){
                        ret = false ;
                    }
                    else{
                        ret = true ;
                    }
                }
            }
        }catch( Exception e ){
            ret = false ;
        }finally{
            child = null ;
        }
        
        return ret ;
    }
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        long last ;
        long timing ;
        
        String name = null ;
        
        try{
            
            UtilCom.idleTime() ;
            
            synchronized( m_writeSync.get() ){
                
                synchronized( m_sync.get() ){
                    last = m_lastTime ;
                    timing = m_updateTimer ;
                    name = m_userFileName ;
                }
                
                // ファイルアップデートチェック.
                if(
                    last != NOT_UPDATE_TIME &&
                    last + timing <= System.currentTimeMillis()
                )
                {
                    
                    // 現在のユーザ管理をファイル出力.
                    this.save( name ) ;
                    
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ex ){
            throw ex ;
        }catch( Exception e ){
        }finally{
            name = null ;
        }
        
    }
    
    
    
    /**
     * ユーザ管理初期処理.
     */
    private final void initUserManager( String name,long timing )
        throws AccessException
    {
        int i ;
        int len ;
        
        try{
            
            // 基本条件を生成.
            m_sync.create() ;
            m_writeSync.create() ;
            
            synchronized( m_sync.get() ){
                synchronized( m_writeSync.get() ){
                    
                    // 対象ユーザ管理ファイルが存在する場合.
                    if( IOCom.isFileExists( name ) == true ){
                        
                        // 以前のユーザ管理情報をロード.
                        this.load( name ) ;
                        
                    }
                    
                    // ユーザ管理ファイル名を登録.
                    m_userFileName = name ;
                    
                    // 前回書き込み時間を無効に設定.
                    m_lastTime = NOT_UPDATE_TIME ;
                    
                    // スレッド開始.
                    m_thread.create( this ) ;
                    m_thread.startThread() ;
                    
                    // 更新タイミングをセット.
                    m_updateTimer = ( timing <= MIN_SAVE_TIMING ) ?
                        MIN_SAVE_TIMING : (
                        ( timing >= MAX_SAVE_TIMING ) ?
                        MAX_SAVE_TIMING : timing
                    ) ;
                    
                    // 必要なユーザを生成.
                    synchronized( m_sync.get() ){
                        
                        // ROOTユーザ設定.
                        if( m_table.isData( ROOT_USER ) == false ){
                            
                            // デフォルトのROOTユーザを設定.
                            this.addUser(
                                ROOT_USER,
                                UserManagerEngine.ROOT_PASSWD,
                                true,true,
                                1,null
                            ) ;
                            
                        }
                        
                        // GUESTユーザ設定.
                        if( m_table.isData( GUEST_USER ) == false ){
                            
                            // デフォルトのROOTユーザを設定.
                            this.addUser(
                                GUEST_USER,
                                ROOT_PASSWD,
                                false,false,
                                DEF_LOGIN,null
                            ) ;
                            
                        }
                        
                    }
                    
                }
                
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ユーザマネージャ情報をロード.
     */
    private final void load( String name )
        throws AccessException
    {
        int i ;
        int len,lenJ ;
        int binLen ;
        int pnt ;
        int step ;
        
        StringBuffer buf = null ;
        CodeBase32 cb32 = null ;
        CharTable tbl = null ;
        NumberTable idTbl = null ;
        IdManager idMan = null ;
        
        String user = null ;
        UserChild child = null ;
        
        byte[] bin = null ;
        byte[] tmp = null ;
        byte[] key = null ;
        byte[] priKey = null ;
        
        try{
            
            if( IOCom.isRead( name ) == true ){
                
                bin = IOCom.getFile( name ) ;
                
                tmp = UserManagerEngine.USER_HEADER.getBytes(
                    HEADER_CHARSET
                ) ;
                
                if(
                    tmp == null || 
                    ( binLen = bin.length ) <= ( len = tmp.length )
                )
                {
                    throw new AccessException(
                        "対象のユーザユーザ管理ファイル(" + name +
                        ")は不明なヘッダです"
                    ) ;
                }
                
                for( i = 0 ; i < len ; i ++ ){
                    if( tmp[ i ] != bin[ i ] ){
                        throw new AccessException(
                            "対象のユーザユーザ管理ファイル(" + name +
                            ")は不明なヘッダです"
                        ) ;
                    }
                }
                
                // 暗号コードを取得.
                pnt = len ;
                key = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
                System.arraycopy( bin,pnt,key,0,CodeBase32.CODE32_KEY_LENGTH ) ;
                pnt += CodeBase32.CODE32_KEY_LENGTH ;
                
                // ステップコードを取得.
                step = ( int )( bin[ pnt ] & 0x000000ff ) ;
                pnt += 1 ;
                
                // プライベート暗号コードを生成.
                buf = new StringBuffer() ;
                for( i = 0 ; i < CodeBase32.CODE32_KEY_LENGTH ; i ++ ){
                    buf.append( ( ( int )key[ i ] ) ) ;
                }
                priKey = CodeBase32.convertStringByCode32Key( buf.toString() ) ;
                buf = null ;
                
                // ユーザ情報が設定されている場合.
                if( pnt <= binLen ){
                    
                    // 暗号解析.
                    cb32 = new CodeBase32( priKey ) ;
                    cb32.analysis( key,step,bin,pnt,binLen - pnt ) ;
                    
                    // 解析結果から、ユーザ名/パスワード/グループ名を取得.
                    tbl = m_table ;
                    idTbl = m_idTable ;
                    idMan = m_idMan ;
                    tbl.clear() ;
                    idTbl.clear() ;
                    idMan.clear() ;
                    
                    // ユーザ情報を取得.
                    for( ;; ){
                        
                        // 情報の終端の場合.
                        if( pnt >= binLen ){
                            break ;
                        }
                        
                        // ユーザ名長を取得.
                        len = ConvertParam.convertInt( pnt,bin ) ;
                        pnt += 4 ;
                        
                        // ユーザ名を取得.
                        tmp = new byte[ len ] ;
                        System.arraycopy( bin,pnt,tmp,0,len ) ;
                        user = new String( tmp,CHARSET ) ;
                        pnt += len ;
                        
                        // ユーザ要素を生成..
                        child = new UserChild() ;
                        child.m_max = -2 ;
                        
                        // ユーザIDを取得.
                        child.m_userID = ConvertParam.convertInt( pnt,bin ) ;
                        pnt += 4 ;
                        
                        // パスワード長を取得.
                        len = ConvertParam.convertInt( pnt,bin ) ;
                        pnt += 4 ;
                        
                        // パスワードありの場合.
                        if( len > 0 ){
                            
                            // パスワードを取得.
                            tmp = new byte[ len ] ;
                            System.arraycopy( bin,pnt,tmp,0,len ) ;
                            child.m_passwd = new String( tmp,CHARSET ) ;
                            pnt += len ;
                            
                        }
                        // パスワード無しの場合.
                        else{
                            
                            child.m_passwd = UserManagerEngine.NOSET_PASSWD ;
                            
                        }
                        
                        // ROOT権限を取得.
                        child.m_rootOwner = ConvertParam.convertBoolean( pnt,bin ) ;
                        pnt += 1 ;
                        
                        // ROOTグループ権限を取得.
                        child.m_rootGroup = ConvertParam.convertBoolean( pnt,bin ) ;
                        pnt += 1 ;
                        
                        // 最大接続数を取得.
                        child.m_max = ConvertParam.convertInt( pnt,bin ) ;
                        pnt += 4 ;
                        
                        // 拡張コード長を取得.
                        len = ConvertParam.convertInt( pnt,bin ) ;
                        pnt += 4 ;
                        
                        // 拡張コードが存在する場合.
                        if( len > 0 ){
                            
                            child.m_extension = new String[ len ] ;
                            
                            for( i = 0 ; i < len ; i ++ ){
                                
                                // １つの拡張コード長を取得.
                                lenJ = ConvertParam.convertInt( pnt,bin ) ;
                                pnt += 4 ;
                                
                                // １つの拡張コード名を取得.
                                tmp = new byte[ lenJ ] ;
                                System.arraycopy( bin,pnt,tmp,0,lenJ ) ;
                                child.m_extension[ i ] = new String( tmp,CHARSET ) ;
                                pnt += lenJ ;
                                
                            }
                            
                        }
                        else{
                            child.m_extension = null ;
                        }
                        
                        if(
                            user == null ||
                            child.m_passwd == null ||
                            child.m_max == -2
                        )
                        {
                            throw new AccessException(
                                "対象のユーザ設定は(point:" + pnt +
                                ")は不正な情報です"
                            ) ;
                        }
                        
                        // ユーザ情報を追加.
                        tbl.add( user,child ) ;
                        idTbl.add( child.m_userID,user ) ;
                        idMan.addUseID( child.m_userID ) ;
                        
                    }
                    
                }
                // ユーザ情報が存在しない場合.
                else{
                    m_table.clear() ;
                }
            // ロード対象ファイルが存在しない場合.
            }else{
                m_table.clear() ;
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            buf = null ;
            cb32 = null ;
            tbl = null ;
            
            user = null ;
            child = null ;
            
            bin = null ;
            tmp = null ;
            key = null ;
            priKey = null ;
        }
        
    }
    
    /**
     * ユーザマネージャ情報をセーブ.
     */
    private final void save( String name )
        throws AccessException
    {
        int i,j ;
        int len,lenJ ;
        int off ;
        int step ;
        
        ByteUtil bu = null ;
        CodeBase32 cb32 = null ;
        CharTable tbl = null ;
        UserChild child = null ;
        StringBuffer buf = null ;
        
        byte[] key = null ;
        byte[] priKey = null ;
        byte[] tmp = null ;
        String[] names = null ;
        
        try{
            
            bu = new ByteUtil() ;
            
            // ユーザ管理ヘッダを設定.
            bu.put(
                UserManagerEngine.USER_HEADER.getBytes(
                    HEADER_CHARSET
                )
            ) ;
            
            // キー情報を設定.
            key = CodeBase32.getPublicKey() ;
            bu.add( key ) ;
            
            // ダミーオフセット値をセット.
            bu.add( ( byte )0 ) ;
            
            // 暗号オフセット値を取得.
            off = bu.size() ;
            
            tbl = m_table ;
            names = tbl.getNames() ;
            
            // ユーザ名が存在する場合.
            if( names != null ){
                
                len = names.length ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    // ユーザ情報を取得.
                    child = ( UserChild )tbl.get( names[ i ] ) ;
                    
                    // ユーザ名を設定.
                    tmp = names[ i ].getBytes( CHARSET ) ;
                    bu.put( ConvertParam.convertInt( tmp.length ) ) ;
                    bu.put( tmp ) ;
                    
                    // ユーザIDを設定.
                    bu.put( ConvertParam.convertInt( child.m_userID ) ) ;
                    
                    // パスワードを設定.
                    if( child.m_passwd.length() <= 0 ){
                        
                        // パスワードなし.
                        bu.put( ConvertParam.convertInt( 0 ) ) ;
                        
                    }
                    else{
                        
                        // パスワードあり.
                        tmp = child.m_passwd.getBytes( CHARSET ) ;
                        bu.put( ConvertParam.convertInt( tmp.length ) ) ;
                        bu.put( tmp ) ;
                        
                    }
                    
                    // ROOT権限を設定.
                    bu.put( ConvertParam.convertBoolean( child.m_rootOwner ) ) ;
                    
                    // ROOTグループ権限を設定.
                    bu.put( ConvertParam.convertBoolean( child.m_rootGroup ) ) ;
                    
                    // 最大接続数を設定.
                    bu.put( ConvertParam.convertInt( child.m_max ) ) ;
                    
                    // 拡張コードを設定.
                    if( child.m_extension != null ){
                        
                        lenJ = child.m_extension.length ;
                        bu.put( ConvertParam.convertInt( lenJ ) ) ;
                        
                        for( j = 0 ; j < lenJ ; j ++ ){
                            tmp = child.m_extension[ j ].getBytes( CHARSET ) ;
                            bu.put( ConvertParam.convertInt( tmp.length ) ) ;
                            bu.put( tmp ) ;
                        }
                        
                    }
                    else{
                        // 拡張コードなし.
                        bu.put( ConvertParam.convertInt( 0 ) ) ;
                    }
                }
                
            }
            
            // 対象バイナリ情報を取得.
            tmp = bu.get() ;
            
            // プライベート暗号コードを生成.
            buf = new StringBuffer() ;
            for( i = 0 ; i < CodeBase32.CODE32_KEY_LENGTH ; i ++ ){
                buf.append( ( ( int )key[ i ] ) ) ;
            }
            priKey = CodeBase32.convertStringByCode32Key( buf.toString() ) ;
            buf = null ;
            
            // バイナリ情報を暗号化.
            cb32 = new CodeBase32( priKey ) ;
            
            // 暗号処理.
            step = cb32.encryption( key,tmp,off,tmp.length - off ) ;
            
            // ステップコードを付加.
            tmp[ off-1 ] = ( byte )( step & 0x000000ff ) ;
            
            // 対象データをファイル出力.
            IOCom.setFile( name,true,tmp ) ;
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            
            if( bu != null ){
                bu.clear() ;
            }
            
            bu = null ;
            cb32 = null ;
            tbl = null ;
            child = null ;
            buf = null ;
            
            key = null ;
            priKey = null ;
            tmp = null ;
            names = null ;
            
            m_lastTime = NOT_UPDATE_TIME ;
            
        }
    }
    
    /**
     * 変更／削除禁止ユーザチェック.
     * mode=trueの場合、GUESTユーザ禁止+ROOTユーザ禁止.
     * mode=falseの場合、ROOTユーザ禁止.
     */
    private static final boolean checkUserName( boolean mode,String user )
    {
        return (
            user.equals( ROOT_USER ) == true ||
            ( mode == true && user.equals( GUEST_USER ) == true )
        ) ? true : false ;
        
    }
    
}

/**
 * ユーザ要素.
 */
class UserChild{
    
    public int m_userID = IdManager.NOT_ID ;
    public int m_nowUserCount = 0 ;
    public int m_max = UserManagerEngine.STOP_MAX_LOGIN ;
    public String m_passwd = null ;
    public boolean m_rootOwner = false ;
    public boolean m_rootGroup = false ;
    public String[] m_extension = null ;
    
    public final void clear(){
        m_userID = IdManager.NOT_ID ;
        m_nowUserCount = 0 ;
        m_max = UserManagerEngine.STOP_MAX_LOGIN ;
        m_passwd = null ;
        m_rootOwner = false ;
        m_rootGroup = false ;
        m_extension = null ;
    }
}
