/*
 * @(#)UserManager.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.sys.user;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.IdManager;

/**
 * ユーザマネージャ.
 * <BR><BR>
 * ユーザの管理をサポートします.
 *  
 * @version 2005/09/18
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class UserManager
{
    
    /**
     * ROOTユーザ名.
     */
    public static final String ROOT_USER = UserManagerEngine.ROOT_USER ;
    
    /**
     * GUESTユーザ名.
     */
    public static final String GUEST_USER = UserManagerEngine.GUEST_USER ;
    
    
    
    /**
     * ユーザ管理エンジン.
     */
    private UserManagerEngine m_engine = null ;
    
    /**
     * コンストラクタ.
     */
    private UserManager(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ユーザ管理オブジェクトを生成します.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public UserManager( String name )
        throws InputException,AccessException
    {
        m_engine = new UserManagerEngine() ;
        m_engine.create( name ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ユーザ管理オブジェクトを生成します.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @param timing データがアップデートされてからの保存間隔を設定します.<BR>
     *               また、指定単位はミリ秒です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public UserManager( String name,long timing )
        throws InputException,AccessException
    {
        m_engine = new UserManagerEngine() ;
        m_engine.create( name ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * ユーザ管理オブジェクトを破棄.
     * <BR><BR>
     * 対象のユーザ管理オブジェクトを破棄します.
     */
    public final void destroy()
    {
        try{
            m_engine.clear() ;
        }catch( Exception e ){
        }finally{
            m_engine = null ;
        }
    }
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * ユーザ情報を追加します.<BR>
     * また、最大接続数はデフォルト値(5)で設定されます.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void addUser( String user,String passwd )
        throws InputException
    {
        try{
            m_engine.addUser( user,passwd ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * ユーザ情報を追加します.<BR>
     * また、最大接続数はデフォルト値(5)で設定されます.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @param rootOwner 対象ユーザに対してルート権限を与えるか設定します.
     * @param rootGroup 対象ユーザに対してルートグループ権限を与えるか設定します.
     * @exception InputException 入力例外.
     */
    public final void addUser( String user,String passwd,boolean rootOwner,boolean rootGroup )
        throws InputException
    {
        try{
            m_engine.addUser( user,passwd,rootOwner,rootGroup ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * ユーザ追加.
     * <BR><BR>
     * ユーザ情報を追加します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @param rootOwner 対象ユーザに対してルート権限を与えるか設定します.
     * @param rootGroup 対象ユーザに対してルートグループ権限を与えるか設定します.
     * @param max ユーザ同時接続数を設定します.<BR>
     *            設定可能な最大値は[512]です.<BR>
     *            また[0]とするか、最大値以上を設定した場合、同時ユーザ接続数の上限が無くなります.<BR>
     *            また[-1]を設定することで、このユーザを利用出来なくします.
     * @param extension 拡張情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void addUser( String user,String passwd,boolean rootOwner,boolean rootGroup,int max,String[] extension )
        throws InputException
    {
        try{
            m_engine.addUser( user,passwd,rootOwner,rootGroup,max,extension ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * ユーザ削除.
     * <BR><BR>
     * 対象のユーザ名を削除します.
     * <BR>
     * @param user 削除対象のユーザ名を設定します.
     * @exception InputException 入力例外.
     */
    public final void removeUser( String user )
        throws InputException
    {
        try{
            m_engine.removeUser( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するパスワード変更.
     * <BR><BR>
     * 対象のユーザ情報のパスワードを変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param newPasswd 変更対象のパスワード名を設定します.
     * @exception InputException 入力例外.
     */
    public final void renewPasswd( String user,String newPasswd )
        throws InputException
    {
        try{
            m_engine.renewPasswd( user,newPasswd ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するROOT権限を変更.
     * <BR><BR>
     * 対象のユーザ情報のROOT権限を変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param owner 対象ユーザに対してルート権限を与えるか設定します.
     * @exception InputException 入力例外.
     */
    public final void renewRootOwner( String user,boolean owner )
        throws InputException
    {
        try{
            m_engine.renewRootOwner( user,owner ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するROOTグループを変更.
     * <BR><BR>
     * 対象のユーザ情報のROOTグループを変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param group 対象ユーザに対してルートグループを与えるか設定します.
     * @exception InputException 入力例外.
     */
    public final void renewRootGroup( String user,boolean group )
        throws InputException
    {
        try{
            m_engine.renewRootGroup( user,group ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対する最大接続数を変更.
     * <BR><BR>
     * 対象のユーザ情報の最大接続数を変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param newMax 変更対象の最大接続数を設定します.
     * @exception InputException 入力例外.
     */
    public final void renewMaxUserCount( String user,int newMax )
        throws InputException
    {
        try{
            m_engine.renewMaxUserCount( user,newMax ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対する拡張情報を変更.
     * <BR><BR>
     * 対象のユーザ情報の拡張情報を変更します.
     * <BR>
     * @param user 変更対象のユーザ名を設定します.
     * @param newExtension 変更対象の拡張情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void renewExtension( String user,String[] newExtension )
        throws InputException
    {
        try{
            m_engine.renewExtension( user,newExtension ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザのログインカウントを１インクリメント.
     * <BR><BR>
     * 対象ユーザのログインカウントを１インクリメントします.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @exception InputException 入力例外.
     * @exception LoginException ログイン例外.
     */
    public final void addUserCount( String user )
        throws InputException,LoginException
    {
        try{
            m_engine.addUserCount( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( LoginException le ){
            throw le ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザのログインカウントを１デクリメント.
     * <BR><BR>
     * 対象ユーザのログインカウントを１デクリメントします.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @exception InputException 入力例外.
     */
    public final void removeUserCount( String user )
        throws InputException
    {
        try{
            m_engine.removeUserCount( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 対象ユーザに対するパスワード取得.
     * <BR><BR>
     * 対象のユーザ名に対するパスワードを取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return String 対象のパスワードが返されます.
     * @exception InputException 入力例外.
     */
    public final String getPasswd( String user )
        throws InputException
    {
        String ret = null ;
        
        try{
            ret = m_engine.getPasswd( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対するルート権限を取得.
     * <BR><BR>
     * 対象のユーザ名に対するルート権限を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return boolean 対象のルート権限が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     * @exception InputException 入力例外.
     */
    public final boolean getRootOwner( String user )
        throws InputException
    {
        boolean ret = false ;
        
        try{
            ret = m_engine.getRootOwner( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対するルートグループ権限を取得.
     * <BR><BR>
     * 対象のユーザ名に対するルートグループ権限を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return boolean 対象のルートグループ権限が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     * @exception InputException 入力例外.
     */
    public final boolean getRootGroupOwner( String user )
        throws InputException
    {
        boolean ret = false ;
        
        try{
            ret = m_engine.getRootGroupOwner( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対する現在の接続数を取得.
     * <BR><BR>
     * 対象のユーザ名に対する現在の接続数を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return int 現在の接続数が返されます.
     * @exception InputException 入力例外.
     */
    public final int getNowUserCount( String user )
        throws InputException
    {
        int ret = -1 ;
        
        try{
            ret = m_engine.getNowUserCount( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対する最大接続数を取得.
     * <BR><BR>
     * 対象のユーザ名に対する最大接続数を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return int 対象の最大接続数が返されます.<BR>
     *             [-1]が返された場合、ログイン出来ません.
     *             [0]が返された場合、無制限にログイン出来ます.
     * @exception InputException 入力例外.
     */
    public final int getMaxUserCount( String user )
        throws InputException
    {
        int ret = -1 ;
        
        try{
            ret = m_engine.getMaxUserCount( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザに対する拡張情報を取得.
     * <BR><BR>
     * 対象のユーザ名に対する拡張情報を取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return String[] 対象の拡張情報が返されます.
     * @exception InputException 入力例外.
     */
    public final String[] getExtension( String user )
        throws InputException
    {
        String[] ret = null ;
        
        try{
            ret = m_engine.getExtension( user ) ;
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザ名に対するユーザIDを取得.
     * <BR><BR>
     * 対象のユーザに対するユーザIDを取得します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @return int 対象のユーザIDが返されます.<BR>
     *             [IdManager.NOT_ID]が返された場合、対象のユーザ名は
     *             存在しません
     */
    public final int getUserNameByUserID( String user )
    {
        int ret = IdManager.NOT_ID ;
        
        try{
            ret = m_engine.getUserNameByUserID( user ) ;
        }catch( Exception e ){
            ret = IdManager.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザIDに対するユーザ名を取得.
     * <BR><BR>
     * 対象のユーザIDに対するユーザ名を取得します.
     * <BR>
     * @param id 対象のユーザIDを設定します.
     * @return String 対象のユーザ名が返されます.
     */
    public final String getUserIDByUserName( int id )
    {
        String ret = null ;
        
        try{
            ret = m_engine.getUserIDByUserName( id ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * ユーザ名一覧を取得.
     * <BR><BR>
     * 管理ユーザ名一覧を取得します.
     * <BR>
     * @return String[] 管理されているユーザ名一覧が返されます.
     */
    public final String[] getUsers()
    {
        String[] ret = null ;
        
        try{
            ret = m_engine.getUsers() ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザ名数を取得.
     * <BR><BR>
     * 管理ユーザ名数を取得します.
     * <BR>
     * @return int 管理されているユーザ名数を取得します.
     */
    public final int getUserLength()
    {
        int ret = 0 ;
        
        try{
            ret = m_engine.getUserLength() ;
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ユーザ名のパスワードが正しいかチェック.
     * <BR><BR>
     * 対象ユーザ名のパスワードが正しいかチェックします.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワード情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象条件は正しいです.<BR>
     *                 [false]が返された場合、対象条件は正しくありません.
     */
    public final boolean isUserPasswd( String user,String passwd )
    {
        boolean ret ;
        
        String chkPasswd = null ;
        
        try{
            if( m_engine.isUser( user ) == true ){
                
                chkPasswd = m_engine.getPasswd( user ) ;
                
                if(
                    ( chkPasswd == null || chkPasswd.length() <= 0 ) &&
                    ( passwd == null || passwd.length() <= 0 )
                )
                {
                    ret = true ;
                }
                else if( passwd.equals( chkPasswd ) == true ){
                    ret = true ;
                }
                else{
                    ret = false ;
                }
            }
            else{
                ret = false ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ユーザ存在チェック.
     * <BR><BR>
     * 対象ユーザ名が存在するかチェックします.
     * <BR>
     * @param user チェック対象のユーザ名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合対象ユーザ名は存在します.<BR>
     *                 [false]が返された場合対象ユーザ名は存在しません.
     */
    public final boolean isUser( String user )
    {
        boolean ret = false ;
        
        try{
            ret = m_engine.isUser( user ) ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 利用可能ユーザチェック.
     * <BR><BR>
     * 利用可能なユーザチェックを行います.
     * <BR>
     * @param user チェック対象のユーザ名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合対象ユーザ名は存在します.<BR>
     *                 [false]が返された場合対象ユーザ名は存在しません.
     */
    public final boolean isUseUser( String user )
    {
        boolean ret = false ;
        
        try{
            ret = m_engine.isUseUser( user ) ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

