/*
 * @(#)NativeCmd.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.sys.cmd;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.ByteUtil;
import com.JRcServer.commons.util.UtilCom;

/**
 * ODコマンド実行.
 * <BR><BR>
 * OSのコマンドを実施する為の処理です.<BR>
 * この処理は基本的にコマンド条件を実施してコンソールに出力されたデータを
 * 返します.
 * 
 * @version 1.0.0 2004/10/15
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class NativeCmd
{
    
    /**
     * 実行結果なし.
     */
    public static final String NO_RESULT = "ok." ;
    
    /**
     * 処理終了ループ.
     */
    private static final int MAX_COUNT = 20 ;
    
    /**
     * 待機時間.
     */
    private static final int WAIT_TIME = 5 ;
    
    /**
     * バッファ長.
     */
    private static final int BUFFER_LENGTH = 4096 ;
    
    /**
     * 戻り値.
     */
    private int m_retCode = Integer.MAX_VALUE ;
    
    /**
     * 実行コマンド情報.
     */
    private String m_setCommand = null ;
    
    /**
     * 作業ディレクトリ名設定.
     */
    private String m_directory = null ;
    
    /**
     * コンストラクタ.
     */
    public NativeCmd()
    {
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_retCode = Integer.MAX_VALUE ;
        m_setCommand = null ;
        m_directory = null ;
    }
    
    /**
     * 作業ディレクトリを設定.
     * <BR><BR>
     * 対象の作業ディレクトリを設定します.
     * <BR>
     * @param dir 作業ディレクトリを設定します.
     */
    public final void setDirectory( String dir )
    {
        if( dir == null || dir.length() <= 0 ){
            m_directory = null ;
        }
        else{
            m_directory = dir ;
        }
    }
    
    /**
     * コマンド実行.
     * <BR><BR>
     * コマンド処理を実行します.
     * <BR>
     * @param mode 実行結果取得モードを設定します.<BR>
     *             [true]を設定した場合小文字変換して返されます.<BR>
     *             [false]を設定した場合変換しません.
     * @param cmd 実行対象のコマンド内容を設定します.
     * @return String[] コンソールに出力された情報が返されます.
     * @exception NoExitProcessException プロセス実行例外.
     */
    public final String[] execution( boolean mode,ArrayList cmd )
        throws NoExitProcessException
    {
        int i ;
        int len ;
        
        String[] input = null ;
        String[] ret = null ;
        
        if( cmd != null ){
            
            len = cmd.size() ;
            input = new String[ len ] ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                input[ i ] = ( String )cmd.get( i ) ;
                
            }
            
            ret = this.runExec( mode,input,null,null ) ;
            input = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * コマンド実行.
     * <BR><BR>
     * コマンド処理を実行します.
     * <BR>
     * @param mode 実行結果取得モードを設定します.<BR>
     *             [true]を設定した場合小文字変換して返されます.<BR>
     *             [false]を設定した場合変換しません.
     * @param cmd 実行対象のコマンド内容を設定します.
     * @return String[] コンソールに出力された情報が返されます.
     * @exception NoExitProcessException プロセス実行例外.
     */
    public final String[] execution( boolean mode,String[] cmd )
        throws NoExitProcessException
    {
        return this.runExec( mode,cmd,null,null ) ;
    }
    
    /**
     * コマンド実行.
     * <BR><BR>
     * コマンド処理を実行します.
     * <BR>
     * @param mode 実行結果取得モードを設定します.<BR>
     *             [true]を設定した場合小文字変換して返されます.<BR>
     *             [false]を設定した場合変換しません.
     * @param cmd 実行対象のコマンド内容を設定します.
     * @return String[] コンソールに出力された情報が返されます.
     * @exception NoExitProcessException プロセス実行例外.
     */
    public final String[] execution( boolean mode,String cmd )
        throws NoExitProcessException
    {
        return this.runExec( mode,null,cmd,null ) ;
    }
    
    /**
     * コマンド実行.
     * <BR><BR>
     * コマンド処理を実行します.
     * <BR>
     * @param mode 実行結果取得モードを設定します.<BR>
     *             [true]を設定した場合小文字変換して返されます.<BR>
     *             [false]を設定した場合変換しません.
     * @param cmd 実行対象のコマンド内容を設定します.
     * @param env コマンド実行時の環境変数を設定します.
     * @return String[] コンソールに出力された情報が返されます.
     * @exception NoExitProcessException プロセス実行例外.
     */
    public final String[] execution( boolean mode,ArrayList cmd,String[] env )
        throws NoExitProcessException
    {
        int i ;
        int len ;
        
        String[] input = null ;
        String[] ret = null ;
        
        if( cmd != null ){
            
            len = cmd.size() ;
            input = new String[ len ] ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                input[ i ] = ( String )cmd.get( i ) ;
                
            }
            
            ret = this.runExec( mode,input,null,env ) ;
            input = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * コマンド実行.
     * <BR><BR>
     * コマンド処理を実行します.
     * <BR>
     * @param mode 実行結果取得モードを設定します.<BR>
     *             [true]を設定した場合小文字変換して返されます.<BR>
     *             [false]を設定した場合変換しません.
     * @param cmd 実行対象のコマンド内容を設定します.
     * @param env コマンド実行時の環境変数を設定します.
     * @return String[] コンソールに出力された情報が返されます.
     * @exception NoExitProcessException プロセス実行例外.
     */
    public final String[] execution( boolean mode,String[] cmd,String[] env )
        throws NoExitProcessException
    {
        return this.runExec( mode,cmd,null,env ) ;
    }
    
    /**
     * コマンド実行.
     * <BR><BR>
     * コマンド処理を実行します.
     * <BR>
     * @param mode 実行結果取得モードを設定します.<BR>
     *             [true]を設定した場合小文字変換して返されます.<BR>
     *             [false]を設定した場合変換しません.
     * @param cmd 実行対象のコマンド内容を設定します.
     * @param env コマンド実行時の環境変数を設定します.
     * @return String[] コンソールに出力された情報が返されます.
     * @exception NoExitProcessException プロセス実行例外.
     */
    public final String[] execution( boolean mode,String cmd,String[] env )
        throws NoExitProcessException
    {
        return this.runExec( mode,null,cmd,env ) ;
    }
    
    /**
     * 実行コマンド情報を取得.
     * <BR><BR>
     * 実行されたコマンド情報を取得します.
     * <BR>
     * @return String 実行されたコマンド情報が返されます.
     */
    public final String getCommand()
    {
        return m_setCommand ;
    }
    
    /**
     * 作業ディレクトリを取得.
     * <BR><BR>
     * 作業ディレクトリを取得します.
     * <BR>
     * @return String 作業ディレクトリが返されます.<BR>
     *                [null]が返された場合、作業ディレクトリは設定されていません.
     */
    public final String getDirectory()
    {
        return m_directory ;
    }
    
    /**
     * 実行結果コードを取得.
     * <BR><BR>
     * 実行結果コードを取得します.
     * <BR>
     * @return int 実行結果コードが返されます.<BR>
     *              実行時にエラーが発生したか実行されていない場合
     *              [Integer.MAX_VALUE]が返されます.
     */
    public final int getReturnCode()
    {
        return m_retCode ;
    }
    
    /**
     * コマンド実行.
     */
    private final String[] runExec( boolean mode,String[] param,String cmd,String[] env )
        throws NoExitProcessException
    {
        int i ;
        int len ;
        
        Process proc = null ;
        ByteUtil bu = null ;
        String dir = null ;
        StringBuffer buf = null ;
        String output = null ;
        InputStream out = null ;
        String[] ret = null ;
        ArrayList lst = null ;
        
        try{
            
            dir = m_directory ;
            
            // 実行生成.
            param = CmdCommon.getCmdHeader( param,cmd ) ;
            
            // パラメータありの処理.
            if( param != null && param.length > 1 ){
                
                if( env != null ){
                    if( dir == null || dir.length() <= 0 ){
                        proc = Runtime.getRuntime().exec( param,env ) ;
                    }
                    else{
                        proc = Runtime.getRuntime().exec( param,env,new File( dir ) ) ;
                    }
                }
                else{
                    if( dir == null || dir.length() <= 0 ){
                        proc = Runtime.getRuntime().exec( param ) ;
                    }
                    else{
                        proc = Runtime.getRuntime().exec( param,null,new File( dir ) ) ;
                    }
                }
                
                buf = new StringBuffer() ;
                len = param.length ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    buf.append( param[ i ] ) ;
                    buf.append( " " ) ;
                    
                }
                
                m_setCommand = buf.toString() ;
                buf = null ;
                
            }
            // パラメータなしの処理.
            else{
                
                if( param != null && param.length == 1 ){
                    cmd = param[ 0 ] ;
                }
                
                if( env != null ){
                    if( dir == null || dir.length() <= 0 ){
                        proc = Runtime.getRuntime().exec( cmd,env ) ;
                    }
                    else{
                        proc = Runtime.getRuntime().exec( cmd,env,new File( dir ) ) ;
                    }
                }
                else{
                    if( dir == null || dir.length() <= 0 ){
                        proc = Runtime.getRuntime().exec( cmd ) ;
                    }
                    else{
                        proc = Runtime.getRuntime().exec( new String[]{ cmd },null,new File( dir ) ) ;
                    }
                }
                
                m_setCommand = cmd ;
                
            }
            
            out = new BufferedInputStream( proc.getInputStream() ) ;
            bu = new ByteUtil() ;
            
            // 処理が終了するまで待つ.
            NativeCmd.getResult( bu,out,NativeCmd.MAX_COUNT ) ;
            
            // 戻り値が存在する場合、文字に変換する.
            if( bu.size() > 0 ){
                
                output = new String( bu.get(),BaseDef.THIS_CHARSET ) ;
                bu.clear() ;
                bu = null ;
                
                if( output != null && output.trim().length() > 0 ){
                    
                    lst = UtilCom.pauseString( output,BaseDef.ENTER_SEPARATOR ) ;
                    output = null ;
                    len = lst.size() ;
                    ret = new String[ len ] ;
                    
                    if( mode == true ){
                        for( i = 0 ; i < len ; i ++ ){
                            
                            ret[ i ] = UtilCom.trimPlus( ( String )lst.get( i ) ).toLowerCase() ;
                            
                        }
                    }
                    else{
                        for( i = 0 ; i < len ; i ++ ){
                            
                            ret[ i ] = UtilCom.trimPlus( ( String )lst.get( i ) ) ;
                            
                        }
                    }
                }
                else{
                    ret = new String[ 1 ] ;
                    ret[ 0 ] = NO_RESULT ;
                }
                
            }
            else{
                ret = new String[ 1 ] ;
                ret[ 0 ] = NO_RESULT ;
            }
            
        }catch( IOException io ){
            io.printStackTrace() ;
            m_retCode = Integer.MAX_VALUE ;
            throw new NoExitProcessException( io ) ;
        }catch( AccessException ac ){
            ac.printStackTrace() ;
            m_retCode = Integer.MAX_VALUE ;
            throw new NoExitProcessException( ac ) ;
        }catch( Exception t ){
            t.printStackTrace() ;
            m_retCode = Integer.MAX_VALUE ;
            throw new NoExitProcessException( t ) ;
        }finally{
            
            try{
                out.close() ;
            }catch( Exception tt ){
            }
            
            try{
                proc.destroy() ;
            }catch( Exception tt ){
            }
            
            try{
                lst.clear() ;
            }catch( Exception tt ){
            }
            
            try{
                bu.clear() ;
            }catch( Exception tt ){
            }
            
            proc = null ;
            buf = null ;
            out = null ;
            lst = null ;
            bu = null ;
            output = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * バッファ情報を取得.
     */
    private static final void getResult( ByteUtil out,InputStream is,int roop )
        throws InputException,AccessException
    {
        int cnt ;
        int len ;
        
        byte[] buf = null ;
        
        try{
            
            buf = new byte[ NativeCmd.BUFFER_LENGTH ] ;
            
            // 処理が終了するまで待つ.
            for ( cnt = 0 ;;) {
                
                if( ( len = is.read( buf ) ) == -1 ){
                    
                    try{ Thread.sleep( NativeCmd.WAIT_TIME ) ; }catch( Exception ee ){}
                    cnt ++ ;
                    
                    if( cnt >= roop ){
                        break ;
                    }
                    
                    continue ;
                }
                
                out.add( buf,0,len ) ;
                cnt = 0 ;
                
            }
        }catch( InputException in ){
            throw in ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }finally{
            
            try{
                is.close() ;
            }catch( Exception e ){
            }
            
            buf = null ;
            
        }
    }
    
}
